package de.fzi.wim.oimodeler.oimodelgraph.graph;

import java.util.List;
import java.util.Set;
import java.util.Iterator;
import java.util.HashSet;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

import de.fzi.wim.guibase.graphview.graph.*;

/**
 * Visitor interface for change events.
 */
public class OIModelGraphChangeVisitor implements ChangeVisitor {
    /** The set of property URIs that need ot be filtered. */
    protected static final Set s_filterPropertyURIs=new HashSet();
    static {
        s_filterPropertyURIs.add(KAONVocabularyAdaptor.INSTANCE.getReferences());
    }

    /** The graph managed by this visitor. */
    protected OIModelGraph m_oimodelGraph;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelGraph                  the graph of this visitor
     */
    public OIModelGraphChangeVisitor(OIModelGraph oimodelGraph) {
        m_oimodelGraph=oimodelGraph;
    }
    /**
     * Processes the list and applies the changes to the graph.
     *
     * @param changeEvents                  the list of the changes
     */
    public void applyChanges(List changeEvents) throws KAONException {
        synchronized (m_oimodelGraph) {
            try {
                m_oimodelGraph.suspendEvents();
                Iterator iterator=changeEvents.iterator();
                while (iterator.hasNext()) {
                    ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                    changeEvent.accept(this);
                }
            }
            finally {
                m_oimodelGraph.resumeEvents();
            }
        }
    }
    /**
     * Returns the node for the concept.
     *
     * @param concept                       the concept
     * @return                              the node for given concept
     */
    protected ConceptNode getNode(Concept concept) {
        return (ConceptNode)m_oimodelGraph.getNodeForEntity(concept);
    }
    /**
     * Returns <code>true</code> if given entity should be shown in the graph.
     *
     * @param entity                        the entity
     * @return                              <code>true</code> if the entity should be shown in the graph
     * @throws KAONException                thrown if there is an error
     */
    protected boolean showEntity(Entity entity) throws KAONException {
        return m_oimodelGraph.getOIModelGraphAnchor().getViewFilter().showEntity(entity);
    }
    /**
     * Returns <code>true</code> if given property instance should be shown in the graph.
     *
     * @param propertyInstance              the property instance
     * @return                              <code>true</code> if the property instance should be shown in the graph
     * @throws KAONException                thrown if there is an error
     */
    protected boolean showPropertyInstance(PropertyInstance propertyInstance) throws KAONException {
        return m_oimodelGraph.getOIModelGraphAnchor().getViewFilter().showPropertyInstance(propertyInstance);
    }
    /**
     * Returns the node for the property.
     *
     * @param property                      the property
     * @return                              the node for given property
     */
    protected PropertyNode getNode(Property property) {
        return (PropertyNode)m_oimodelGraph.getNodeForEntity(property);
    }
    /**
     * Returns the node for the instance.
     *
     * @param instance                      the instance
     * @return                              the node for given instance
     */
    protected InstanceNode getNode(Instance instance) {
        return (InstanceNode)m_oimodelGraph.getNodeForEntity(instance);
    }
    /**
     * Updates the lexical value of the node.
     *
     * @param entity                        the entity
     * @throws KAONException                thrown if there is a problem
     */
    protected void updateLexicon(Entity entity) throws KAONException {
        EntityNode node=m_oimodelGraph.getNodeForEntity(entity);
        if (node!=null) {
            node.updateLabel(m_oimodelGraph);
            m_oimodelGraph.notifyUpdated();
        }
    }
    /**
     * Updates the label of the property instances node for given property.
     *
     * @param property                      the property
     * @throws KAONException                thrown if there is a problem
     */
    protected void updatePropertyInstancesLexicon(Property property) throws KAONException {
        Iterator nodes=m_oimodelGraph.getNonEntityNodes().iterator();
        while (nodes.hasNext()) {
            Node node=(Node)nodes.next();
            if (node instanceof PropertyInstancesNode) {
                PropertyInstancesNode propertyInstancesNode=(PropertyInstancesNode)node;
                if (propertyInstancesNode.getProperty().equals(property)) {
                    propertyInstancesNode.updateLabel(m_oimodelGraph);
                    m_oimodelGraph.notifyUpdated();
                }
            }
        }
    }
    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) throws KAONException {
        // This is kind of a hack to make added spanning instances show in the graph.
        if ((event.getEntity() instanceof Concept) || (event.getEntity() instanceof Property)) {
            Concept root=m_oimodelGraph.getOIModel().getRootConcept();
            ConceptNode rootConceptNode=(ConceptNode)m_oimodelGraph.getNodeForEntity(root);
            if (rootConceptNode!=null && rootConceptNode.getInstancesExpanded()!=ConceptNode.NOT_LOADED)
                visit(new AddInstanceOf(root,event.getEntity().getSpanningInstance()));
        }
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) throws KAONException {
        // This is kind of a hack to make removed spanning instances show in the graph.
        if ((event.getEntity() instanceof Concept) || (event.getEntity() instanceof Property)) {
            Concept root=m_oimodelGraph.getOIModel().getRootConcept();
            ConceptNode rootConceptNode=(ConceptNode)m_oimodelGraph.getNodeForEntity(root);
            if (rootConceptNode!=null && rootConceptNode.getInstancesExpanded()!=ConceptNode.NOT_LOADED)
                visit(new RemoveInstanceOf(root,event.getEntity().getSpanningInstance()));
            EntityNode spanningEntityNode=m_oimodelGraph.getNodeForEntity(event.getEntity().getSpanningInstance());
            if (spanningEntityNode!=null)
                m_oimodelGraph.remove(spanningEntityNode);
        }
        else if (event.getEntity() instanceof Instance) {
            Instance instance=(Instance)event.getEntity();
            ConceptNode spanningConceptNode=(ConceptNode)m_oimodelGraph.getNodeForEntity(instance.getSpanningConcept());
            if (spanningConceptNode!=null)
                m_oimodelGraph.remove(spanningConceptNode);
            PropertyNode spanningPropertyNode=(PropertyNode)m_oimodelGraph.getNodeForEntity(instance.getSpanningProperty());
            if (spanningPropertyNode!=null)
                m_oimodelGraph.remove(spanningPropertyNode);
        }
        EntityNode entityNode=m_oimodelGraph.getNodeForEntity(event.getEntity());
        if (entityNode!=null)
            m_oimodelGraph.remove(entityNode);
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) throws KAONException {
        ConceptNode superConceptNode=getNode(event.getSuperConcept());
        ConceptNode subConceptNode=getNode(event.getSubConcept());
        if (superConceptNode!=null && subConceptNode==null && showEntity(event.getSubConcept())) {
            subConceptNode=new ConceptNode(event.getSubConcept(),superConceptNode);
            m_oimodelGraph.add(subConceptNode);
        }
        if (subConceptNode!=null && superConceptNode==null && showEntity(event.getSuperConcept())) {
            superConceptNode=new ConceptNode(event.getSuperConcept(),subConceptNode);
            m_oimodelGraph.add(superConceptNode);
        }
        if (superConceptNode!=null && subConceptNode!=null) {
            ConceptHierarchyEdge edge=new ConceptHierarchyEdge(superConceptNode,subConceptNode,ConceptHierarchyEdge.UNIT_LENGTH);
            m_oimodelGraph.add(edge);
        }
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) {
        ConceptNode superConceptNode=getNode(event.getSuperConcept());
        ConceptNode subConceptNode=getNode(event.getSubConcept());
        if (superConceptNode!=null && subConceptNode!=null) {
            ConceptHierarchyEdge edge=ConceptHierarchyEdge.getEdge(m_oimodelGraph,superConceptNode,subConceptNode);
            if (edge!=null)
                m_oimodelGraph.remove(edge);
        }
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) throws KAONException {
        PropertyNode propertyNode=getNode(event.getProperty());
        ConceptNode conceptNode=getNode(event.getConcept());
        if (propertyNode!=null && conceptNode==null && showEntity(event.getConcept())) {
            conceptNode=new ConceptNode(event.getConcept(),propertyNode);
            m_oimodelGraph.add(conceptNode);
        }
        if (conceptNode!=null && propertyNode==null && showEntity(event.getProperty())) {
            propertyNode=new PropertyNode(event.getProperty(),conceptNode);
            m_oimodelGraph.add(propertyNode);
        }
        if (propertyNode!=null && conceptNode!=null) {
            PropertyDomainEdge edge=new PropertyDomainEdge(propertyNode,conceptNode);
            m_oimodelGraph.add(edge);
        }
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) {
        PropertyNode propertyNode=getNode(event.getProperty());
        ConceptNode conceptNode=getNode(event.getConcept());
        if (propertyNode!=null && conceptNode!=null) {
            PropertyDomainEdge edge=PropertyDomainEdge.getEdge(m_oimodelGraph,propertyNode,conceptNode);
            if (edge!=null)
                m_oimodelGraph.remove(edge);
        }
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) throws KAONException {
        PropertyNode propertyNode=getNode(event.getProperty());
        ConceptNode conceptNode=getNode(event.getConcept());
        if (propertyNode!=null && conceptNode==null && showEntity(event.getConcept())) {
            conceptNode=new ConceptNode(event.getConcept(),propertyNode);
            m_oimodelGraph.add(conceptNode);
        }
        if (conceptNode!=null && propertyNode==null && showEntity(event.getProperty())) {
            propertyNode=new PropertyNode(event.getProperty(),conceptNode);
            m_oimodelGraph.add(propertyNode);
        }
        if (propertyNode!=null && conceptNode!=null) {
            PropertyRangeEdge edge=new PropertyRangeEdge(propertyNode,conceptNode);
            m_oimodelGraph.add(edge);
        }
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) {
        PropertyNode propertyNode=getNode(event.getProperty());
        ConceptNode conceptNode=getNode(event.getConcept());
        if (propertyNode!=null && conceptNode!=null) {
            PropertyRangeEdge edge=PropertyRangeEdge.getEdge(m_oimodelGraph,propertyNode,conceptNode);
            if (edge!=null)
                m_oimodelGraph.remove(edge);
        }
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) throws KAONException {
        PropertyNode propertyNode=getNode(event.getProperty());
        if (propertyNode!=null) {
            propertyNode.updateIsAttribute();
            m_oimodelGraph.notifyUpdated();
        }
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) {
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) {
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) throws KAONException {
        PropertyNode superPropertyNode=getNode(event.getSuperProperty());
        PropertyNode subPropertyNode=getNode(event.getSubProperty());
        if (superPropertyNode!=null && subPropertyNode==null && showEntity(event.getSubProperty())) {
            subPropertyNode=new PropertyNode(event.getSubProperty(),superPropertyNode);
            m_oimodelGraph.add(subPropertyNode);
        }
        if (subPropertyNode!=null && superPropertyNode==null && showEntity(event.getSuperProperty())) {
            superPropertyNode=new PropertyNode(event.getSuperProperty(),subPropertyNode);
            m_oimodelGraph.add(superPropertyNode);
        }
        if (superPropertyNode!=null && subPropertyNode!=null) {
            PropertyHierarchyEdge edge=new PropertyHierarchyEdge(superPropertyNode,subPropertyNode,PropertyHierarchyEdge.UNIT_LENGTH);
            m_oimodelGraph.add(edge);
        }
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) {
        PropertyNode superPropertyNode=getNode(event.getSuperProperty());
        PropertyNode subPropertyNode=getNode(event.getSubProperty());
        if (superPropertyNode!=null && subPropertyNode!=null) {
            PropertyHierarchyEdge edge=PropertyHierarchyEdge.getEdge(m_oimodelGraph,superPropertyNode,subPropertyNode);
            if (edge!=null)
                m_oimodelGraph.remove(edge);
        }
    }
    /**
     * Visits an event for setting the inverse relationship between properties.
     */
    public void visit(SetInverseProperties event) {
    }
    /**
     * Visits an event for removing an inverse relationship between properties.
     */
    public void visit(SetNoInverseProperties event) {
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) {
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) {
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) throws KAONException {
        ConceptNode conceptNode=getNode(event.getConcept());
        InstanceNode instanceNode=getNode(event.getInstance());
        if (conceptNode!=null && instanceNode==null && showEntity(event.getInstance())) {
            instanceNode=new InstanceNode(event.getInstance(),conceptNode);
            m_oimodelGraph.add(instanceNode);
        }
        if (instanceNode!=null && conceptNode==null && showEntity(event.getConcept())) {
            conceptNode=new ConceptNode(event.getConcept(),instanceNode);
            m_oimodelGraph.add(conceptNode);
        }
        if (conceptNode!=null && instanceNode!=null)
            conceptNode.connectInstanceNode(m_oimodelGraph,instanceNode);
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) throws KAONException {
        ConceptNode conceptNode=getNode(event.getConcept());
        InstanceNode instanceNode=getNode(event.getInstance());
        if (conceptNode!=null && instanceNode!=null)
            conceptNode.disconnectInstanceNode(m_oimodelGraph,instanceNode);
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        if (propertyInstanceRelevant(propertyInstance) && showPropertyInstance(propertyInstance)) {
            InstanceNode sourceInstanceNode=getNode(propertyInstance.getSourceInstance());
            if (sourceInstanceNode!=null)
                sourceInstanceNode.addPropertyInstance(m_oimodelGraph,propertyInstance);
        }
        checkLexicalProperty(propertyInstance);
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        if (propertyInstanceRelevant(propertyInstance)) {
            InstanceNode sourceInstanceNode=getNode(propertyInstance.getSourceInstance());
            if (sourceInstanceNode!=null)
                sourceInstanceNode.removePropertyInstance(m_oimodelGraph,propertyInstance);
        }
        checkLexicalProperty(propertyInstance);
    }
    /**
     * Visits an event for setting the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        if (propertyInstanceRelevant(propertyInstance)) {
            InstanceNode sourceInstanceNode=getNode(propertyInstance.getSourceInstance());
            if (sourceInstanceNode!=null) {
                PropertyInstance[] removedPropertyInstances=event.getRemovedInstances();
                for (int i=0;i<removedPropertyInstances.length;i++)
                    sourceInstanceNode.removePropertyInstance(m_oimodelGraph,removedPropertyInstances[i]);
                sourceInstanceNode.addPropertyInstance(m_oimodelGraph,propertyInstance);
            }
        }
        checkLexicalProperty(propertyInstance);
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        if (propertyInstanceRelevant(propertyInstance)) {
            InstanceNode sourceInstanceNode=getNode(propertyInstance.getSourceInstance());
            if (sourceInstanceNode!=null)
                sourceInstanceNode.changePropertyInstance(m_oimodelGraph,propertyInstance,event.getNewTargetValue());
        }
        checkLexicalProperty(propertyInstance);
    }
    /**
     * Returns <code>true</code> if this property instance is relevant for presenting in the graph.
     *
     * @param propertyInstance                      the property instance
     * @return                                      <code>true</code> if this property instance should be skipped
     * @throws KAONException                        thrown if there is a problem
     */
    protected boolean propertyInstanceRelevant(PropertyInstance propertyInstance) throws KAONException {
        if (!s_filterPropertyURIs.contains(propertyInstance.getProperty().getURI()))
            return true;
        InstanceNode sourceInstanceNode=getNode(propertyInstance.getSourceInstance());
        if (sourceInstanceNode!=null && sourceInstanceNode.getPropertyInstancesFromExpanded()!=InstanceNode.NOT_LOADED)
            return true;
        Object targetValue=propertyInstance.getTargetValue();
        if (targetValue instanceof Instance) {
            InstanceNode targetInstanceNode=getNode((Instance)targetValue);
            if (targetInstanceNode!=null && targetInstanceNode.getPropertyInstancesToExpanded()!=InstanceNode.NOT_LOADED)
                return true;
        }
        return false;
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) {
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) {
    }
    /**
     * Checks whether this event updated the lexical value of the node.
     *
     * @param propertyInstance                      the property instance that was updated
     */
    protected void checkLexicalProperty(PropertyInstance propertyInstance) throws KAONException {
        String uri=propertyInstance.getProperty().getURI();
        if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getReferences())) {
            Object targetValue=propertyInstance.getTargetValue();
            if (targetValue instanceof Instance) {
                Instance instance=(Instance)targetValue;
                updateLexicon(instance);
                updateLexicon(instance.getSpanningConcept());
                updateLexicon(instance.getSpanningProperty());
                updatePropertyInstancesLexicon(instance.getSpanningProperty());
            }
        }
        else if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getValue()) || uri.equals(KAONVocabularyAdaptor.INSTANCE.getInLanguage())) {
            LexicalEntry lexicalEntry=propertyInstance.getOIModel().getLexicalEntry(propertyInstance.getSourceInstance().getURI());
            Iterator entities=lexicalEntry.getReferencedEntities().iterator();
            while (entities.hasNext()) {
                Instance instance=(Instance)entities.next();
                updateLexicon(instance);
                updateLexicon(instance.getSpanningConcept());
                updateLexicon(instance.getSpanningProperty());
                updatePropertyInstancesLexicon(instance.getSpanningProperty());
            }
        }
    }
}
