package de.fzi.wim.oimodeler.oimodelgraph.graph;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Collection;
import java.util.AbstractSequentialList;
import java.util.ListIterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

import de.fzi.wim.guibase.graphview.graph.*;

import de.fzi.wim.oimodeler.oimodelgraph.*;

/**
 * A graph representing an OI-model.
 */
public class OIModelGraph extends AbstractGraph implements OIModelListener {
    /** The OI-model graph anchor. */
    protected OIModelGraphAnchor m_oimodelGraphAnchor;
    /** The list of all nodes. */
    protected List m_nodes;
    /** The list of entity nodes in the graph indexed by their Z-order. */
    protected List m_entityNodes;
    /** The list of non-entity nodes - these nodes come in the list before the entity nodes. */
    protected List m_nonEntityNodes;
    /** The map of nodes in the graph indexed by the OI-model entity. */
    protected Map m_nodesByEntity;
    /** The list of edges in the graph indexed by their Z-order. */
    protected List m_edges;
    /** The map of edges in the graph indexed by the NodePair object. */
    protected Map m_edgesByNodePairs;
    /** The language used for displaying the labels of the graph. */
    protected String m_languageURI;
    /** The change visitor for the OI-model graph. */
    protected OIModelGraphChangeVisitor m_changeVisitor;
    /** The map of prefixes with their namespaces. */
    protected Map m_namespacePrefixes;
    /** The set of nodes added since the events were suspended. */
    protected Set m_addedNodes;
    /** The set of nodes removed since the events were suspended. */
    protected Set m_removedNodes;
    /** The set of edges added since the events were suspended. */
    protected Set m_addedEdges;
    /** The set of edges removed since the events were suspended. */
    protected Set m_removedEdges;
    /** The number of times the events were suspended. */
    protected int m_suspendCount;
    /** Set to <code>true</code> if the graph is live. */
    protected boolean m_isLive;

    /**
     * Creates a graph and attaches it to the OI-model.
     *
     * @param oimodelGraphAnchor            the OI-model graph anchor
     * @param languageURI                   the URI of the language used for displaying the labels
     */
    public OIModelGraph(OIModelGraphAnchor oimodelGraphAnchor,String languageURI) {
        m_oimodelGraphAnchor=oimodelGraphAnchor;
        m_languageURI=languageURI;
        m_entityNodes=new LinkedList();
        m_nonEntityNodes=new LinkedList();
        m_nodes=new JoinedSequentialList(new List[] { m_nonEntityNodes,m_entityNodes });
        m_nodesByEntity=new HashMap();
        m_edges=new LinkedList();
        m_edgesByNodePairs=new HashMap();
        m_changeVisitor=new OIModelGraphChangeVisitor(this);
        m_namespacePrefixes=new HashMap();
        m_addedNodes=new HashSet();
        m_removedNodes=new HashSet();
        m_addedEdges=new HashSet();
        m_removedEdges=new HashSet();
        m_suspendCount=0;
    }
    /**
     * Determines whether this grpah is live.
     *
     * @param isLive                <code>true</code> if the graph should be live
     * @throws KAONException        thrown if there is a problem
     */
    public synchronized void setIsLive(boolean isLive) throws KAONException {
        if (isLive!=getIsLive()) {
            m_isLive=isLive;
            if (m_isLive) {
                m_oimodelGraphAnchor.getOIModel().addOIModelListener(this);
                m_oimodelGraphAnchor.getOIModel().refresh();
            }
            else {
                m_oimodelGraphAnchor.getOIModel().removeOIModelListener(this);
                m_entityNodes.clear();
                m_nonEntityNodes.clear();
                m_nodesByEntity.clear();
                m_edges.clear();
                m_edgesByNodePairs.clear();
            }
        }
    }
    /**
     * Returns <code>true</code> if the graph is live.
     *
     * @return                      <code>true</code> if the graph is live
     */
    public synchronized boolean getIsLive() {
        return m_isLive;
    }
    /**
     * Returns the OI-model of the given graph.
     *
     * @return                      the OI-model of the given graph
     */
    public synchronized OIModel getOIModel() {
        return m_oimodelGraphAnchor.getOIModel();
    }
    /**
     * Returns the OI-model graph anchor.
     *
     * @return                      the OI-model graph anchor
     */
    public synchronized OIModelGraphAnchor getOIModelGraphAnchor() {
        return m_oimodelGraphAnchor;
    }
    /**
     * Returns a collection of the nodes in the graph.
     *
     * @return                      a collection of the nodes in the graph
     */
    public synchronized List getNodes() {
        return m_nodes;
    }
    /**
     * Returns a list of entity nodes in the graph.
     *
     * @return                      the list of the entity nodes in the graph
     */
    public synchronized List getEntityNodes() {
        return m_entityNodes;
    }
    /**
     * Returns a list of non-entity nodes in the graph.
     *
     * @return                      the list of the non-entity nodes in the graph
     */
    public synchronized List getNonEntityNodes() {
        return m_nonEntityNodes;
    }
    /**
     * Returns a collection of the edges in the graph.
     *
     * @return                      the collection of the edges in the graph
     */
    public synchronized List getEdges() {
        return m_edges;
    }
    /**
     * Suspends the events on this graph.
     */
    public synchronized void suspendEvents() {
        m_suspendCount++;
    }
    /**
     * Resumes the events on this graph.
     */
    public synchronized void resumeEvents() {
        m_suspendCount--;
        fireBulkChangeNotifications();
    }
    /**
     * Fires the notifications about the bulk changes.
     */
    protected void fireBulkChangeNotifications() {
        if (m_suspendCount<=0) {
            Collection nodes=null;
            Collection edges=null;
            if (!m_removedNodes.isEmpty())
                nodes=m_removedNodes;
            if (!m_removedEdges.isEmpty())
                edges=m_removedEdges;
            if (nodes!=null || edges!=null)
                fireElementsRemoved(nodes,edges);
            m_removedNodes.clear();
            m_removedEdges.clear();
            nodes=null;
            edges=null;
            if (!m_addedNodes.isEmpty())
                nodes=m_addedNodes;
            if (!m_addedEdges.isEmpty())
                edges=m_addedEdges;
            if (nodes!=null || edges!=null)
                fireElementsAdded(nodes,edges);
            m_addedNodes.clear();
            m_addedEdges.clear();
        }
    }
    /**
     * Checks whether supplied node is in graph.
     *
     * @param node                  the node
     * @return                      <code>true</code> if the node is in the graph
     */
    public synchronized boolean contains(Node node) {
        return m_entityNodes.contains(node) || m_nonEntityNodes.contains(node);
    }
    /**
     * Adds a node to the graph.
     *
     * @param node                  the node added
     * @throws KAONException        thrown if node cannot be added
     */
    public synchronized void add(Node node) throws KAONException {
        boolean alreadyContainsNode=true;
        if (node instanceof EntityNode) {
            EntityNode entityNode=(EntityNode)node;
            Entity entity=entityNode.getEntity();
            if (!m_nodesByEntity.containsKey(entity)) {
                alreadyContainsNode=false;
                m_nodesByEntity.put(entityNode.getEntity(),entityNode);
                m_entityNodes.add(entityNode);
                entityNode.updateLabel(this);
            }
            if (entityNode instanceof InstanceNode) {
                InstanceNode instanceNode=(InstanceNode)entityNode;
                ConceptNode spanningConceptNode=(ConceptNode)getNodeForEntity(instanceNode.getInstance().getSpanningConcept());
                if (spanningConceptNode!=null && spanningConceptNode.getSpanningObjectsExpanded()!=ConceptNode.NOT_LOADED && SpanningObjectEdge.getEdge(this,spanningConceptNode,instanceNode)==null)
                    add(new SpanningObjectEdge(spanningConceptNode,instanceNode));
                PropertyNode spanningPropertyNode=(PropertyNode)getNodeForEntity(instanceNode.getInstance().getSpanningProperty());
                if (spanningPropertyNode!=null && spanningPropertyNode.getSpanningObjectsExpanded()!=ConceptNode.NOT_LOADED && SpanningObjectEdge.getEdge(this,spanningPropertyNode,instanceNode)==null)
                    add(new SpanningObjectEdge(spanningPropertyNode,instanceNode));
            }
            else if (entityNode instanceof ConceptNode) {
                ConceptNode conceptNode=(ConceptNode)entityNode;
                InstanceNode spanningInstanceNode=(InstanceNode)getNodeForEntity(conceptNode.getConcept().getSpanningInstance());
                if (spanningInstanceNode!=null && spanningInstanceNode.getSpanningObjectsExpanded()!=ConceptNode.NOT_LOADED && SpanningObjectEdge.getEdge(this,conceptNode,spanningInstanceNode)==null)
                    add(new SpanningObjectEdge(conceptNode,spanningInstanceNode));
            }
            else if (entityNode instanceof PropertyNode) {
                PropertyNode propertyNode=(PropertyNode)entityNode;
                InstanceNode spanningInstanceNode=(InstanceNode)getNodeForEntity(propertyNode.getProperty().getSpanningInstance());
                if (spanningInstanceNode!=null && spanningInstanceNode.getSpanningObjectsExpanded()!=ConceptNode.NOT_LOADED && SpanningObjectEdge.getEdge(this,propertyNode,spanningInstanceNode)==null)
                    add(new SpanningObjectEdge(propertyNode,spanningInstanceNode));
            }
        }
        else {
            if (!m_nonEntityNodes.contains(node)) {
                alreadyContainsNode=false;
                m_nonEntityNodes.add(node);
                if (node instanceof PropertyInstancesNode)
                    ((PropertyInstancesNode)node).updateLabel(this);
            }
        }
        if (!alreadyContainsNode) {
            if (!m_removedNodes.remove(node))
                m_addedNodes.add(node);
            fireBulkChangeNotifications();
        }
    }
    /**
     * Adds an edge to the graph.
     *
     * @param edge                  the edge added
     */
    public synchronized void add(OIModelEdge edge) {
        NodePair nodePair=new NodePair(edge.getFrom(),edge.getTo());
        if (!m_edgesByNodePairs.containsKey(nodePair)) {
            m_edgesByNodePairs.put(nodePair,edge);
            m_edges.add(edge);
            ((DefaultNode)edge.getFrom()).notifyEdgeAdded(edge);
            ((DefaultNode)edge.getTo()).notifyEdgeAdded(edge);
            if (!m_removedEdges.remove(edge))
                m_addedEdges.add(edge);
            fireBulkChangeNotifications();
        }
    }
    /**
     * Removes a node from the graph.
     *
     * @param node                  the node removed
     * @throws KAONException        thrown if node cannot be removed
     */
    public synchronized void remove(Node node) throws KAONException {
        // one of the lists will contain the node...
        if (m_entityNodes.remove(node) || m_nonEntityNodes.remove(node)) {
            if (node instanceof EntityNode) {
                EntityNode entityNode=(EntityNode)node;
                m_nodesByEntity.remove(entityNode.getEntity());
                if (entityNode instanceof InstanceNode) {
                    InstanceNode instanceNode=(InstanceNode)entityNode;
                    ConceptNode spanningConceptNode=(ConceptNode)getNodeForEntity(instanceNode.getInstance().getSpanningConcept());
                    if (spanningConceptNode!=null) {
                        SpanningObjectEdge spanningObjectEdge=SpanningObjectEdge.getEdge(this,spanningConceptNode,instanceNode);
                        if (spanningObjectEdge!=null)
                            remove(spanningObjectEdge);
                    }
                    PropertyNode spanningPropertyNode=(PropertyNode)getNodeForEntity(instanceNode.getInstance().getSpanningProperty());
                    if (spanningPropertyNode!=null) {
                        SpanningObjectEdge spanningObjectEdge=SpanningObjectEdge.getEdge(this,spanningPropertyNode,instanceNode);
                        if (spanningObjectEdge!=null)
                            remove(spanningObjectEdge);
                    }
                }
                else if (entityNode instanceof ConceptNode) {
                    ConceptNode conceptNode=(ConceptNode)entityNode;
                    InstanceNode spanningInstanceNode=(InstanceNode)getNodeForEntity(conceptNode.getConcept().getSpanningInstance());
                    if (spanningInstanceNode!=null) {
                        SpanningObjectEdge spanningObjectEdge=SpanningObjectEdge.getEdge(this,conceptNode,spanningInstanceNode);
                        if (spanningObjectEdge!=null)
                            remove(spanningObjectEdge);
                    }
                }
                else if (entityNode instanceof PropertyNode) {
                    PropertyNode propertyNode=(PropertyNode)entityNode;
                    InstanceNode spanningInstanceNode=(InstanceNode)getNodeForEntity(propertyNode.getProperty().getSpanningInstance());
                    if (spanningInstanceNode!=null) {
                        SpanningObjectEdge spanningObjectEdge=SpanningObjectEdge.getEdge(this,propertyNode,spanningInstanceNode);
                        if (spanningObjectEdge!=null)
                            remove(spanningObjectEdge);
                    }
                }
            }
            if (!m_addedNodes.remove(node))
                m_removedNodes.add(node);
            fireBulkChangeNotifications();
        }
    }
    /**
     * Removes an edge from the graph.
     *
     * @param edge                  the edge removed
     */
    public synchronized void remove(OIModelEdge edge) {
        NodePair nodePair=new NodePair(edge.getFrom(),edge.getTo());
        if (m_edgesByNodePairs.remove(nodePair)!=null) {
            m_edges.remove(edge);
            ((DefaultNode)edge.getFrom()).notifyEdgeRemoved(edge);
            ((DefaultNode)edge.getTo()).notifyEdgeRemoved(edge);
            if (!m_addedEdges.remove(edge))
                m_removedEdges.add(edge);
            fireBulkChangeNotifications();
        }
    }
    /**
     * Clears the graph.
     */
    public synchronized void clear() {
        m_entityNodes.clear();
        m_nonEntityNodes.clear();
        m_nodesByEntity.clear();
        m_edges.clear();
        m_edgesByNodePairs.clear();
        fireGraphContentsChanged();
    }
    /**
     * Sets the language for the graph.
     *
     * @param languageURI                   the URI of the new language
     * @throws KAONException                thrown if there is a problem
     */
    public synchronized void setLanguageURI(String languageURI) throws KAONException {
        m_languageURI=languageURI;
        Iterator nodes=getNodes().iterator();
        while (nodes.hasNext()) {
            Node node=(Node)nodes.next();
            if (node instanceof EntityNode) {
                EntityNode entityNode=(EntityNode)node;
                entityNode.updateLabel(this);
            }
            else if (node instanceof PropertyInstancesNode)
                ((PropertyInstancesNode)node).updateLabel(this);
        }
        notifyUpdated();
    }
    /**
     * Returns the current language URI.
     *
     * @return                              the URI of the current language
     */
    public synchronized String getLanguageURI() {
        return m_languageURI;
    }
    /**
     * Returns a node for given entity.
     *
     * @param entity                        the entity
     * @return                              the node for the entity (or <code>null</code> if the node doesn't exist)
     */
    public synchronized EntityNode getNodeForEntity(Entity entity) {
        return (EntityNode)m_nodesByEntity.get(entity);
    }
    /**
     * Returns an edge from supplied node to supplied node.
     *
     * @param from                          the node from which the edge points
     * @param to                            the node to which  the edge points
     * @return                              the edge between nodes (or <code>null</code> if no such edge exists)
     */
    public synchronized OIModelEdge getEdgeBetweenNodes(Node from,Node to) {
        NodePair nodePair=new NodePair(from,to);
        return (OIModelEdge)m_edgesByNodePairs.get(nodePair);
    }
    /**
     * Clears the graph and displays the root concept given entity at the (0,0) coordinates.
     *
     * @throws KAONException                thrown if there is a problem
     */
    public synchronized void displayRootConcept() throws KAONException {
        clear();
        ConceptNode rootConceptNode=new ConceptNode(m_oimodelGraphAnchor.getOIModel().getRootConcept(),null);
        add(rootConceptNode);
    }
    /**
     * Moves given node into foreground.
     *
     * @param node                          the node that is moved into the foreground
     */
    public synchronized void moveIntoForeground(Node node) {
        if (!moveIntoForegroud(node,m_entityNodes))
            if (!moveIntoForegroud(node,m_nonEntityNodes))
                return;
       notifyUpdated();
    }
    /**
     * Processes given list and moves the supplied node into foreground.
     *
     * @param node                          the node that is moved into the foreground
     * @param list                          the list
     * @return                              <code>true</code> if the node was found and moved
     */
    protected boolean moveIntoForegroud(Node node,List list) {
        Iterator iterator=list.iterator();
        while (iterator.hasNext()) {
            Node current=(Node)iterator.next();
            if (node==current) {
                iterator.remove();
                list.add(node);
                return true;
            }
        }
        return false;
    }
    /**
     * Removes all nodes from the graph apart from those passed in the collection.
     *
     * @param entities                      the collection of entities to show
     * @throws KAONException                thrown if there is an error
     */
    public synchronized void showOnlyEntities(Collection entities) throws KAONException {
        Set nodesToRemove=new HashSet(m_entityNodes);
        nodesToRemove.addAll(m_nonEntityNodes);
        Set edgesToRemove=new HashSet(m_edges);
        Iterator iterator=entities.iterator();
        while (iterator.hasNext()) {
            Entity entity=(Entity)iterator.next();
            EntityNode entityNode=getNodeForEntity(entity);
            if (entityNode!=null) {
                nodesToRemove.remove(entityNode);
                entityNode.resetExpandedState();
            }
        }
        iterator=edgesToRemove.iterator();
        while (iterator.hasNext()) {
            Edge edge=(Edge)iterator.next();
            if (!nodesToRemove.contains(edge.getFrom()) && !nodesToRemove.contains(edge.getTo()))
                iterator.remove();
        }
        suspendEvents();
        iterator=nodesToRemove.iterator();
        while (iterator.hasNext()) {
            Node node=(Node)iterator.next();
            remove(node);
        }
        iterator=edgesToRemove.iterator();
        while (iterator.hasNext()) {
            OIModelEdge edge=(OIModelEdge)iterator.next();
            remove(edge);
        }
        resumeEvents();
    }
    /**
     * Hides given entities.
     *
     * @param entities                      the collection of entities to hide
     * @throws KAONException                thrown if there is an error
     */
    public synchronized void hideEntities(Collection entities) throws KAONException {
        Collection nodes=new HashSet();
        Iterator iterator=entities.iterator();
        while (iterator.hasNext()) {
            Entity entity=(Entity)iterator.next();
            EntityNode entityNode=getNodeForEntity(entity);
            if (entityNode!=null)
                nodes.add(entityNode);
        }
        hideNodes(nodes);
    }
    /**
     * Hides given nodes.
     *
     * @param nodes                         the collection of nodes to hide
     * @throws KAONException                thrown if there is an error
     */
    public synchronized void hideNodes(Collection nodes) throws KAONException {
        try {
            suspendEvents();
            List edges=new LinkedList();
            Iterator iterator=nodes.iterator();
            while (iterator.hasNext()) {
                Node node=(Node)iterator.next();
                edges.addAll(node.getEdgesFrom());
                edges.addAll(node.getEdgesTo());
            }
            iterator=edges.iterator();
            while (iterator.hasNext()) {
                OIModelEdge oimodelEdge=(OIModelEdge)iterator.next();
                oimodelEdge.hideEdge(this);
            }
            iterator=nodes.iterator();
            while (iterator.hasNext()) {
                Node node=(Node)iterator.next();
                remove(node);
            }
        }
        finally {
            resumeEvents();
        }
    }
    /**
     * Disposes of this graph.
     */
    public synchronized void dispose() {
        if (m_oimodelGraphAnchor!=null) {
            try {
                setIsLive(false);
            }
            catch (KAONException ignored) {
            }
            m_oimodelGraphAnchor=null;
        }
    }
    /**
     * Called when a bulk of change events is processed in the pool.
     *
     * @param oimodel                   OI-model that was changed
     * @param changeEvents              list of change events that occured
     */
    public synchronized void modelChanged(OIModel oimodel,List changeEvents) {
        try {
            m_changeVisitor.applyChanges(changeEvents);
        }
        catch (KAONException e) {
        }
        notifyUpdated();
    }
    /**
     * Called when model is refreshed.
     *
     * @param oimodel                   OI-model that was refreshed
     */
    public synchronized void modelRefreshed(OIModel oimodel) {
        try {
            displayRootConcept();
        }
        catch (KAONException e) {
        }
    }
    /**
     * Called when model is being deleted.
     *
     * @param oimodel                   OI-model that is being deleted
     */
    public synchronized void modelDeleted(OIModel oimodel) {
    }
    /**
     * Registers a namespace prefix.
     *
     * @param namespace                 the namespace
     * @param prefix                    the prefix
     */
    public synchronized void registerNamespacePrefix(String namespace,String prefix) {
        m_namespacePrefixes.put(namespace,prefix);
    }
    /**
     * Possibly replaces the namespace prefix within the URI.
     *
     * @param uri                       the URI
     * @return                          the URI with possibly replaced namespace
     */
    public synchronized String replaceURIPrefix(String uri) {
        int namespaceEnd=uri.length()-1;
        do {
            char c=uri.charAt(namespaceEnd);
            if (c=='#' || c==':' || c=='/')
                break;
            namespaceEnd--;
        } while (namespaceEnd>=0);
        namespaceEnd++;
        String namespace=uri.substring(0,namespaceEnd);
        String prefix=(String)m_namespacePrefixes.get(namespace);
        if (prefix!=null) {
            String localName=uri.substring(namespaceEnd);
            return prefix+":"+localName;
        }
        else
            return uri;
    }

    /**
     * The node pair structure.
     */
    protected static class NodePair {
        public Node m_from;
        public Node m_to;
        protected int m_hashCode;

        public NodePair(Node from,Node to) {
            m_from=from;
            m_to=to;
            m_hashCode=m_from.hashCode()+7*m_to.hashCode();
        }
        public int hashCode() {
            return m_hashCode;
        }
        public boolean equals(Object that) {
            if (this==that)
                return true;
            if (!(that instanceof NodePair))
                return false;
            NodePair thatNodePair=(NodePair)that;
            return m_from.equals(thatNodePair.m_from) && m_to.equals(thatNodePair.m_to);
        }
    }

    /**
     * A virtual join of two lists.
     */
    protected class JoinedSequentialList extends AbstractSequentialList {
        /** The lists. */
        protected List[] m_lists;

        public JoinedSequentialList(List[] lists) {
            m_lists=new List[lists.length];
            for (int i=0;i<lists.length;i++)
                m_lists[i]=lists[i];
        }
        public ListIterator listIterator(int index) {
            return new JoinedSequentialListIterator(index);
        }
        public int size() {
            int size=0;
            for (int i=0;i<m_lists.length;i++)
                size+=m_lists[i].size();
            return size;
        }

        /**
         * An iterator for the joined list.
         */
        public class JoinedSequentialListIterator implements ListIterator {
            protected ListIterator[] m_iterators;
            protected int m_current;
            protected int m_index;

            public JoinedSequentialListIterator(int index) {
                m_iterators=new ListIterator[m_lists.length];
                m_current=0;
                int processed=0;
                for (int i=0;i<m_lists.length;i++) {
                    if (index>=processed) {
                        int maxIndex=Math.min(index-processed,m_lists[i].size());
                        m_iterators[i]=m_lists[i].listIterator(maxIndex);
                        m_current=i;
                    }
                    else
                        m_iterators[i]=m_lists[i].listIterator();
                    processed+=m_lists[i].size();
                }
                m_index=index;
            }
            public boolean hasNext() {
                return m_iterators[m_current].hasNext();
            }
            public Object next() {
                Object result=m_iterators[m_current].next();
                if (!m_iterators[m_current].hasNext() && m_current<m_iterators.length-1)
                    m_current++;
                m_index++;
                return result;
            }
            public boolean hasPrevious() {
                return m_iterators[m_current].hasPrevious();
            }
            public Object previous() {
                Object result=m_iterators[m_current].previous();
                if (!m_iterators[m_current].hasPrevious() && m_current>0)
                    m_current--;
                m_index++;
                return result;
            }
            public int nextIndex() {
                return m_index;
            }
            public int previousIndex() {
                return m_index-1;
            }
            public void remove() {
                throw new UnsupportedOperationException();
            }
            public void set(Object o) {
                throw new UnsupportedOperationException();
            }
            public void add(Object o) {
                throw new UnsupportedOperationException();
            }
        }
    }
}
