package de.fzi.wim.oimodeler.oimodelgraph.graph;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.guibase.graphview.graph.*;

import de.fzi.wim.oimodeler.viewfilter.*;

/**
 * A node in the OI-model graph representing an instance.
 */
public class InstanceNode extends EntityNode {
    /** The expansion state of the parent concepts. */
    protected int m_parentConceptsExpanded;
    /** The expansion state of the property instances from. */
    protected int m_propertyInstancesFromExpanded;
    /** The expansion state of the property instances to. */
    protected int m_propertyInstancesToExpanded;
    /** The map of PropertyInstancesNode objects indexed by properties. */
    protected Map m_propertyInstancesNodes;

    /**
     * Creates a node for given entity.
     *
     * @param instance                      the instance of the node
     * @param originalNode                  the original node (may be <code>null</code>)
     * @throws KAONException                thrown if there is an error
     */
    public InstanceNode(Instance instance,Node originalNode) throws KAONException {
        super(instance,originalNode);
        m_propertyInstancesNodes=new HashMap();
    }
    /**
     * Returns the instance of this node.
     *
     * @return                              the instance of this node
     */
    public Instance getInstance() {
        return (Instance)m_entity;
    }
    /**
     * Expands the spanning objects obf this entity.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandSpanningObjects(final OIModelGraph oimodelGraph) throws KAONException {
        m_spanningObjectsExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Concept m_spanningConcept;
            protected Property m_spanningProperty;
            public void backgroundJob() throws KAONException {
                m_spanningConcept=getInstance().getSpanningConcept();
                m_spanningProperty=getInstance().getSpanningProperty();
                Set objects=new HashSet();
                if ((m_spanningConcept.getLoadedState() & OIModel.LOAD_CONCEPT_BASICS)==0)
                    objects.add(m_spanningConcept);
                if ((m_spanningProperty.getLoadedState() & OIModel.LOAD_PROPERTY_BASICS)==0)
                    objects.add(m_spanningProperty);
                if (!objects.isEmpty())
                    oimodelGraph.getOIModel().loadObjects(objects,OIModel.LOAD_LEXICON | OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_PROPERTY_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(InstanceNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                if (m_spanningConcept.isInOIModel()) {
                                    ConceptNode spanningConceptNode=(ConceptNode)oimodelGraph.getNodeForEntity(m_spanningConcept);
                                    if (spanningConceptNode==null) {
                                        spanningConceptNode=new ConceptNode(m_spanningConcept,InstanceNode.this);
                                        oimodelGraph.add(spanningConceptNode);
                                    }
                                    if (SpanningObjectEdge.getEdge(oimodelGraph,spanningConceptNode,InstanceNode.this)==null)
                                        oimodelGraph.add(new SpanningObjectEdge(spanningConceptNode,InstanceNode.this));
                                }
                                if (m_spanningProperty.isInOIModel()) {
                                    PropertyNode spanningPropertyNode=(PropertyNode)oimodelGraph.getNodeForEntity(m_spanningProperty);
                                    if (spanningPropertyNode==null) {
                                        spanningPropertyNode=new PropertyNode(m_spanningProperty,InstanceNode.this);
                                        oimodelGraph.add(spanningPropertyNode);
                                    }
                                    if (SpanningObjectEdge.getEdge(oimodelGraph,spanningPropertyNode,InstanceNode.this)==null)
                                        oimodelGraph.add(new SpanningObjectEdge(spanningPropertyNode,InstanceNode.this));
                                }
                                m_spanningObjectsExpanded=LOADED;
                                oimodelGraph.notifyUpdated();
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_spanningObjectsExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion state of the parent concepts.
     *
     * @return                              the expansion state of the parent concepts
     */
    public int getParentConceptsExpanded() {
        return m_parentConceptsExpanded;
    }
    /**
     * Expands the parent concepts in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandParentConcepts(final OIModelGraph oimodelGraph) throws KAONException {
        m_parentConceptsExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_parentConcepts;
            public void backgroundJob() throws KAONException {
                m_parentConcepts=getInstance().getParentConcepts();
                oimodelGraph.getOIModel().loadObjects(m_parentConcepts,OIModel.LOAD_LEXICON | OIModel.LOAD_CONCEPT_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(InstanceNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                                Concept rootConcept=oimodelGraph.getOIModel().getRootConcept();
                                Iterator iterator=m_parentConcepts.iterator();
                                while (iterator.hasNext()) {
                                    Concept parentConcept=(Concept)iterator.next();
                                    if (viewFilter.showEntity(parentConcept) && !parentConcept.equals(rootConcept)) {
                                        ConceptNode parentConceptNode=(ConceptNode)oimodelGraph.getNodeForEntity(parentConcept);
                                        if (parentConceptNode==null) {
                                            parentConceptNode=new ConceptNode(parentConcept,InstanceNode.this);
                                            oimodelGraph.add(parentConceptNode);
                                        }
                                        parentConceptNode.connectInstanceNode(oimodelGraph,InstanceNode.this);
                                    }
                                }
                                m_parentConceptsExpanded=LOADED;
                                oimodelGraph.notifyUpdated();
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_parentConceptsExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion state of the property instances from.
     *
     * @return                              the expansion state of the property instances from
     */
    public int getPropertyInstancesFromExpanded() {
        return m_propertyInstancesFromExpanded;
    }
    /**
     * Expands the property instances from the concept in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @param completion                    the completion for the loading of property instances
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandPropertyInstancesFrom(final OIModelGraph oimodelGraph,final PropertyInstanceLoadingCompletion completion) throws KAONException {
        m_propertyInstancesFromExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_propertyInstancesFrom;
            public void backgroundJob() throws KAONException {
                m_propertyInstancesFrom=getInstance().getFromPropertyInstances();
                Set set=new HashSet();
                Iterator iterator=m_propertyInstancesFrom.iterator();
                while (iterator.hasNext()) {
                    PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                    set.add(propertyInstance.getProperty());
                }
                oimodelGraph.getOIModel().loadObjects(set,OIModel.LOAD_LEXICON | OIModel.LOAD_PROPERTY_BASICS);
            }
            public void completionJob() {
                m_propertyInstancesFromExpanded=NOT_LOADED;
                oimodelGraph.notifyUpdated();
                completion.propertyInstancesLoaded(m_propertyInstancesFrom);
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_propertyInstancesFromExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Expands the given property instances from the concept in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @param propertyInstances             set of instances to expand
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandPropertyInstancesFrom(OIModelGraph oimodelGraph,Set propertyInstances) throws KAONException {
        synchronized (oimodelGraph) {
            synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                if (oimodelGraph.contains(this))
                    try {
                        oimodelGraph.suspendEvents();
                        ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                        Iterator iterator=propertyInstances.iterator();
                        while (iterator.hasNext()) {
                            PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                            if (viewFilter.showPropertyInstance(propertyInstance))
                                addPropertyInstance(oimodelGraph,propertyInstance);
                        }
                    }
                    finally {
                        oimodelGraph.resumeEvents();
                        oimodelGraph.notifyUpdated();
                    }
            }
        }
    }
    /**
     * Notifies the node that all property instances from are expanded.
     *
     * @param oimodelGraph                  the graph
     */
    public void notifyAllPropertyInstancesFromExpanded(OIModelGraph oimodelGraph) {
        synchronized (oimodelGraph) {
            m_propertyInstancesFromExpanded=LOADED;
            oimodelGraph.notifyUpdated();
        }
    }
    /**
     * Expands the property instances to the concept in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @param completion                    the completion for the loading of property instances
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandPropertyInstancesTo(final OIModelGraph oimodelGraph,final PropertyInstanceLoadingCompletion completion) throws KAONException {
        m_propertyInstancesToExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_propertyInstancesTo;
            public void backgroundJob() throws KAONException {
                m_propertyInstancesTo=getInstance().getToPropertyInstances();
                Set set=new HashSet();
                Iterator iterator=m_propertyInstancesTo.iterator();
                while (iterator.hasNext()) {
                    PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                    set.add(propertyInstance.getProperty());
                }
                oimodelGraph.getOIModel().loadObjects(set,OIModel.LOAD_LEXICON | OIModel.LOAD_PROPERTY_BASICS);
            }
            public void completionJob() {
                m_propertyInstancesToExpanded=NOT_LOADED;
                oimodelGraph.notifyUpdated();
                completion.propertyInstancesLoaded(m_propertyInstancesTo);
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_propertyInstancesToExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Expands the given property instances to the concept in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @param propertyInstances             set of instances to expand
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandPropertyInstancesTo(OIModelGraph oimodelGraph,Set propertyInstances) throws KAONException {
        synchronized (oimodelGraph) {
            synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                if (oimodelGraph.contains(this))
                    try {
                        oimodelGraph.suspendEvents();
                        ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                        Iterator iterator=propertyInstances.iterator();
                        while (iterator.hasNext()) {
                            PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                            if (viewFilter.showPropertyInstance(propertyInstance)) {
                                InstanceNode sourceInstanceNode=(InstanceNode)oimodelGraph.getNodeForEntity(propertyInstance.getSourceInstance());
                                if (sourceInstanceNode==null) {
                                    sourceInstanceNode=new InstanceNode(propertyInstance.getSourceInstance(),InstanceNode.this);
                                    oimodelGraph.add(sourceInstanceNode);
                                }
                                sourceInstanceNode.addPropertyInstance(oimodelGraph,propertyInstance);
                            }
                        }
                    }
                    finally {
                        oimodelGraph.resumeEvents();
                        oimodelGraph.notifyUpdated();
                    }
            }
        }
    }
    /**
     * Notifies the node that all property instances to are expanded.
     *
     * @param oimodelGraph                  the graph
     */
    public void notifyAllPropertyInstancesToExpanded(OIModelGraph oimodelGraph) {
        synchronized (oimodelGraph) {
            m_propertyInstancesToExpanded=LOADED;
            oimodelGraph.notifyUpdated();
        }
    }
    /**
     * Returns the expansion state of the property instances to.
     *
     * @return                              the expansion state of the property instances to
     */
    public int getPropertyInstancesToExpanded() {
        return m_propertyInstancesToExpanded;
    }
    /**
     * Adds a property instance to this node.
     *
     * @param oimodelGraph                  the graph
     * @param propertyInstance              the property instance
     * @throws KAONException                thrown if there is a problem
     */
    public void addPropertyInstance(OIModelGraph oimodelGraph,PropertyInstance propertyInstance) throws KAONException {
        synchronized (oimodelGraph) {
            try {
                oimodelGraph.suspendEvents();
                Property property=propertyInstance.getProperty();
                PropertyInstancesNode propertyInstancesNode=(PropertyInstancesNode)m_propertyInstancesNodes.get(property);
                if (propertyInstancesNode==null) {
                    propertyInstancesNode=new PropertyInstancesNode(property,(Instance)m_entity,this);
                    m_propertyInstancesNodes.put(property,propertyInstancesNode);
                    oimodelGraph.add(propertyInstancesNode);
                    oimodelGraph.add(new PropertyInstanceEdge(this,propertyInstancesNode));
                }
                propertyInstancesNode.addPropertyInstance(oimodelGraph,propertyInstance);
            }
            finally {
                oimodelGraph.resumeEvents();
            }
        }
    }
    /**
     * Removes a property instance from this node.
     *
     * @param oimodelGraph                  the graph
     * @param propertyInstance              the property instance
     * @throws KAONException                thrown if there is a problem
     */
    public void removePropertyInstance(OIModelGraph oimodelGraph,PropertyInstance propertyInstance) throws KAONException {
        synchronized (oimodelGraph) {
            try {
                oimodelGraph.suspendEvents();
                Property property=propertyInstance.getProperty();
                PropertyInstancesNode propertyInstancesNode=(PropertyInstancesNode)m_propertyInstancesNodes.get(property);
                if (propertyInstancesNode!=null) {
                    if (propertyInstancesNode.removePropertyInstance(oimodelGraph,propertyInstance)) {
                        m_propertyInstancesNodes.remove(property);
                        oimodelGraph.remove(oimodelGraph.getEdgeBetweenNodes(this,propertyInstancesNode));
                        oimodelGraph.remove(propertyInstancesNode);
                    }
                }
            }
            finally {
                oimodelGraph.resumeEvents();
            }
        }
    }
    /**
     * Changes a property instance at this node.
     *
     * @param oimodelGraph                  the graph
     * @param propertyInstance              the property instance
     * @param newTargetValue                the new target value of the instance
     * @throws KAONException                thrown if there is a problem
     */
    public void changePropertyInstance(OIModelGraph oimodelGraph,PropertyInstance propertyInstance,Object newTargetValue) throws KAONException {
        synchronized (oimodelGraph) {
            try {
                oimodelGraph.suspendEvents();
                Property property=propertyInstance.getProperty();
                PropertyInstancesNode propertyInstancesNode=(PropertyInstancesNode)m_propertyInstancesNodes.get(property);
                if (propertyInstancesNode!=null)
                    propertyInstancesNode.changePropertyInstance(oimodelGraph,propertyInstance,newTargetValue);
            }
            finally {
                oimodelGraph.resumeEvents();
            }
        }
    }
    /**
     * Resets the node's expanded states.
     */
    public void resetExpandedState() {
        super.resetExpandedState();
        m_parentConceptsExpanded=NOT_LOADED;
        m_propertyInstancesFromExpanded=NOT_LOADED;
        m_propertyInstancesToExpanded=NOT_LOADED;
        m_propertyInstancesNodes.clear();
    }

    /**
     * An object of this type is called whenever property instances are loaded.
     */
    public static interface PropertyInstanceLoadingCompletion {
        void propertyInstancesLoaded(Set set);
    }
}
