package de.fzi.wim.oimodeler.oimodelgraph.graph;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;

import de.fzi.wim.guibase.graphview.graph.*;

/**
 * A manager for nodes that use fan-out to display related nodes.
 */
public abstract class FanOutManager {
    /** Maximum number of children. */
    public static final int MAXIMUM_NUMBER_OF_CHILDREN=5;
    /** The owner of the fanout. */
    protected EntityNode m_owner;
    /** The root of the fanout. */
    protected FanOutNode m_root;
    /** The map of the leaves to their fan-out parents. */
    protected Map m_leavesToFanOut;

    /**
     * Creates an instance of this class.
     *
     * @param owner                         the owner of the fanout
     */
    public FanOutManager(EntityNode owner) {
        m_owner=owner;
        m_root=null;
        m_leavesToFanOut=new HashMap();
    }
    /**
     * Adds a new node to the fanout subtree.
     *
     * @param oimodelGraph                  the graph in which the nodes live
     * @param node                          the new node to be added
     * @throws KAONException                thrown if there is a problem
     */
    public void addNode(OIModelGraph oimodelGraph,Node node) throws KAONException {
        synchronized (oimodelGraph) {
            if (m_leavesToFanOut.containsKey(node))
                return;
            try {
                oimodelGraph.suspendEvents();
                if (m_root==null) {
                    m_root=new FanOutNode(m_owner.getEntity(),m_owner,null);
                    oimodelGraph.add(m_root);
                    oimodelGraph.add(createEdge(m_root,m_owner));
                }
                List queue=new LinkedList();
                queue.add(m_root);
                while (true) {
                    FanOutNode current=(FanOutNode)queue.remove(0);
                    if (current.getChildren().size()<MAXIMUM_NUMBER_OF_CHILDREN) {
                        oimodelGraph.add(createEdge(node,current));
                        m_leavesToFanOut.put(node,current);
                        current.getChildren().add(node);
                        return;
                    }
                    Iterator children=current.getChildren().iterator();
                    while (children.hasNext()) {
                        Node child=(Node)children.next();
                        if (child instanceof FanOutNode)
                            queue.add(child);
                        else {
                            oimodelGraph.remove(oimodelGraph.getEdgeBetweenNodes(child,current));
                            current.getChildren().remove(child);
                            FanOutNode newFanOutNode=new FanOutNode(m_owner.getEntity(),current,current);
                            oimodelGraph.add(newFanOutNode);
                            oimodelGraph.add(createEdge(newFanOutNode,current));
                            oimodelGraph.add(createEdge(child,newFanOutNode));
                            oimodelGraph.add(createEdge(node,newFanOutNode));
                            m_leavesToFanOut.put(child,newFanOutNode);
                            m_leavesToFanOut.put(node,newFanOutNode);
                            current.getChildren().add(newFanOutNode);
                            newFanOutNode.getChildren().add(child);
                            newFanOutNode.getChildren().add(node);
                            return;
                        }
                    }
                }
            }
            finally {
                oimodelGraph.resumeEvents();
            }
        }
    }
    /**
     * Removes a new node from the fanout subtree.
     *
     * @param oimodelGraph                  the graph in which the nodes live
     * @param node                          the node to be removed
     * @throws KAONException                thrown if there is a problem
     */
    public void removeNode(OIModelGraph oimodelGraph,Node node) throws KAONException {
        synchronized (oimodelGraph) {
            if (m_root==null)
                return;
            FanOutNode fanOutNode=(FanOutNode)m_leavesToFanOut.remove(node);
            if (fanOutNode==null)
                return;
            try {
                oimodelGraph.suspendEvents();
                oimodelGraph.remove(oimodelGraph.getEdgeBetweenNodes(node,fanOutNode));
                fanOutNode.getChildren().remove(node);
                if (fanOutNode.getChildren().size()==1) {
                    FanOutNode parentNode=fanOutNode.getParentNode();
                    Node child=(Node)fanOutNode.getChildren().iterator().next();
                    if (parentNode!=null) {
                        fanOutNode.getChildren().remove(child);
                        parentNode.getChildren().add(child);
                        if (child instanceof FanOutNode)
                            ((FanOutNode)child).setParentNode(parentNode);
                        else
                            m_leavesToFanOut.put(child,parentNode);
                        oimodelGraph.remove(oimodelGraph.getEdgeBetweenNodes(child,fanOutNode));
                        oimodelGraph.add(createEdge(child,parentNode));
                    }
                    else if (child instanceof FanOutNode) {
                        fanOutNode.getChildren().remove(child);
                        m_root=(FanOutNode)child;
                        m_root.setParentNode(null);
                        oimodelGraph.remove(oimodelGraph.getEdgeBetweenNodes(child,fanOutNode));
                        oimodelGraph.add(createEdge(child,m_owner));
                    }
                }
                if (fanOutNode.getChildren().isEmpty()) {
                    oimodelGraph.remove(fanOutNode);
                    FanOutNode parentNode=fanOutNode.getParentNode();
                    if (parentNode!=null) {
                        oimodelGraph.remove(oimodelGraph.getEdgeBetweenNodes(fanOutNode,parentNode));
                        parentNode.getChildren().remove(fanOutNode);
                    }
                    else {
                        oimodelGraph.remove(oimodelGraph.getEdgeBetweenNodes(fanOutNode,m_owner));
                        if (m_root==fanOutNode)
                            m_root=null;
                    }
                }
            }
            finally {
                oimodelGraph.resumeEvents();
            }
        }
    }
    /**
     * Clears this fan-out manager.
     */
    public void clear() {
        m_root=null;
        m_leavesToFanOut.clear();
    }
    /**
     * Creates an edge between two nodes.
     *
     * @param from                          the node from which the edge points
     * @param to                            the node to which the edge points
     * @return                              the new edge
     */
    protected abstract OIModelEdge createEdge(Node from,Node to);
}
