package de.fzi.wim.oimodeler.oimodelgraph.graph;

import java.util.Random;
import javax.swing.SwingUtilities;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.guibase.graphview.graph.*;

/**
 * A node in the OI-model graph representing an entity.
 */
public abstract class EntityNode extends DefaultNode {
    /** Specifies that elements haven't been loaded. */
    public static final int NOT_LOADED=0;
    /** Specifies that elements are being loaded. */
    public static final int LOADING=1;
    /** Specifies that elements are loaded. */
    public static final int LOADED=2;
    /** The random generator. */
    protected static Random s_random=new Random();

    /** The entity stored in this node. */
    protected Entity m_entity;
    /** The source OI-model of the entity. */
    protected OIModel m_sourceOIModel;
    /** The label of the node. */
    protected String m_label;
    /** Set to <code>true</code> if the spanning objects of this object are expanded. */
    protected int m_spanningObjectsExpanded;

    /**
     * Creates a node for given entity.
     *
     * @param entity                        the entity of the node
     * @param originalNode                  the original node (may be <code>null</code>)
     * @throws KAONException                thrown if there is an error
     */
    public EntityNode(Entity entity,Node originalNode) throws KAONException {
        m_entity=entity;
        m_sourceOIModel=m_entity.getSourceOIModel();
        if (originalNode!=null)
            setLocation(originalNode.getX()+s_random.nextInt(60)-30,originalNode.getY()+s_random.nextInt(60)-30);
    }
    /**
     * Returns the entity of this node.
     *
     * @return                              the entity of this node
     */
    public Entity getEntity() {
        return m_entity;
    }
    /**
     * Updates the label for given language.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void updateLabel(OIModelGraph oimodelGraph) throws KAONException {
        m_label=m_entity.getLabel(oimodelGraph.getLanguageURI());
        if (m_label==null)
            m_label=oimodelGraph.replaceURIPrefix(m_entity.getURI());
    }
    /**
     * Return the label of this node.
     *
     * @return                               the label of this node
     */
    public String getLabel() {
        return m_label;
    }
    /**
     * Returns whether spanning objects were expanded.
     *
     * @return                              the status specifying whether spanning objects were expanded
     */
    public int getSpanningObjectsExpanded() {
        return m_spanningObjectsExpanded;
    }
    /**
     * Resets the node's expanded states.
     */
    public void resetExpandedState() {
        m_spanningObjectsExpanded=NOT_LOADED;
    }
    /**
     * Checks whether the entity of this node is declared in specified OI-model.
     *
     * @param oimodel                       the OI-model
     * @return                              <code>true</code> if the entity of this object is declared in supplied OI-model
     */
    public boolean isDeclaredIn(OIModel oimodel) {
        return m_sourceOIModel==oimodel;
    }
    /**
     * Processes the background action.
     *
     * @param oimodelGraph                  the graph
     * @param action                        the action to execute
     * @throws KAONException                thrown if there is an error
     */
    protected void doAction(final OIModelGraph oimodelGraph,final BackgroundAction action) throws KAONException {
        if ((oimodelGraph.getOIModelGraphAnchor().getOIModel().getCapabilities() & OIModel.CAPABILITY_SUPPORTS_OPTIMIZED_LOADING)==0) {
            try {
                action.backgroundJob();
                action.completionJob();
            }
            catch (KAONException e) {
                action.errorJob();
                throw e;
            }
        }
        else
            try {
                oimodelGraph.getOIModelGraphAnchor().getThreadPool().executeTask(new Runnable() {
                    public void run() {
                        try {
                            action.backgroundJob();
                            SwingUtilities.invokeLater(new Runnable() {
                                public void run() {
                                    try {
                                        action.completionJob();
                                    }
                                    catch (KAONException e) {
                                        action.errorJob();
                                        oimodelGraph.getOIModelGraphAnchor().notifyError(e);
                                    }
                                }
                            });
                        }
                        catch (final KAONException e) {
                            SwingUtilities.invokeLater(new Runnable() {
                                public void run() {
                                    action.errorJob();
                                    oimodelGraph.getOIModelGraphAnchor().notifyError(e);
                                }
                            });
                        }
                    }
                });
            }
            catch (InterruptedException e) {
                action.errorJob();
                throw new KAONException("Thread interrupted.",e);
            }
    }
    /**
     * Expands the spanning objects obf this entity.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public abstract void expandSpanningObjects(OIModelGraph oimodelGraph) throws KAONException;

    /**
     * The action that is to be processed in the background.
     */
    protected static interface BackgroundAction {
        /**
         * This method will be called in the background.
         *
         * @throws KAONException            thrown if there is a problem with accessing the ontology
         */
        void backgroundJob() throws KAONException;
        /**
         * This method will be called in the foreground after the background job.
         *
         * @throws KAONException            thrown if there is a problem with accessing the ontology
         */
        void completionJob() throws KAONException;
        /**
         * This method will be called in the foreground after an error is encountered.
         *
         * @throws KAONException            thrown if there is a problem with accessing the ontology
         */
        void errorJob();
    }
}
