package de.fzi.wim.oimodeler.oimodelgraph.graph;

import java.util.Set;
import java.util.Iterator;
import java.util.Collections;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.guibase.graphview.graph.*;

import de.fzi.wim.oimodeler.viewfilter.*;

/**
 * A node in the OI-model graph representing a concept.
 */
public class ConceptNode extends EntityNode {
    /** The expansion status of the subconcepts. */
    protected int m_subConceptsExpanded;
    /** The expansion status of the superconcepts. */
    protected int m_superConceptsExpanded;
    /** The expansion status of the properties from. */
    protected int m_propertiesFromExpanded;
    /** The expansion status of the properties to. */
    protected int m_propertiesToExpanded;
    /** The expansion status of the concept instances. */
    protected int m_instancesExpanded;
    /** The fan-out manager for instances. */
    protected FanOutManager m_fanOutManager;

    /**
     * Creates a node for given entity.
     *
     * @param concept                       the concept of the node
     * @param originalNode                  the original node (may be <code>null</code>)
     * @throws KAONException                thrown if there is an error
     */
    public ConceptNode(Concept concept,Node originalNode) throws KAONException {
        super(concept,originalNode);
        m_fanOutManager=new FanOutManager(this) {
            protected OIModelEdge createEdge(Node from,Node to) {
                return new ConceptInstanceEdge(from,to);
            }
        };
    }
    /**
     * Returns the concept of this node.
     *
     * @return                              the concept of this node
     */
    public Concept getConcept() {
        return (Concept)m_entity;
    }
    /**
     * Expands the spanning objects obf this entity.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandSpanningObjects(final OIModelGraph oimodelGraph) throws KAONException {
        m_spanningObjectsExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Instance m_spanningInstance;
            public void backgroundJob() throws KAONException {
                m_spanningInstance=getConcept().getSpanningInstance();
                if ((m_spanningInstance.getLoadedState() & OIModel.LOAD_INSTANCE_BASICS)==0)
                    oimodelGraph.getOIModel().loadObjects(Collections.singleton(m_spanningInstance),OIModel.LOAD_LEXICON | OIModel.LOAD_INSTANCE_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(ConceptNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                if (m_spanningInstance.isInOIModel()) {
                                    InstanceNode spanningInstanceNode=(InstanceNode)oimodelGraph.getNodeForEntity(m_spanningInstance);
                                    if (spanningInstanceNode==null) {
                                        spanningInstanceNode=new InstanceNode(m_spanningInstance,ConceptNode.this);
                                        oimodelGraph.add(spanningInstanceNode);
                                    }
                                    if (SpanningObjectEdge.getEdge(oimodelGraph,ConceptNode.this,spanningInstanceNode)==null)
                                        oimodelGraph.add(new SpanningObjectEdge(ConceptNode.this,spanningInstanceNode));
                                }
                                m_spanningObjectsExpanded=LOADED;
                                oimodelGraph.notifyUpdated();
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_spanningObjectsExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion status of the subconcepts.
     *
     * @return                              the expansion status of subconcepts
     */
    public int getSubConceptsExpanded() {
        return m_subConceptsExpanded;
    }
    /**
     * Expands the subconcepts in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandSubConcepts(final OIModelGraph oimodelGraph) throws KAONException {
        m_subConceptsExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_subConcepts;
            public void backgroundJob() throws KAONException {
                m_subConcepts=getConcept().getSubConcepts();
                oimodelGraph.getOIModel().loadObjects(m_subConcepts,OIModel.LOAD_LEXICON | OIModel.LOAD_CONCEPT_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(ConceptNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                                double edgeLength=ConceptHierarchyEdge.UNIT_LENGTH;
                                int count=0;
                                Iterator iterator=m_subConcepts.iterator();
                                while (iterator.hasNext()) {
                                    Concept subConcept=(Concept)iterator.next();
                                    if (viewFilter.showEntity(subConcept)) {
                                        ConceptNode subConceptNode=(ConceptNode)oimodelGraph.getNodeForEntity(subConcept);
                                        if (subConceptNode==null) {
                                            subConceptNode=new ConceptNode(subConcept,ConceptNode.this);
                                            oimodelGraph.add(subConceptNode);
                                        }
                                        if (ConceptHierarchyEdge.getEdge(oimodelGraph,ConceptNode.this,subConceptNode)==null)
                                            oimodelGraph.add(new ConceptHierarchyEdge(ConceptNode.this,subConceptNode,edgeLength));
                                        count++;
                                        if (count % 6==0)
                                            edgeLength+=ConceptHierarchyEdge.UNIT_LENGTH;
                                    }
                                }
                                m_subConceptsExpanded=LOADED;
                                oimodelGraph.notifyUpdated();
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_subConceptsExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion status of the superconcepts.
     *
     * @return                              the expansion status of superconcepts
     */
    public int getSuperConceptsExpanded() {
        return m_superConceptsExpanded;
    }
    /**
     * Expands the superconcepts in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandSuperConcepts(final OIModelGraph oimodelGraph) throws KAONException {
        m_superConceptsExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_superConcepts;
            public void backgroundJob() throws KAONException {
                m_superConcepts=getConcept().getSuperConcepts();
                oimodelGraph.getOIModel().loadObjects(m_superConcepts,OIModel.LOAD_LEXICON  | OIModel.LOAD_CONCEPT_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(ConceptNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                                double edgeLength=ConceptHierarchyEdge.UNIT_LENGTH;
                                int count=0;
                                Iterator iterator=m_superConcepts.iterator();
                                while (iterator.hasNext()) {
                                    Concept superConcept=(Concept)iterator.next();
                                    if (viewFilter.showEntity(superConcept)) {
                                        ConceptNode superConceptNode=(ConceptNode)oimodelGraph.getNodeForEntity(superConcept);
                                        if (superConceptNode==null) {
                                            superConceptNode=new ConceptNode(superConcept,ConceptNode.this);
                                            oimodelGraph.add(superConceptNode);
                                        }
                                        if (ConceptHierarchyEdge.getEdge(oimodelGraph,superConceptNode,ConceptNode.this)==null)
                                            oimodelGraph.add(new ConceptHierarchyEdge(superConceptNode,ConceptNode.this,edgeLength));
                                        count++;
                                        if (count % 6==0)
                                            edgeLength+=ConceptHierarchyEdge.UNIT_LENGTH;
                                    }
                                }
                                oimodelGraph.notifyUpdated();
                                m_superConceptsExpanded=LOADED;
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_superConceptsExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion status of the properties from.
     *
     * @return                              the expansion status of the properties from
     */
    public int getPropertiesFromExpanded() {
        return m_propertiesFromExpanded;
    }
    /**
     * Expands the properties from the concept in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandPropertiesFrom(final OIModelGraph oimodelGraph) throws KAONException {
        m_propertiesFromExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_propertiesFrom;
            public void backgroundJob() throws KAONException {
                m_propertiesFrom=getConcept().getPropertiesFromConcept();
                oimodelGraph.getOIModel().loadObjects(m_propertiesFrom,OIModel.LOAD_LEXICON | OIModel.LOAD_PROPERTY_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(ConceptNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                                Iterator iterator=m_propertiesFrom.iterator();
                                while (iterator.hasNext()) {
                                    Property property=(Property)iterator.next();
                                    if (viewFilter.showEntity(property)) {
                                        PropertyNode propertyNode=(PropertyNode)oimodelGraph.getNodeForEntity(property);
                                        if (propertyNode==null) {
                                            propertyNode=new PropertyNode(property,ConceptNode.this);
                                            oimodelGraph.add(propertyNode);
                                        }
                                        if (PropertyDomainEdge.getEdge(oimodelGraph,propertyNode,ConceptNode.this)==null)
                                            oimodelGraph.add(new PropertyDomainEdge(propertyNode,ConceptNode.this));
                                    }
                                }
                                oimodelGraph.notifyUpdated();
                                m_propertiesFromExpanded=LOADED;
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_propertiesFromExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion status of the properties to.
     *
     * @return                              the expansion status of the properties to
     */
    public int getPropertiesToExpanded() {
        return m_propertiesToExpanded;
    }
    /**
     * Expands the properties to the concept in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandPropertiesTo(final OIModelGraph oimodelGraph) throws KAONException {
        m_propertiesToExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_propertiesTo;
            public void backgroundJob() throws KAONException {
                m_propertiesTo=getConcept().getPropertiesToConcept();
                oimodelGraph.getOIModel().loadObjects(m_propertiesTo,OIModel.LOAD_LEXICON | OIModel.LOAD_PROPERTY_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(ConceptNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                                Iterator iterator=m_propertiesTo.iterator();
                                while (iterator.hasNext()) {
                                    Property property=(Property)iterator.next();
                                    if (viewFilter.showEntity(property)) {
                                        PropertyNode propertyNode=(PropertyNode)oimodelGraph.getNodeForEntity(property);
                                        if (propertyNode==null) {
                                            propertyNode=new PropertyNode(property,ConceptNode.this);
                                            oimodelGraph.add(propertyNode);
                                        }
                                        if (PropertyRangeEdge.getEdge(oimodelGraph,propertyNode,ConceptNode.this)==null)
                                            oimodelGraph.add(new PropertyRangeEdge(propertyNode,ConceptNode.this));
                                    }
                                }
                                oimodelGraph.notifyUpdated();
                                m_propertiesToExpanded=LOADED;
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_propertiesToExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion status of instance.
     *
     * @return                              the expansion status of instances
     */
    public int getInstancesExpanded() {
        return m_instancesExpanded;
    }
    /**
     * Expands the instances from the concept in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandInstances(final OIModelGraph oimodelGraph) throws KAONException {
        m_instancesExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_instances;
            public void backgroundJob() throws KAONException {
                m_instances=getConcept().getInstances();
                oimodelGraph.getOIModel().loadObjects(m_instances,OIModel.LOAD_LEXICON | OIModel.LOAD_INSTANCE_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(ConceptNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                                Iterator iterator=m_instances.iterator();
                                while (iterator.hasNext()) {
                                    Instance instance=(Instance)iterator.next();
                                    if (viewFilter.showEntity(instance)) {
                                        InstanceNode instanceNode=(InstanceNode)oimodelGraph.getNodeForEntity(instance);
                                        if (instanceNode==null) {
                                            instanceNode=new InstanceNode(instance,ConceptNode.this);
                                            oimodelGraph.add(instanceNode);
                                        }
                                        connectInstanceNode(oimodelGraph,instanceNode);
                                    }
                                }
                                oimodelGraph.notifyUpdated();
                                m_instancesExpanded=LOADED;
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_instancesExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Connects given instance node to this concept node.
     *
     * @param oimodelGraph                  the graph
     * @param instanceNode                  the instance node to connect
     * @throws KAONException                thrown if there is a problem
     */
    public void connectInstanceNode(OIModelGraph oimodelGraph,InstanceNode instanceNode) throws KAONException {
        m_fanOutManager.addNode(oimodelGraph,instanceNode);
    }
    /**
     * Disconnects given instance node from this concept node.
     *
     * @param oimodelGraph                  the graph
     * @param instanceNode                  the instance node to disconnect
     * @throws KAONException                thrown if there is a problem
     */
    public void disconnectInstanceNode(OIModelGraph oimodelGraph,InstanceNode instanceNode) throws KAONException {
        m_fanOutManager.removeNode(oimodelGraph,instanceNode);
    }
    /**
     * Resets the node's expanded states.
     */
    public void resetExpandedState() {
        super.resetExpandedState();
        m_subConceptsExpanded=NOT_LOADED;
        m_superConceptsExpanded=NOT_LOADED;
        m_propertiesFromExpanded=NOT_LOADED;
        m_propertiesToExpanded=NOT_LOADED;
        m_instancesExpanded=NOT_LOADED;
        m_fanOutManager.clear();
    }
}
