package de.fzi.wim.oimodeler.oimodelgraph.controller;

import java.util.List;
import java.util.LinkedList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.lang.reflect.Constructor;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.awt.event.ActionEvent;
import javax.swing.JPopupMenu;
import org.w3c.dom.Element;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

import de.fzi.wim.guibase.actions.*;
import de.fzi.wim.guibase.menus.*;
import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.view.*;

import de.fzi.wim.oimodeler.*;
import de.fzi.wim.oimodeler.actions.*;
import de.fzi.wim.oimodeler.oimodelgraph.graph.*;
import de.fzi.wim.oimodeler.ui.*;

/**
 * A manipulator providing the context menu for nodes.
 */
public class ContextMenuManipulator extends AbstractManipulator {
    /** The name of this manipulator. */
    public static final String NAME="ContextMenuManipulator";

    /** The OI-modeler viewable. */
    protected OIModelerViewable m_oimodelerViewable;
    /** The over which the context menu was opened. */
    protected EntityNode m_sourceNode;
    /** The edge over which the context menu was opened. */
    protected Edge m_sourceEdge;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable                     the viewable
     */
    public ContextMenuManipulator(OIModelerViewable oimodelerViewable) {
        m_oimodelerViewable=oimodelerViewable;
    }
    public String getName() {
        return NAME;
    }
    public void mousePressed(MouseEvent e) {
        doPopup(e);
    }
    public void mouseReleased(MouseEvent e) {
        doPopup(e);
    }
    protected void doPopup(MouseEvent e) {
        if (m_graphPane.isEnabled() && e.isPopupTrigger()) {
            Point point=e.getPoint();
            Node node=m_graphPane.getNodeAtPoint(point);
            OIModelEdge edge=(OIModelEdge)m_graphPane.getNearestEdge(point);
            Collection selection=Collections.EMPTY_SET;
            String popupName=null;
            if (node instanceof EntityNode) {
                selection=Collections.singleton(((EntityNode)node).getEntity());
                popupName="popup.oimodeler.graph.entityNode";
            }
            else if (node!=null)
                popupName="popup.oimodeler.graph.notEntityNode";
            else if (edge!=null)
                popupName="popup.oimodeler.graph.edge";
            else
                popupName="popup.oimodeler.graph.noObject";
            if (popupName!=null) {
                Element element=m_oimodelerViewable.getModule().getAppDriver().getConfigurationNode("popupmenu","key",popupName);
                if (element!=null) {
                    GraphMenuBuilder graphMenuBuilder=new GraphMenuBuilder(node,edge,selection);
                    JPopupMenu popupMenu=graphMenuBuilder.createPopupMenu(element);
                    popupMenu.show(m_graphPane,point.x,point.y);
                    e.consume();
                }
            }
        }
    }

    /**
     * Abstract context menu action.
     */
    protected abstract class AbstarctContextMenuAction extends AbstractSmartAction {
        public AbstarctContextMenuAction(String actionID) {
            super(actionID,m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager());
        }
        protected void applyChange(ChangeEvent changeEvent) {
            applyChanges(Collections.singletonList(changeEvent));
        }
        protected void applyChanges(List changeEvents) {
            m_oimodelerViewable.changeOIModel(changeEvents);
        }
    }

    /**
     * Abstract action for nodes.
     */
    protected abstract class AbstractNodeAction extends AbstarctContextMenuAction {
        /** The node that was clicked. */
        protected Node m_node;

        public AbstractNodeAction(String actionID,Node node) {
            super(actionID);
            m_node=node;
        }
    }

    /**
     * The action for fixing the node.
     */
    protected class PinDownNode extends AbstractNodeAction {
        public PinDownNode(Node node) {
            super("oimodeler.pinDownNode",node);
        }
        public void actionPerformed(ActionEvent e) {
            m_node.setFixed(!m_node.isFixed());
            m_graphPane.getGraph().notifyUpdated();
        }
        public void updateAction() {
            setSelected(m_node.isFixed());
        }
    }

    /**
     * The action for hiding the node.
     */
    protected class HideNode extends AbstractNodeAction {
        public HideNode(Node node) {
            super("oimodeler.hideNode",node);
        }
        public void actionPerformed(ActionEvent e) {
            try {
                ((OIModelGraph)m_graphPane.getGraph()).hideNodes(Collections.singleton(m_node));
            }
            catch (KAONException error) {
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(error);
            }
        }
    }

    /**
     * Abstract action for edges.
     */
    protected abstract class AbstractEdgeAction extends AbstarctContextMenuAction {
        /** The edge that was clicked. */
        protected OIModelEdge m_edge;

        public AbstractEdgeAction(String actionID,OIModelEdge edge) {
            super(actionID);
            m_edge=edge;
        }
    }

    /**
     * The action for deleting the edge.
     */
    protected class DeleteEdge extends AbstractEdgeAction {
        public DeleteEdge(OIModelEdge edge) {
            super(getDeleteEdgeString(edge),edge);
        }
        public void actionPerformed(ActionEvent e) {
            try {
                if (m_edge instanceof ConceptHierarchyEdge)
                    deleteEdge((ConceptHierarchyEdge)m_edge);
                else if (m_edge instanceof PropertyHierarchyEdge)
                    deleteEdge((PropertyHierarchyEdge)m_edge);
                else if (m_edge instanceof PropertyDomainEdge)
                    deleteEdge((PropertyDomainEdge)m_edge);
                else if (m_edge instanceof PropertyRangeEdge)
                    deleteEdge((PropertyRangeEdge)m_edge);
                else if (m_edge instanceof ConceptInstanceEdge)
                    deleteEdge((ConceptInstanceEdge)m_edge);
                else if (m_edge instanceof PropertyInstanceEdge)
                    deleteEdge((PropertyInstanceEdge)m_edge);
            }
            catch (KAONException error) {
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(error);
            }
        }
        protected void deleteEdge(ConceptHierarchyEdge edge) throws KAONException {
            applyChange(new RemoveSubConcept(edge.getSubConcept().getSuperSubConceptOIModel(edge.getSuperConcept()),null,edge.getSuperConcept(),edge.getSubConcept()));
        }
        protected void deleteEdge(PropertyHierarchyEdge edge) throws KAONException {
            applyChange(new RemoveSubProperty(edge.getSubProperty().getSuperSubPropertyOIModel(edge.getSuperProperty()),null,edge.getSuperProperty(),edge.getSubProperty()));
        }
        protected void deleteEdge(PropertyDomainEdge edge) throws KAONException {
            applyChange(new RemovePropertyDomain(edge.getProperty().getDomainConceptOIModel(edge.getConcept()),null,edge.getProperty(),edge.getConcept()));
        }
        protected void deleteEdge(PropertyRangeEdge edge) throws KAONException {
            applyChange(new RemovePropertyRange(edge.getProperty().getRangeConceptOIModel(edge.getConcept()),null,edge.getProperty(),edge.getConcept()));
        }
        protected void deleteEdge(ConceptInstanceEdge edge) throws KAONException {
            Concept concept;
            if (edge.getTo() instanceof ConceptNode)
                concept=((ConceptNode)edge.getTo()).getConcept();
            else
                concept=(Concept)((FanOutNode)edge.getTo()).getEntity();
            List events=new LinkedList();
            List queue=new LinkedList();
            queue.add(edge.getFrom());
            while (!queue.isEmpty()) {
                Node node=(Node)queue.remove(0);
                if (node instanceof InstanceNode) {
                    InstanceNode instanceNode=(InstanceNode)node;
                    events.add(new RemoveInstanceOf(instanceNode.getInstance().getConceptInstanceOIModel(concept),null,concept,instanceNode.getInstance()));
                }
                else if (node instanceof FanOutNode) {
                    FanOutNode fanOutNode=(FanOutNode)node;
                    queue.addAll(fanOutNode.getChildren());
                }
            }
            if (!events.isEmpty())
                applyChanges(events);
        }
        protected void deleteEdge(PropertyInstanceEdge edge) throws KAONException {
            if (edge.getTo() instanceof PropertyInstancesNode) {
                PropertyInstancesNode targetNode=(PropertyInstancesNode)edge.getTo();
                Property property=targetNode.getProperty();
                Instance sourceInstance=targetNode.getSourceInstance();
                List events=new LinkedList();
                Iterator iterator=targetNode.getTargetValues().iterator();
                while (iterator.hasNext()) {
                    Object targetValue=iterator.next();
                    PropertyInstance propertyInstance=property.getOIModel().getPropertyInstance(property,sourceInstance,targetValue);
                    events.add(new RemovePropertyInstance(propertyInstance.getSourceOIModel(),null,propertyInstance));
                }
                applyChanges(events);
            }
            else {
                PropertyInstance propertyInstance;
                if (edge.getTo() instanceof LiteralPropertyInstanceValueNode)
                    propertyInstance=((LiteralPropertyInstanceValueNode)edge.getTo()).getPropertyInstance();
                else {
                    PropertyInstancesNode sourceNode=(PropertyInstancesNode)edge.getFrom();
                    Property property=sourceNode.getProperty();
                    Instance sourceInstance=sourceNode.getSourceInstance();
                    Instance targetInstance=((InstanceNode)edge.getTo()).getInstance();
                    propertyInstance=property.getOIModel().getPropertyInstance(property,sourceInstance,targetInstance);
                }
                applyChange(new RemovePropertyInstance(propertyInstance.getSourceOIModel(),null,propertyInstance));
            }
        }
    }
    protected static String getDeleteEdgeString(OIModelEdge edge) {
        if (edge instanceof ConceptHierarchyEdge)
            return "oimodeler.deleteConceptHierarchy";
        else if (edge instanceof PropertyHierarchyEdge)
            return "oimodeler.deletePropertyHierarchy";
        else if (edge instanceof PropertyDomainEdge)
            return "oimodeler.deletePropertyDomain";
        else if (edge instanceof PropertyRangeEdge)
            return "oimodeler.deletePropertyRange";
        else if (edge instanceof ConceptInstanceEdge)
            return "oimodeler.deleteConceptInstance";
        else if (edge instanceof PropertyInstanceEdge)
            return "oimodeler.deletePropertyInstance";
        else
            return null;
    }

    /**
     * The action for hiding the edge.
     */
    protected class HideEdge extends AbstractEdgeAction {
        public HideEdge(OIModelEdge oimodelEdge) {
            super("oimodeler.hideEdge",oimodelEdge);
        }
        public void actionPerformed(ActionEvent e) {
            try {
                m_edge.hideEdge((OIModelGraph)m_graphPane.getGraph());
            }
            catch (KAONException error) {
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(error);
            }
        }
    }

    /**
     * The menu builder customized for the graph.
     */
    protected class GraphMenuBuilder extends LocalizedMenuBuilder {
        /** The node being selected. */
        protected Node m_node;
        /** The edge being selected. */
        protected OIModelEdge m_edge;
        /** The selection being selected. */
        protected Collection m_selection;

        public GraphMenuBuilder(Node node,OIModelEdge edge,Collection selection) {
            super(m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager(),m_oimodelerViewable.getModule().getAppDriver(),m_oimodelerViewable.getModule().getSmartActionMap());
            m_node=node;
            m_edge=edge;
            m_selection=selection;
        }
        protected SmartAction getAction(String actionID) {
            if ("action.oimodeler.graph.pinDownNode".equals(actionID))
                return new PinDownNode(m_node);
            else if ("action.oimodeler.graph.hideNode".equals(actionID))
                return new HideNode(m_node);
            else if ("action.oimodeler.graph.deleteEdge".equals(actionID))
                return new DeleteEdge(m_edge);
            else if ("action.oimodeler.graph.hideEdge".equals(actionID))
                return new HideEdge(m_edge);
            SmartAction existingAction=super.getAction(actionID);
            if (existingAction instanceof AbstractEditorAction) {
                try {
                    Class clazz=existingAction.getClass();
                    Constructor constructor=clazz.getConstructor(new Class[] { OIModelerModule.class });
                    AbstractEditorAction smartAction=(AbstractEditorAction)constructor.newInstance(new Object[] { (OIModelerModule)m_oimodelerViewable.getModule() });
                    smartAction.setOverride(m_oimodelerViewable,m_selection);
                    return smartAction;
                }
                catch (Exception ignored) {
                }
            }
            return null;
        }
    }
}
