package de.fzi.wim.oimodeler.lexicon;

import java.util.List;
import java.util.LinkedList;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Collections;
import java.util.Iterator;
import java.util.Comparator;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

import de.fzi.wim.guibase.localization.*;

import de.fzi.wim.oimodeler.*;
import de.fzi.wim.oimodeler.ui.*;

/**
 * The lexical table model.
 */
public class LexiconTableModel extends AbstractTableModel implements OIModelListener {
    /** THe OI-modeler viewable. */
    protected OIModelerViewable m_oimodelerViewable;
    /** The entity for which the lexicon is displayed. */
    protected Entity m_entity;
    /** The list of elements. */
    protected List m_elements;
    /** Determines the index of the column on which the model is sorted. */
    protected int m_sortedColumn;
    /** Determines whether the sort is ascending. */
    protected boolean m_sortAscending;
    /** The change visitor. */
    protected ChangeVisitor m_changeVisitor;
    /** The map of type URIs to their names. */
    protected Map m_typeURIsNames;
    /** The map of language URIs to their names. */
    protected Map m_languageURIsNames;
    /** The type URI of the new item. */
    protected String m_newItemTypeURI;
    /** The language URI of the new item. */
    protected String m_newItemLanguageURI;
    /** The value of the new item. */
    protected String m_newItemValue;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable                 the viewable
     */
    public LexiconTableModel(OIModelerViewable oimodelerViewable) {
        m_oimodelerViewable=oimodelerViewable;
        m_elements=Collections.EMPTY_LIST;
        m_sortedColumn=-1;
        m_changeVisitor=createOIModelChangeVisitor();
        m_oimodelerViewable.getOIModel().addOIModelListener(this);
        LocalizationManager localizationManager=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager();
        m_typeURIsNames=new HashMap();
        try {
            KAONConnection connection=KAONManager.getKAONConnection(KAONManager.s_parametersAPIOnRDF);
            OIModel lexicalOIModel=connection.openOIModelLogical(KAONConnection.LEXICAL_OIMODEL_URI);
            Concept lexicalEntry=lexicalOIModel.getConcept(KAONVocabularyAdaptor.INSTANCE.getLexicalEntry());
            Iterator lexicalEntries=lexicalEntry.getSubConcepts().iterator();
            while (lexicalEntries.hasNext()) {
                Concept concept=(Concept)lexicalEntries.next();
                String typeURI=concept.getURI();
                int hashPosition=typeURI.indexOf('#');
                if (hashPosition!=-1) {
                    String name=typeURI.substring(hashPosition+1);
                    m_typeURIsNames.put(typeURI,localizationManager.getPhrase("oimodeler.lexicon."+name));
                }
            }
            connection.close();
        }
        catch (KAONException cantHappen) {
        }
        m_languageURIsNames=new HashMap();
        String[][] languages=((OIModelerModule)oimodelerViewable.getModule()).getLanguages();
        for (int i=0;i<languages.length;i++)
            m_languageURIsNames.put(KAONVocabularyAdaptor.INSTANCE.getLanguageURI(languages[i][0]),languages[i][1]);
        clearTemporaryRow();
    }
    /**
     * Disposes of this model.
     */
    public void dispose() {
        if (m_oimodelerViewable!=null) {
            m_oimodelerViewable.getOIModel().removeOIModelListener(this);
            m_oimodelerViewable=null;
            m_entity=null;
            m_elements=null;
        }
    }
    /**
     * Returns the entity that has been shown.
     *
     * @return                                  the entity that has been shown
     */
    public Entity getShownEntity() {
        return m_entity;
    }
    /**
     * Sets the entity whose lexicon is displayed.
     *
     * @param entity                            the entity
     */
    public void showEntity(final Entity entity) {
        try {
            m_entity=entity;
            if (m_entity==null)
                m_elements=Collections.EMPTY_LIST;
            else if ((m_entity.getOIModel().getCapabilities() & OIModel.CAPABILITY_SUPPORTS_OPTIMIZED_LOADING)!=0) {
                m_elements=null;
                m_oimodelerViewable.getModule().getAppDriver().getThreadPool().executeTask(new Runnable() {
                    public void run() {
                        try {
                            final List elements=loadModel(entity);
                            SwingUtilities.invokeLater(new Runnable() {
                                public void run() {
                                    // if we are still supposed to load this element
                                    if (m_entity==entity) {
                                        m_elements=elements;
                                        clearTemporaryRow();
                                        if (m_sortedColumn>0)
                                            sortModel(m_sortedColumn,m_sortAscending);
                                        else
                                            fireTableDataChanged();
                                    }
                                }
                            });
                        }
                        catch (final KAONException e) {
                            SwingUtilities.invokeLater(new Runnable() {
                                public void run() {
                                    // if we are still supposed to load this element
                                    if (m_entity==entity) {
                                        clearTemporaryRow();
                                        m_elements=Collections.EMPTY_LIST;
                                        fireTableDataChanged();
                                    }
                                    m_oimodelerViewable.errorLoadingEntity(m_entity,e);
                                }
                            });
                        }
                    }
                });
            }
            else {
                m_elements=loadModel(entity);
                if (m_sortedColumn>0)
                    sortModel(m_sortedColumn,m_sortAscending);
            }
        }
        catch (KAONException e) {
            m_elements=Collections.EMPTY_LIST;
            m_oimodelerViewable.errorLoadingEntity(m_entity,e);
        }
        catch (InterruptedException e) {
            m_elements=Collections.EMPTY_LIST;
            m_oimodelerViewable.errorLoadingEntity(m_entity,e);
        }
        clearTemporaryRow();
        fireTableDataChanged();
    }
    /**
     * Clears the temporary row.
     */
    protected void clearTemporaryRow() {
        m_newItemTypeURI="";
        m_newItemLanguageURI="";
        m_newItemValue="";
    }
    /**
     * Loads the model.
     *
     * @param entity                            the entity
     * @return                                  the list of model elements
     * @throws KAONException                    thrown if there is a problem
     */
    protected List loadModel(Entity entity) throws KAONException {
        List elements=new ArrayList();
        synchronized (entity.getOIModel().getKAONConnection()) {
            Iterator lexicalEntries=entity.getLexicalEntries().iterator();
            while (lexicalEntries.hasNext()) {
                LexicalEntry lexicalEntry=(LexicalEntry)lexicalEntries.next();
                TableRow tableRow=createTableRow(lexicalEntry);
                tableRow.updateData();
                elements.add(tableRow);
            }
        }
        return elements;
    }
    /**
     * Displays an empty model.
     */
    public void showNothing() {
        showEntity(null);
    }
    /**
     * Returns the class of the column.
     *
     * @param columnIndex                   the index of the column
     * @return                              the column class
     */
    public Class getColumnClass(int columnIndex) {
        return String.class;
    }
    /**
     * Returns the number of columns.
     *
     * @return                              the number of columns
     */
    public int getColumnCount() {
        return 3;
    }
    /**
     * Returns the name of the column.
     *
     * @param columnIndex                   the name of the column
     * @return                              the column name
     */
    public String getColumnName(int columnIndex) {
        LocalizationManager localizationManager=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager();
        switch (columnIndex) {
        case 0:
            return localizationManager.getPhrase("oimodeler.lexicalEntryType");
        case 1:
            return localizationManager.getPhrase("oimodeler.lexicalEntryLanguage");
        case 2:
        default:
            return localizationManager.getPhrase("oimodeler.lexicalEntryValue");
        }
    }
    /**
     * Returns the number of rows.
     *
     * @return                              the number of rows
     */
    public int getRowCount() {
        if (m_elements==null)
            return 1;
        else
            return m_elements.size()+1;
    }
    /**
     * Returns the value at given position.
     *
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     * @return                              the value at given row and column
     */
    public Object getValueAt(int rowIndex,int columnIndex) {
        if (m_elements==null) {
            if (columnIndex==0)
                return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("oimodeler.loadingWait");
            else
                return null;
        }
        else if (rowIndex<m_elements.size()) {
            TableRow tableRow=(TableRow)m_elements.get(rowIndex);
            return tableRow.getRowValue(columnIndex);
        }
        else {
            switch (columnIndex) {
            case 0:
                {
                    String value=(String)m_typeURIsNames.get(m_newItemTypeURI);
                    if (value==null)
                        value=m_newItemTypeURI;
                    return value;
                }
            case 1:
                {
                    String value=(String)m_languageURIsNames.get(m_newItemLanguageURI);
                    if (value==null)
                        value=m_newItemLanguageURI;
                    return value;
                }
            case 2:
            default:
                return m_newItemValue;
            }
        }
    }
    /**
     * Checks if the cell is editable.
     *
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     * @return                              <code>true</code> if the cell at given row and column is editable
     */
    public boolean isCellEditable(int rowIndex,int columnIndex) {
        if (m_elements==null)
            return false;
        else if (rowIndex<m_elements.size())
            return columnIndex==2;
        else
            return true;
    }
    /**
     * Sets the value at given position.
     *
     * @param aValue                        the value
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     */
    public void setValueAt(Object aValue,int rowIndex,int columnIndex) {
        if (aValue==null)
            aValue="";
        Object oldValue=getValueAt(rowIndex,columnIndex);
        if (aValue.equals(oldValue))
            return;
        if (rowIndex<m_elements.size()) {
            if (columnIndex==2) {
                TableRow tableRow=getTableData(rowIndex);
                if (!aValue.equals(tableRow.m_value))
                    try {
                        LexicalEntry lexicalEntry=tableRow.m_lexicalEntry;
                        Property propertyValue=m_oimodelerViewable.getOIModel().getProperty(KAONVocabularyAdaptor.INSTANCE.getValue());
                        PropertyInstance propertyInstance=m_oimodelerViewable.getOIModel().getPropertyInstance(propertyValue,lexicalEntry,tableRow.m_value);
                        m_oimodelerViewable.changeOIModel(Collections.singletonList(new ChangePropertyInstanceValue(propertyInstance.getSourceOIModel(),null,propertyInstance,aValue.toString())));
                    }
                    catch (KAONException e) {
                        m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
                    }
            }
        }
        else {
            switch (columnIndex) {
            case 0:
                {
                    m_newItemTypeURI="";
                    Iterator iterator=m_typeURIsNames.keySet().iterator();
                    while (iterator.hasNext()) {
                        String key=(String)iterator.next();
                        String value=(String)m_typeURIsNames.get(key);
                        if (value.equals(aValue)) {
                            m_newItemTypeURI=key;
                            break;
                        }
                    }
                }
                break;
            case 1:
                {
                    m_newItemLanguageURI="";
                    Iterator iterator=m_languageURIsNames.keySet().iterator();
                    while (iterator.hasNext()) {
                        String key=(String)iterator.next();
                        String value=(String)m_languageURIsNames.get(key);
                        if (value.equals(aValue)) {
                            m_newItemLanguageURI=key;
                            break;
                        }
                    }
                }
                break;
            case 2:
                m_newItemValue=aValue.toString();
                fireTableCellUpdated(rowIndex,2);
                break;
            }
            if (m_newItemTypeURI.length()!=0 && m_newItemLanguageURI.length()!=0 && m_newItemValue.length()!=0)
                try {
                    List events=new LinkedList();
                    LexicalEntry lexicalEntry=m_oimodelerViewable.getActiveOIModel().getLexicalEntry(m_oimodelerViewable.getOIModel().createNewURI());
                    LexiconUtil.createLexicalEntry(lexicalEntry,m_newItemTypeURI,m_newItemValue,m_newItemLanguageURI,m_entity,events);
                    m_oimodelerViewable.changeOIModel(events);
                }
                catch (KAONException e) {
                    m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
                }
        }
    }
    /**
     * Returns the data of the table in given row.
     *
     * @param rowIndex                      the row
     * @return                              the data in the table
     */
    public TableRow getTableData(int rowIndex) {
        return (TableRow)m_elements.get(rowIndex);
    }
    /**
     * Returns the language of the new item.
     *
     * @return                              the language of the new item
     */
    public String getNewItemLanguageURI() {
        return m_newItemLanguageURI;
    }
    /**
     * Returns the index of the row with given lexical entry.
     *
     * @param lexicalEntry                  the lexical entry
     * @return                              the row with given lexical entry (or -1)
     */
    public int getLexicalEntryRow(LexicalEntry lexicalEntry) {
        for (int i=0;i<m_elements.size();i++) {
            TableRow tableRow=(TableRow)m_elements.get(i);
            if (tableRow.m_lexicalEntry.equals(lexicalEntry))
                return i;
        }
        return -1;
    }
    /**
     * Called when a bulk of change events is processed in the pool.
     *
     * @param oimodel                       OI-model that was changed
     * @param changeEvents                  list of change events that occured
     */
    public void modelChanged(OIModel oimodel,List changeEvents) {
        if (m_entity!=null && m_elements!=null)
            try {
                Iterator iterator=changeEvents.iterator();
                while (iterator.hasNext()) {
                    ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                    changeEvent.accept(m_changeVisitor);
                }
            }
            catch (KAONException e) {
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
            }
    }
    /**
     * Called when model is refreshed.
     *
     * @param oimodel                       OI-model that was refreshed
     */
    public void modelRefreshed(OIModel oimodel) {
        if (m_entity!=null)
            showEntity(m_entity);
    }
    /**
     * Called when model is being deleted.
     *
     * @param oimodel                       OI-model that is being deleted
     */
    public void modelDeleted(OIModel oimodel) {
    }
    /**
     * Sorts the table model.
     *
     * @param columnIndex                   the index of the column
     * @param ascending                     if <code>true</code> sorts the model ascending
     */
    public void sortModel(int columnIndex,boolean ascending) {
        m_sortedColumn=columnIndex;
        m_sortAscending=ascending;
        final Comparator comparator=ColumnComparator.COMPARATORS[columnIndex];
        if (ascending)
            Collections.sort(m_elements,comparator);
        else
            Collections.sort(m_elements,new Comparator() {
                public int compare(Object o1,Object o2) {
                    return -comparator.compare(o1,o2);
                }
            });
        fireTableDataChanged();
    }
    /**
     * Returns the map of languages.
     *
     * @return                              the map of languages
     */
    public Map getLanguageURIsNames() {
        return m_languageURIsNames;
    }
    /**
     * Returns the map of types.
     *
     * @return                              the map of types
     */
    public Map getTypeURIsNames() {
        return m_typeURIsNames;
    }
    /**
     * Creates the table row.
     *
     * @param lexicalEntry                  the lexical entry
     * @return                              the table row
     */
    protected TableRow createTableRow(LexicalEntry lexicalEntry) {
        return new TableRow(lexicalEntry);
    }
    /**
     * Creates the change visitor for the OIModel.
     *
     * @return                              the change visitor to use
     */
    protected ChangeVisitor createOIModelChangeVisitor() {
        return new OIModelChangeVisitor();
    }

    /**
     * The visitor for OI-model events.
     */
    protected class OIModelChangeVisitor extends NullChangeEventVisitor {
        public void visit(AddPropertyInstance event) throws KAONException {
            if (m_entity!=null) {
                PropertyInstance propertyInstance=event.getPropertyInstance();
                String uri=propertyInstance.getProperty().getURI();
                if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getReferences()) && propertyInstance.getTargetValue().equals(m_entity.getSpanningInstance())) {
                    LexicalEntry lexicalEntry=propertyInstance.getOIModel().getLexicalEntry(propertyInstance.getSourceInstance().getURI());
                    TableRow tableRow=createTableRow(lexicalEntry);
                    tableRow.updateData();
                    m_elements.add(tableRow);
                    clearTemporaryRow();
                    fireTableRowsInserted(m_elements.size(),m_elements.size());
                }
                else if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getValue())) {
                    LexicalEntry lexicalEntry=propertyInstance.getOIModel().getLexicalEntry(propertyInstance.getSourceInstance().getURI());
                    int row=getLexicalEntryRow(lexicalEntry);
                    if (row!=-1) {
                        getTableData(row).updateData();
                        fireTableCellUpdated(row,2);
                    }
                }
                else if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getInLanguage())) {
                    LexicalEntry lexicalEntry=propertyInstance.getOIModel().getLexicalEntry(propertyInstance.getSourceInstance().getURI());
                    int row=getLexicalEntryRow(lexicalEntry);
                    if (row!=-1) {
                        getTableData(row).updateData();
                        fireTableCellUpdated(row,1);
                    }
                }
            }
        }
        public void visit(RemovePropertyInstance event) throws KAONException {
            if (m_entity!=null) {
                PropertyInstance propertyInstance=event.getPropertyInstance();
                String uri=propertyInstance.getProperty().getURI();
                if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getReferences()) && propertyInstance.getTargetValue().equals(m_entity.getSpanningInstance())) {
                    LexicalEntry lexicalEntry=propertyInstance.getOIModel().getLexicalEntry(propertyInstance.getSourceInstance().getURI());
                    int row=getLexicalEntryRow(lexicalEntry);
                    if (row!=-1) {
                        m_elements.remove(row);
                        fireTableRowsDeleted(row,row);
                    }
                }
                else if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getValue())) {
                    LexicalEntry lexicalEntry=propertyInstance.getOIModel().getLexicalEntry(propertyInstance.getSourceInstance().getURI());
                    int row=getLexicalEntryRow(lexicalEntry);
                    if (row!=-1) {
                        getTableData(row).updateData();
                        fireTableCellUpdated(row,2);
                    }
                }
                else if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getInLanguage())) {
                    LexicalEntry lexicalEntry=propertyInstance.getOIModel().getLexicalEntry(propertyInstance.getSourceInstance().getURI());
                    int row=getLexicalEntryRow(lexicalEntry);
                    if (row!=-1) {
                        getTableData(row).updateData();
                        fireTableCellUpdated(row,1);
                    }
                }
            }
        }
        public void visit(ChangePropertyInstanceValue event) throws KAONException {
            if (m_entity!=null) {
                PropertyInstance propertyInstance=event.getPropertyInstance();
                String uri=propertyInstance.getProperty().getURI();
                if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getReferences()) && propertyInstance.getTargetValue().equals(m_entity.getSpanningInstance()))
                    showEntity(m_entity);
                else if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getValue())) {
                    LexicalEntry lexicalEntry=propertyInstance.getOIModel().getLexicalEntry(propertyInstance.getSourceInstance().getURI());
                    int row=getLexicalEntryRow(lexicalEntry);
                    if (row!=-1) {
                        getTableData(row).updateData();
                        fireTableCellUpdated(row,2);
                    }
                }
                else if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getInLanguage())) {
                    LexicalEntry lexicalEntry=propertyInstance.getOIModel().getLexicalEntry(propertyInstance.getSourceInstance().getURI());
                    int row=getLexicalEntryRow(lexicalEntry);
                    if (row!=-1) {
                        getTableData(row).updateData();
                        fireTableCellUpdated(row,1);
                    }
                }
            }
        }
        public void visit(SetPropertyInstanceValue event) throws KAONException {
            if (m_entity!=null) {
                PropertyInstance propertyInstance=event.getPropertyInstance();
                String uri=propertyInstance.getProperty().getURI();
                if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getReferences()) && propertyInstance.getTargetValue().equals(m_entity.getSpanningInstance()))
                    showEntity(m_entity);
                else if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getValue()))
                    showEntity(m_entity);
                else if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getInLanguage()))
                    showEntity(m_entity);
            }
        }
    }

    /**
     * The element in the model.
     */
    public class TableRow {
        /** The lexical entry. */
        public LexicalEntry m_lexicalEntry;
        /** The type URI of the lexical entry. */
        public String m_typeURI;
        /** The language URI of the lexical entry. */
        public String m_languageURI;
        /** The value of the lexical entry. */
        public String m_value;

        /**
         * Creates an instance of this class.
         *
         * @param lexicalEntry                  the lexical entry
         */
        public TableRow(LexicalEntry lexicalEntry) {
            m_lexicalEntry=lexicalEntry;
        }
        /**
         * Updates this node.
         *
         * @throws KAONException                thrown if there is an error
         */
        public void updateData() throws KAONException {
            m_typeURI=m_lexicalEntry.getTypeURI();
            m_languageURI=m_lexicalEntry.getInLanguage();
            m_value=m_lexicalEntry.getValue();
        }
        /**
         * Returns the value in given row.
         *
         * @param columnIndex                   the column whose value is requested
         * @return                              the value in given column
         */
        public Object getRowValue(int columnIndex) {
            switch (columnIndex) {
            case 0:
                {
                    String value=(String)m_typeURIsNames.get(m_typeURI);
                    if (value==null)
                        value=m_typeURI;
                    return value;
                }
            case 1:
                {
                    String value=(String)m_languageURIsNames.get(m_languageURI);
                    if (value==null)
                        value=m_languageURI;
                    return value;
                }
            case 2:
            default:
                return m_value;
            }
        }
    }

    /**
     * The comparator for the table model.
     */
    protected static class ColumnComparator implements Comparator {
        public static final Comparator[] COMPARATORS=new Comparator[3];
        static {
            for (int i=0;i<COMPARATORS.length;i++)
                COMPARATORS[i]=new ColumnComparator(i);
        }

        protected int m_column;

        public ColumnComparator(int column) {
            m_column=column;
        }
        public int compare(Object o1,Object o2) {
            Object value1=((TableRow)o1).getRowValue(m_column);
            Object value2=((TableRow)o2).getRowValue(m_column);
            if (value1 instanceof String)
                return ((String)value1).compareToIgnoreCase((String)value2);
            else if (value1 instanceof Boolean) {
                boolean b1=((Boolean)value1).booleanValue();
                boolean b2=((Boolean)value2).booleanValue();
                if (b1 && !b2)
                    return -1;
                else if (!b1 && b2)
                    return 1;
            }
            return 0;
        }
    }
}
