package de.fzi.wim.oimodeler.inspector;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Collection;
import java.util.Collections;
import java.io.IOException;
import java.awt.Color;
import java.awt.Insets;
import java.awt.Dimension;
import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.ComponentOrientation;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusAdapter;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.datatransfer.UnsupportedFlavorException;
import javax.swing.Icon;
import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JLabel;
import javax.swing.JComponent;
import javax.swing.JCheckBox;
import javax.swing.JTextField;
import javax.swing.text.PlainDocument;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

import de.fzi.wim.guibase.dnd.*;
import de.fzi.wim.guibase.menus.*;
import de.fzi.wim.guibase.localization.*;
import de.fzi.wim.guibase.util.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.entityhierarchy.*;
import de.fzi.wim.oimodeler.propertyconcepts.*;
import de.fzi.wim.oimodeler.selection.*;
import de.fzi.wim.oimodeler.lexicon.*;

/**
 * The inspector for properties.
 */
public class PropertyInspector extends AbstractInspectorPanel implements OIModelListener {
    /** The currently shown property. */
    protected Property m_property;
    /** The label showing the property title. */
    protected JLabel m_propertyTitle;
    /** The check box for specifying that the property is an attribute. */
    protected JCheckBox m_isAttribute;
    /** The check box for specifying that the property is symmetric. */
    protected JCheckBox m_isSymmetric;
    /** The check box for specifying that the property is transitive. */
    protected JCheckBox m_isTransitive;
    /** The text field for specifying the inverse property. */
    protected JTextField m_inverseProperty;
    /** The tree of subproperties. */
    protected OIModelTree m_subProperties;
    /** The tree of superproperties. */
    protected OIModelTree m_superProperties;
    /** The table of domain concepts. */
    protected PropertyConceptsTable m_domainConcepts;
    /** The table of range concepts. */
    protected PropertyConceptsTable m_rangeConcepts;
    /** The table for lexicon. */
    protected LexiconTable m_lexiconTable;
    /** Change visitor. */
    protected ChangeVisitor m_changeVisitor;
    /** Set to <code>true</code> if the controls are currently being updated. */
    protected boolean m_controlsUpdating;
    /** property icon */
    protected Icon m_propertyIcon;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable                 the OI-modeler viewable
     */
    public PropertyInspector(OIModelerViewable oimodelerViewable) {
        super(oimodelerViewable);
        m_oimodelerViewable.getOIModel().addOIModelListener(this);
        m_changeVisitor=new OIModelChangeVisitor();
        m_propertyIcon=oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.property");
        m_propertyTitle=new JLabel();
        m_propertyTitle.setOpaque(true);
        m_propertyTitle.setBackground(Color.gray);
        m_propertyTitle.setForeground(Color.white);
        m_propertyTitle.setIcon(m_propertyIcon);
        m_propertyTitle.setIconTextGap(10);
        m_propertyTitle.setBorder(BorderFactory.createEmptyBorder(3,3,3,3));
        LocalizationManager localizationManager=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager();
        m_isAttribute=new JCheckBox();
        m_isAttribute.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                try {
                    if (!m_controlsUpdating)
                        m_oimodelerViewable.changeOIModel(Collections.singletonList(new SetPropertyIsAttribute(m_property.getOIModel(),null,m_property,m_isAttribute.isSelected())));
                }
                catch (KAONException error) {
                    m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(error);
                }
            }
        });
        Mnemonics.setText(m_isAttribute,localizationManager.getPhrase("oimodeler.isAttribute"));
        m_isSymmetric=new JCheckBox();
        m_isSymmetric.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                try {
                    if (!m_controlsUpdating)
                        m_oimodelerViewable.changeOIModel(Collections.singletonList(new SetPropertySymmetric(m_property.getOIModel(),null,m_property,m_isSymmetric.isSelected())));
                }
                catch (KAONException error) {
                    m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(error);
                }
            }
        });
        Mnemonics.setText(m_isSymmetric,localizationManager.getPhrase("oimodeler.isSymmetric"));
        m_isTransitive=new JCheckBox();
        m_isTransitive.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                try {
                    if (!m_controlsUpdating)
                        m_oimodelerViewable.changeOIModel(Collections.singletonList(new SetPropertyTransitive(m_property.getOIModel(),null,m_property,m_isTransitive.isSelected())));
                }
                catch (KAONException error) {
                    m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(error);
                }
            }
        });
        Mnemonics.setText(m_isTransitive,localizationManager.getPhrase("oimodeler.isTransitive"));
        m_inverseProperty=new JTextField();
        m_inverseProperty.setPreferredSize(new Dimension(150,m_inverseProperty.getPreferredSize().height));
        m_inverseProperty.addFocusListener(new FocusAdapter() {
            public void focusGained(FocusEvent e) {
                m_inverseProperty.selectAll();
            }
        });
        m_inverseProperty.setDocument(new PlainDocument() {
            public void insertString(int offset,String string,AttributeSet a) throws BadLocationException {
                if (m_controlsUpdating)
                    super.insertString(offset,string,a);
            }
            public void remove(int offset,int length) throws BadLocationException {
                try {
                    if (m_controlsUpdating)
                        super.remove(offset,length);
                    else if (m_property!=null && m_property.getInverseProperty()!=null)
                        m_oimodelerViewable.changeOIModel(Collections.singletonList(new SetNoInverseProperties(m_property.getInversePropertyOIModel(),null,m_property,m_property.getInverseProperty())));
                }
                catch (KAONException e) {
                    m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
                }
            }
        });
        new InversePropertyDropTarget(m_inverseProperty);
        try {
            m_superProperties=(OIModelTree)createElement(0);
            m_subProperties=(OIModelTree)createElement(1);
            m_lexiconTable=(LexiconTable)createElement(2);
            m_domainConcepts=(PropertyConceptsTable)createElement(3);
            m_rangeConcepts=(PropertyConceptsTable)createElement(4);
        }
        catch (KAONException cantHappen) {
        }
        JPanel inversePropertyPane=new JPanel(new BorderLayout(10,0));
        inversePropertyPane.add(localizationManager.getLabel("oimodeler.inverseProperty"),BorderLayout.WEST);
        inversePropertyPane.add(m_inverseProperty,BorderLayout.EAST);
        JPanel header=new JPanel(new GridBagLayout());
        GridBagConstraints gbc=new GridBagConstraints(0,0,4,1,1.0,0,GridBagConstraints.CENTER,GridBagConstraints.HORIZONTAL,new Insets(0,0,0,0),0,0);
        header.add(m_propertyTitle,gbc);
        gbc.insets.top=5;
        gbc.insets.bottom=5;
        gbc=new GridBagConstraints(0,1,1,1,0.33,0,GridBagConstraints.WEST,GridBagConstraints.NONE,new Insets(0,10,5,10),0,0);
        header.add(m_isAttribute,gbc);
        gbc.anchor=GridBagConstraints.CENTER;
        gbc.gridx=1;
        header.add(m_isSymmetric,gbc);
        gbc.gridx=2;
        header.add(m_isTransitive,gbc);
        gbc.gridx=3;
        gbc.anchor=GridBagConstraints.EAST;
        header.add(inversePropertyPane,gbc);
        m_elements=new JComponent[] {
            createTitledScrollPane(m_superProperties,"oimodeler.superProperties"),
            createTitledScrollPane(m_subProperties,"oimodeler.subProperties"),
            createTitledScrollPane(m_lexiconTable,"oimodeler.lexicon"),
            createTitledScrollPane(m_domainConcepts,"oimodeler.domainConcepts"),
            createTitledScrollPane(m_rangeConcepts,"oimodeler.rangeConcepts"),
        };
        add(m_elementsPanel,BorderLayout.CENTER);
        add(header,BorderLayout.NORTH);
        layoutVisibleElements();
    }
    /**
     * Creates an <code>OIModelTree</code> for super properties. As well as the associated
     * <code>JTreePopupMenuManipulator</code> and <code>DoubleClickSelectionFocuser</code>.
     *
     * @return          the created <code>OIModelTree</code>
     */
    protected OIModelTree createSuperProperties() {
        OIModelTree superProperties=new OIModelTree(m_oimodelerViewable,new SuperPropertyHierarchyNodeLoader(m_oimodelerViewable));
        superProperties.setComponentOrientation(ComponentOrientation.RIGHT_TO_LEFT);
        superProperties.setNodeIcon(m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.property"));
        new JTreePopupMenuManipulator(superProperties,m_oimodelerViewable.getModule().getAppDriver().createPopupMenu("popup.oimodeler.selection"));
        new DoubleClickSelectionFocuser(m_oimodelerViewable,superProperties);
        return superProperties;
    }
    /**
     * Creates an <code>OIModelTree</code> for sub properties. As well as the associated
     * <code>JTreePopupMenuManipulator</code> and <code>DoubleClickSelectionFocuser</code>.
     *
     * @return          the created <code>OIModelTree</code>
     */
    protected OIModelTree createSubProperties() {
        OIModelTree subProperties=new OIModelTree(m_oimodelerViewable,new SubPropertyHierarchyNodeLoader(m_oimodelerViewable));
        subProperties.setNodeIcon(m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.property"));
        new JTreePopupMenuManipulator(subProperties,m_oimodelerViewable.getModule().getAppDriver().createPopupMenu("popup.oimodeler.selection"));
        new DoubleClickSelectionFocuser(m_oimodelerViewable,subProperties);
        return subProperties;
    }
    /**
     * Creates a <code>PropertyConceptsTable</code> for either "domain" or "range"-concepts. As well as the associated
     * <code>JTreePopupMenuManipulator</code> and <code>DoubleClickSelectionFocuser</code>.
     *
     * @param domain    if <code>true</code> a <code>PropertyConceptsTable</code> for "domain"-concepts will be created, a table for "range"-concepts otherwise
     * @return          the created <code>PropertyConceptsTable</code>
     */
    protected PropertyConceptsTable createPropertyConceptsTable(boolean domain) {
        PropertyConceptsTable concepts=new PropertyConceptsTable(m_oimodelerViewable,domain);
        new JTablePopupMenuManipulator(concepts,m_oimodelerViewable.getModule().getAppDriver().createPopupMenu("popup.oimodeler.selection"));
        new DoubleClickSelectionFocuser(m_oimodelerViewable,concepts);
        return concepts;
    }
    /**
     * Disposes of the inspector.
     */
    public void dispose() {
        if (m_superProperties!=null) {
            m_superProperties.dispose();
            m_superProperties=null;
        }
        if (m_subProperties!=null) {
            m_subProperties.dispose();
            m_subProperties=null;
        }
        if (m_domainConcepts!=null) {
            m_domainConcepts.dispose();
            m_domainConcepts=null;
        }
        if (m_rangeConcepts!=null) {
            m_rangeConcepts.dispose();
            m_rangeConcepts=null;
        }
        if (m_lexiconTable!=null) {
            m_lexiconTable.dispose();
            m_lexiconTable=null;
        }
        if (m_oimodelerViewable!=null) {
            m_oimodelerViewable.getOIModel().removeOIModelListener(this);
            m_oimodelerViewable=null;
        }
    }
    /**
     * Creates the element for given index.
     *
     * @param elementIndex          the index of the element
     * @return                      the new element with given index
     * @throws KAONException        thrown if there is an error
     */
    public EntityInfoPage createElement(int elementIndex) throws KAONException {
        switch (elementIndex) {
        case 0:
            return showEntity(createSuperProperties(),m_property);
        case 1:
            return showEntity(createSubProperties(),m_property);
        case 2:
            return showEntity(new LexiconTable(m_oimodelerViewable),m_property);
        case 3:
            return showEntity(createPropertyConceptsTable(true),m_property);
        case 4:
            return showEntity(createPropertyConceptsTable(false),m_property);
        default:
            throw new IllegalArgumentException("Illegal element index "+elementIndex);
        }
    }
    /**
     * Shows the given property.
     *
     * @param property                          the property
     * @throws KAONException                    thrown if there is a problem with accessing the OI-model
     */
    public void showProperty(Property property) throws KAONException {
        m_property=property;
        m_propertyTitle.setText(property.getURI());
        m_superProperties.showEntity(property);
        m_subProperties.showEntity(property);
        m_domainConcepts.showEntity(property);
        m_rangeConcepts.showEntity(property);
        m_lexiconTable.showEntity(property);
        updateControls();
    }
    /**
     * Makes sure nothing is shown.
     */
    public void showNothing() {
        m_property=null;
        m_superProperties.showNothing();
        m_subProperties.showNothing();
        m_domainConcepts.showNothing();
        m_rangeConcepts.showNothing();
        m_lexiconTable.showNothing();
        updateControls();
    }
    /**
     * Sets the language URI.
     *
     * @param languageURI                       the language URI
     * @throws KAONException                    thrown if there is an error
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        m_subProperties.setLanguageURI(languageURI);
        m_superProperties.setLanguageURI(languageURI);
        m_domainConcepts.setLanguageURI(languageURI);
        m_rangeConcepts.setLanguageURI(languageURI);
        updateControls();
    }
    /**
     * Notifies this component that a request is being processed.
     *
     * @param beingProcessed                    <code>true</code> if OI-model is being processed
     */
    public void setBeingProcessed(boolean beingProcessed) {
        m_subProperties.setEnabled(beingProcessed);
        m_superProperties.setEnabled(beingProcessed);
        m_domainConcepts.setEnabled(beingProcessed);
        m_rangeConcepts.setEnabled(beingProcessed);
        m_lexiconTable.setEnabled(beingProcessed);
    }
    /**
     * Called when a bulk of change events is processed in the pool.
     *
     * @param oimodel                   OI-model that was changed
     * @param changeEvents              list of change events that occured
     */
    public void modelChanged(OIModel oimodel,List changeEvents) {
        m_controlsUpdating=true;
        try {
            Iterator iterator=changeEvents.iterator();
            while (iterator.hasNext()) {
                ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                changeEvent.accept(m_changeVisitor);
            }
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
        m_controlsUpdating=false;
    }
    /**
     * Called when model is refreshed.
     *
     * @param oimodel                   OI-model that was refreshed
     */
    public void modelRefreshed(OIModel oimodel) {
        updateControls();
    }
    /**
     * Called when model is being deleted.
     *
     * @param oimodel                   OI-model that is being deleted
     */
    public void modelDeleted(OIModel oimodel) {
    }
    /**
     * Reloads the values of all the controls.
     */
    protected void updateControls() {
        m_controlsUpdating=true;
        try {
            m_isAttribute.setSelected(m_property==null ? false : m_property.isAttribute());
            m_isSymmetric.setSelected(m_property==null ? false : m_property.isSymmetric());
            m_isTransitive.setSelected(m_property==null ? false : m_property.isTransitive());
            Property inverseProperty=m_property==null ? null : m_property.getInverseProperty();
            if (inverseProperty==null)
                m_inverseProperty.setText(null);
            else {
                String label=inverseProperty.getLabel(m_oimodelerViewable.getLanguageURI());
                if (label==null)
                    label=inverseProperty.getURI();
                m_inverseProperty.setText(label);
            }
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
        m_controlsUpdating=false;
    }

    /**
     * The visitor for OI-model events.
     */
    protected class OIModelChangeVisitor extends NullChangeEventVisitor {
        public void visit(RemoveEntity event) {
            if (event.getEntity().equals(m_property))
                showNothing();
        }
        public void visit(SetPropertyIsAttribute event) throws KAONException {
            if (m_property!=null && m_property.equals(event.getProperty()))
                m_isAttribute.setSelected(m_property.isAttribute());
        }
        public void visit(SetPropertySymmetric event) throws KAONException {
            if (m_property!=null && m_property.equals(event.getProperty()))
                m_isSymmetric.setSelected(m_property.isSymmetric());
        }
        public void visit(SetPropertyTransitive event) throws KAONException {
            if (m_property!=null && m_property.equals(event.getProperty()))
                m_isTransitive.setSelected(m_property.isTransitive());
        }
        public void visit(SetInverseProperties event) {
            updateControls();
        }
        public void visit(SetNoInverseProperties event) {
            updateControls();
        }
    }

    /**
     * The drop target for the inverse property field.
     */
    protected class InversePropertyDropTarget implements DropTargetListener {
        /** Drop target for the field. */
        protected DropTarget m_dropTarget;
        /** The text field. */
        protected JTextField m_textField;
        /** The background of the text field. */
        protected Color m_background;

        /**
         * Attachs this manputlator to a text field.
         *
         * @param textField                         the text field
         */
        public InversePropertyDropTarget(JTextField textField) {
            m_dropTarget=new SmartDropTarget(textField,this);
            m_textField=textField;
            m_background=m_textField.getBackground();
        }
        /**
         * Called when a drag operation has encountered the DropTarget.
         *
         * @param dtde                              the event
         */
        public void dragEnter(DropTargetDragEvent dtde) {
            processDragEvent(dtde);
        }
        /**
         * The drag operation has departed the DropTarget without dropping.
         *
         * @param dte                               the event
         */
        public void dragExit(DropTargetEvent dte) {
            m_textField.setBackground(m_background);
        }
        /**
         * Called when a drag operation is ongoing on the DropTarget.
         *
         * @param dtde                              the event
         */
        public void dragOver(DropTargetDragEvent dtde) {
            processDragEvent(dtde);
        }
        /**
         * The drag operation has terminated with a drop on this DropTarget.
         *
         * @param dtde                              the event
         */
        public void drop(DropTargetDropEvent dtde) {
            m_textField.setBackground(m_background);
            if (processDrop(dtde))
                dtde.acceptDrop(dtde.getSourceActions());
            else
                dtde.rejectDrop();
        }
        /**
         * Called if the user has modified the current drop gesture.
         *
         * @param dtde                              the event
         */
        public void dropActionChanged(DropTargetDragEvent dtde) {
        }
        /**
         * Processes a drag event.
         *
         * @param dtde                              the event
         */
        protected void processDragEvent(DropTargetDragEvent dtde) {
            if (!acceptsDrag(dtde)) {
                m_textField.setBackground(m_background);
                dtde.rejectDrag();
            }
            else {
                m_textField.setBackground(Color.lightGray);
                dtde.acceptDrag(dtde.getSourceActions());
            }
        }
        /**
         * Determines whether operation can be accepted.
         *
         * @param dtde                              the event
         * @return                                  <code>true</code> if the drag will be accepted
         */
        protected boolean acceptsDrag(DropTargetDragEvent dtde) {
            return m_property!=null;
        }
        /**
         * Processes the drop event.
         *
         * @param dtde                              the drop event
         * @return                                  <code>true</code> if drop was accepted
         */
        protected boolean processDrop(DropTargetDropEvent dtde) {
            if (m_property==null)
                return false;
            Collection collection;
            try {
                collection=(Collection)dtde.getTransferable().getTransferData(CollectionSelection.s_objectCollectionFlavor);
            }
            catch (IOException ignored) {
                return false;
            }
            catch (UnsupportedFlavorException ignored) {
                return false;
            }
            if (collection.size()!=1)
                return false;
            Object object=collection.iterator().next();
            if (!(object instanceof Property))
                return false;
            Property inverseProperty=(Property)object;
            try {
                List events=new LinkedList();
                if (m_property.getInverseProperty()!=null)
                    events.add(new SetNoInverseProperties(m_property.getInversePropertyOIModel(),null,m_property,m_property.getInverseProperty()));
                events.add(new SetInverseProperties(m_oimodelerViewable.getActiveOIModel(),null,m_property,inverseProperty));
                m_oimodelerViewable.changeOIModel(events);
                return true;
            }
            catch (KAONException e) {
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
                return false;
            }
        }
    }
}
