package de.fzi.wim.oimodeler.inspector;

import java.util.List;
import java.util.Iterator;
import java.awt.Color;
import java.awt.Component;
import java.awt.BorderLayout;
import javax.swing.Icon;
import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JComponent;
import javax.swing.JTree;
import javax.swing.tree.DefaultTreeCellRenderer;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

import de.fzi.wim.guibase.menus.*;
import de.fzi.wim.guibase.lazytreemodel.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.entityhierarchy.*;
import de.fzi.wim.oimodeler.selection.*;
import de.fzi.wim.oimodeler.lexicon.*;

/**
 * The inspector for instances.
 */
public class InstanceInspector extends AbstractInspectorPanel implements OIModelListener {
    /** The currently shown instance. */
    protected Instance m_instance;
    /** The label showing the instance title. */
    protected JLabel m_instanceTitle;
    /** The tree of parent concepts. */
    protected OIModelTree m_parentConcepts;
    /** The tree of transitive properties. */
    protected OIModelTree m_transitiveProperties;
    /** The tree table of property instances. */
    protected ConceptInstancesTreeTable m_propertyInstances;
    /** The table for lexicon. */
    protected LexiconTable m_lexiconTable;
    /** Change visitor. */
    protected ChangeVisitor m_changeVisitor;
    /** instance icon */
    protected Icon m_instanceIcon;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable                 the OI-modeler viewable
     */
    public InstanceInspector(OIModelerViewable oimodelerViewable) {
        super(oimodelerViewable);
        m_changeVisitor=new OIModelChangeVisitor();
        m_oimodelerViewable.getOIModel().addOIModelListener(this);
        m_instanceIcon=oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.instance");
        m_instanceTitle=new JLabel();
        m_instanceTitle.setOpaque(true);
        m_instanceTitle.setBackground(Color.gray);
        m_instanceTitle.setForeground(Color.white);
        m_instanceTitle.setIcon(m_instanceIcon);
        m_instanceTitle.setIconTextGap(10);
        m_instanceTitle.setBorder(BorderFactory.createEmptyBorder(3,3,3,3));
        try {
            m_parentConcepts=(OIModelTree)createElement(0);
            m_transitiveProperties=(OIModelTree)createElement(1);
            m_propertyInstances=(ConceptInstancesTreeTable)createElement(2);
            m_lexiconTable=(LexiconTable)createElement(3);
        }
        catch (KAONException cantHappen) {
        }
        m_elements=new JComponent[] {
            createTitledScrollPane(m_parentConcepts,"oimodeler.parentConcepts"),
            createTitledScrollPane(m_transitiveProperties,"oimodeler.transitiveProperties"),
            createTitledScrollPane(m_propertyInstances,"oimodeler.propertyInstances"),
            createTitledScrollPane(m_lexiconTable,"oimodeler.lexicon"),
        };
        add(m_elementsPanel,BorderLayout.CENTER);
        add(m_instanceTitle,BorderLayout.NORTH);
        layoutVisibleElements();
    }
    /**
     * Creates an <code>OIModelTree</code> for parent concepts. As well as the associated
     * <code>JTreePopupMenuManipulator</code> and <code>DoubleClickSelectionFocuser</code>.
     *
     * @return          the created <code>OIModelTree</code>
     */
    protected OIModelTree createParentConcepts() {
        OIModelTree parentConcepts=new OIModelTree(m_oimodelerViewable,new ParentConceptHierarchyNodeLoader(m_oimodelerViewable));
        parentConcepts.setCellRenderer(new ParentConceptsRenderer());
        new JTreePopupMenuManipulator(parentConcepts,m_oimodelerViewable.getModule().getAppDriver().createPopupMenu("popup.oimodeler.selection"));
        new DoubleClickSelectionFocuser(m_oimodelerViewable,parentConcepts);
        return parentConcepts;
    }
    /**
     * Creates an <code>OIModelTree</code> for transitive properties. As well as the associated
     * <code>JTreePopupMenuManipulator</code> and <code>DoubleClickSelectionFocuser</code>.
     *
     * @return          the created <code>OIModelTree</code>
     */
    protected OIModelTree createTransitiveProperties() {
        OIModelTree transitiveProperties=new OIModelTree(m_oimodelerViewable,new TransitivePropertiesHierarchyNodeLoader(m_oimodelerViewable));
        transitiveProperties.setCellRenderer(new TransitivePropertiesRenderer());
        new JTreePopupMenuManipulator(transitiveProperties,m_oimodelerViewable.getModule().getAppDriver().createPopupMenu("popup.oimodeler.selection"));
        new DoubleClickSelectionFocuser(m_oimodelerViewable,transitiveProperties);
        return transitiveProperties;
    }
    /**
     * Disposes of the inspector.
     */
    public void dispose() {
        if (m_parentConcepts!=null) {
            m_parentConcepts.dispose();
            m_parentConcepts=null;
        }
        if (m_transitiveProperties!=null) {
            m_transitiveProperties.dispose();
            m_transitiveProperties=null;
        }
        if (m_propertyInstances!=null) {
            m_propertyInstances.dispose();
            m_propertyInstances=null;
        }
        if (m_lexiconTable!=null) {
            m_lexiconTable.dispose();
            m_lexiconTable=null;
        }
        if (m_oimodelerViewable!=null) {
            m_oimodelerViewable.getOIModel().removeOIModelListener(this);
            m_oimodelerViewable=null;
        }
    }
    /**
     * Creates the element for given index.
     *
     * @param elementIndex          the index of the element
     * @return                      the new element with given index
     * @throws KAONException        thrown if there is an error
     */
    public EntityInfoPage createElement(int elementIndex) throws KAONException {
        switch (elementIndex) {
        case 0:
            return showEntity(createParentConcepts(),m_instance);
        case 1:
            return showEntity(createTransitiveProperties(),m_instance);
        case 2:
            {
                ConceptInstancesTreeTable element=new ConceptInstancesTreeTable(m_oimodelerViewable);
                element.setRootVisible(true);
                return showEntity(element,m_instance);
            }
        case 3:
            return showEntity(new LexiconTable(m_oimodelerViewable),m_instance);
        default:
            throw new IllegalArgumentException("Illegal element index "+elementIndex);
        }
    }
    /**
     * Shows the given instance.
     *
     * @param instance                          the instance
     * @throws KAONException                    thrown if there is a problem with accessing the OI-model
     */
    public void showInstance(Instance instance) throws KAONException {
        m_instance=instance;
        m_instanceTitle.setText(instance.getURI());
        m_parentConcepts.showEntity(instance);
        m_transitiveProperties.showEntity(instance);
        m_propertyInstances.showEntity(instance);
        m_lexiconTable.showEntity(instance);
    }
    /**
     * Makes sure nothing is shown.
     */
    public void showNothing() {
        m_instance=null;
        m_parentConcepts.showNothing();
        m_transitiveProperties.showNothing();
        m_propertyInstances.showNothing();
        m_lexiconTable.showNothing();
    }
    /**
     * Sets the language URI.
     *
     * @param languageURI                       the language URI
     * @throws KAONException                    thrown if there is an error
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        m_parentConcepts.setLanguageURI(languageURI);
        m_transitiveProperties.setLanguageURI(languageURI);
        m_propertyInstances.setLanguageURI(languageURI);
    }
    /**
     * Notifies this component that a request is being processed.
     *
     * @param beingProcessed                    <code>true</code> if OI-model is being processed
     */
    public void setBeingProcessed(boolean beingProcessed) {
        m_parentConcepts.setEnabled(beingProcessed);
        m_transitiveProperties.setEnabled(beingProcessed);
        m_propertyInstances.setEnabled(beingProcessed);
        m_lexiconTable.setEnabled(beingProcessed);
    }
    /**
     * Called when a bulk of change events is processed in the pool.
     *
     * @param oimodel                   OI-model that was changed
     * @param changeEvents              list of change events that occured
     */
    public void modelChanged(OIModel oimodel,List changeEvents) {
        try {
            Iterator iterator=changeEvents.iterator();
            while (iterator.hasNext()) {
                ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                changeEvent.accept(m_changeVisitor);
            }
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
    }
    /**
     * Called when model is refreshed.
     *
     * @param oimodel                   OI-model that was refreshed
     */
    public void modelRefreshed(OIModel oimodel) {
    }
    /**
     * Called when model is being deleted.
     *
     * @param oimodel                   OI-model that is being deleted
     */
    public void modelDeleted(OIModel oimodel) {
    }

    /**
     * The visitor for OI-model events.
     */
    protected class OIModelChangeVisitor extends NullChangeEventVisitor {
        public void visit(RemoveEntity event) {
            if (event.getEntity().equals(m_instance))
                showNothing();
        }
    }

    /**
     * The cell renderer for the parent concepts.
     */
    protected class ParentConceptsRenderer extends DefaultTreeCellRenderer {
        protected Icon m_concept;
        protected Icon m_instance;

        public ParentConceptsRenderer() {
            m_concept=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.concept");
            m_instance=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.instance");
        }
        public Component getTreeCellRendererComponent(JTree tree,Object value,boolean sel,boolean expanded,boolean leaf,int row,boolean hasFocus) {
            super.getTreeCellRendererComponent(tree,value,sel,expanded,leaf,row,hasFocus);
            Icon icon;
            if (value instanceof DefaultNoChildrenNode)
                icon=null;
            else {
                Entity entity=((EntityHierarchyNode)value).getEntity();
                icon=(entity instanceof Instance) ? m_instance : m_concept;
            }
            if (tree.isEnabled())
                setIcon(icon);
            else
                setDisabledIcon(icon);
            return this;
        }
    }

    /**
     * The cell renderer for the transitive properties.
     */
    protected class TransitivePropertiesRenderer extends DefaultTreeCellRenderer {
        protected Icon m_property;
        protected Icon m_instance;

        public TransitivePropertiesRenderer() {
            m_property=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.property");
            m_instance=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.instance");
        }
        public Component getTreeCellRendererComponent(JTree tree,Object value,boolean sel,boolean expanded,boolean leaf,int row,boolean hasFocus) {
            super.getTreeCellRendererComponent(tree,value,sel,expanded,leaf,row,hasFocus);
            Icon icon;
            if (value instanceof DefaultNoChildrenNode)
                icon=null;
            else {
                Entity entity=((EntityHierarchyNode)value).getEntity();
                icon=(entity instanceof Instance) ? m_instance : m_property;
            }
            if (tree.isEnabled())
                setIcon(icon);
            else
                setDisabledIcon(icon);
            return this;
        }
    }
}
