package de.fzi.wim.oimodeler.inspector;

import java.util.Collection;
import java.util.Collections;
import java.awt.CardLayout;
import java.awt.BorderLayout;
import javax.swing.JPanel;
import javax.swing.JLabel;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.oimodeler.ui.*;

/**
 * The inspector for OI-model entities.
 */
public class EntityInspector extends JPanel {
    /** The OI-modeler viewable. */
    protected OIModelerViewable m_oimodelerViewable;
    /** No objects selected pane. */
    protected JPanel m_noObjectsSelectedPane;
    /** Multiple objects selected pane. */
    protected JPanel m_multipleObjectsSelectedPane;
    /** Error accessing OI-model pane. */
    protected JPanel m_errorAccessingOIModelPane;
    /** The concept inspector. */
    protected ConceptInspector m_conceptInspector;
    /** The property inspector. */
    protected PropertyInspector m_propertyInspector;
    /** The instance inspector. */
    protected InstanceInspector m_instanceInspector;
    /** The current entity. */
    protected Entity m_entity;
    /** Set to <code>true</code> if an error for current entity has been reported. */
    protected boolean m_currentEntityReportedError;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable                 the OI-modeler viewable
     */
    public EntityInspector(OIModelerViewable oimodelerViewable) {
        super(new CardLayout());
        m_oimodelerViewable=oimodelerViewable;
        m_noObjectsSelectedPane=getMessagePanel("oimodeler.noObjectsSelected");
        m_multipleObjectsSelectedPane=getMessagePanel("oimodeler.multipleObjectsSelected");
        m_errorAccessingOIModelPane=getMessagePanel("oimodeler.errorAccessingOIModel");
        m_conceptInspector=new ConceptInspector(m_oimodelerViewable);
        m_propertyInspector=new PropertyInspector(m_oimodelerViewable);
        m_instanceInspector=new InstanceInspector(m_oimodelerViewable);
        add(m_noObjectsSelectedPane,"noObjectsSelected");
        add(m_multipleObjectsSelectedPane,"multipleObjectsSelected");
        add(m_errorAccessingOIModelPane,"errorAccessingOIModel");
        add(m_conceptInspector,"concept");
        add(m_propertyInspector,"property");
        add(m_instanceInspector,"instance");
        showSelection(Collections.EMPTY_LIST);
    }
    /**
     * Disposes of the inspector.
     */
    public void dispose() {
        if (m_conceptInspector!=null) {
            m_conceptInspector.dispose();
            m_conceptInspector=null;
        }
        if (m_propertyInspector!=null) {
            m_propertyInspector.dispose();
            m_propertyInspector=null;
        }
        if (m_instanceInspector!=null) {
            m_instanceInspector.dispose();
            m_instanceInspector=null;
        }
    }
    /**
     * Shows the selection.
     *
     * @param selection                         the selection
     */
    public void showSelection(Collection selection) {
        m_entity=null;
        m_currentEntityReportedError=false;
        CardLayout cardLayout=(CardLayout)getLayout();
        if (selection.isEmpty()) {
            cardLayout.show(this,"noObjectsSelected");
            m_conceptInspector.showNothing();
            m_propertyInspector.showNothing();
            m_instanceInspector.showNothing();
        }
        else if (selection.size()>1) {
            cardLayout.show(this,"multipleObjectsSelected");
            m_conceptInspector.showNothing();
            m_propertyInspector.showNothing();
            m_instanceInspector.showNothing();
        }
        else {
            m_entity=(Entity)selection.iterator().next();
            try {
                try {
                    m_oimodelerViewable.getModule().getAppDriver().startWaitState();
                    if (m_entity instanceof Concept) {
                        m_conceptInspector.showConcept((Concept)m_entity);
                        m_propertyInspector.showNothing();
                        m_instanceInspector.showNothing();
                        cardLayout.show(this,"concept");
                    }
                    else if (m_entity instanceof Property) {
                        m_conceptInspector.showNothing();
                        m_propertyInspector.showProperty((Property)m_entity);
                        m_instanceInspector.showNothing();
                        cardLayout.show(this,"property");
                    }
                    else if (m_entity instanceof Instance) {
                        m_conceptInspector.showNothing();
                        m_propertyInspector.showNothing();
                        m_instanceInspector.showInstance((Instance)m_entity);
                        cardLayout.show(this,"instance");
                    }
                }
                finally {
                    m_oimodelerViewable.getModule().getAppDriver().endWaitState();
                }
            }
            catch (KAONException e) {
                cardLayout.show(this,"errorAccessingOIModel");
                m_conceptInspector.showNothing();
                m_propertyInspector.showNothing();
                m_instanceInspector.showNothing();
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
            }
        }
    }
    /**
     * Notifies this inspector that an error has been reported for an entity. This mehtod then
     * computes whether the error nitification is necessary. This is used to avoid multiple
     * error notifications when the inspector changes.
     *
     * @param entity                            the entity for which the notification is reported
     * @return                                  <code>true</code> if the error should be presented to the user
     */
    public boolean shouldReportError(Entity entity) {
        if (m_entity==null || !m_entity.equals(entity))
            return true;
        else {
            boolean oldValue=m_currentEntityReportedError;
            m_currentEntityReportedError=true;
            return !oldValue;
        }
    }
    /**
     * Creates the panel with given message.
     *
     * @param messageID                         the ID of the message
     * @return                                  the panel with given message
     */
    protected JPanel getMessagePanel(String messageID) {
        JPanel panel=new JPanel(new BorderLayout());
        JLabel label=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getLabel(messageID);
        label.setHorizontalAlignment(JLabel.CENTER);
        panel.add(label,BorderLayout.CENTER);
        return panel;
    }
    /**
     * Sets the current language URI.
     *
     * @param languageURI                       the language URI
     * @throws KAONException                    thrown if there is an error
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        m_conceptInspector.setLanguageURI(languageURI);
        m_propertyInspector.setLanguageURI(languageURI);
        m_instanceInspector.setLanguageURI(languageURI);
    }
    /**
     * Notifies this component that a request is being processed.
     *
     * @param beingProcessed                    <code>true</code> if OI-model is being processed
     */
    public void setBeingProcessed(boolean beingProcessed) {
        m_conceptInspector.setBeingProcessed(beingProcessed);
        m_propertyInspector.setBeingProcessed(beingProcessed);
        m_instanceInspector.setBeingProcessed(beingProcessed);
    }
    /**
     * Called to determine if an element is shown or not.
     *
     * @param elementIndex          the index of the element
     * @return                      <code>true</code> if the given element is shown
     */
    public boolean isElementShown(int elementIndex) {
        if (m_entity instanceof Property)
            return m_propertyInspector.isElementShown(elementIndex);
        else if (m_entity instanceof Concept)
            return m_conceptInspector.isElementShown(elementIndex);
        else if (m_entity instanceof Instance)
            return m_instanceInspector.isElementShown(elementIndex);
        else
            return false;
    }
    /**
     * Called to set visibility of some element.
     *
     * @param elementIndex          the index of the element
     * @param isVisible             the status of the element
     */
    public void setElementVisibility(int elementIndex,boolean isVisible) {
        if (m_entity instanceof Property)
            m_propertyInspector.setElementVisibility(elementIndex,isVisible);
        else if (m_entity instanceof Concept)
            m_conceptInspector.setElementVisibility(elementIndex,isVisible);
        else if (m_entity instanceof Instance)
            m_instanceInspector.setElementVisibility(elementIndex,isVisible);
    }
    /**
     * Returns the names of the elements that the inspector can currently show.
     *
     * @return                      the names of the elements that the inspector can currently show
     */
    public String[] getAvailableElementNames() {
        if (m_entity instanceof Property)
            return m_propertyInspector.getAvailableElementNames();
        else if (m_entity instanceof Concept)
            return m_conceptInspector.getAvailableElementNames();
        else if (m_entity instanceof Instance)
            return m_instanceInspector.getAvailableElementNames();
        else
            return new String[0];
    }
    /**
     * Creates the element with given index.
     *
     * @param elementIndex          the index of the element
     * @return                      new element for given index
     * @throws KAONException        thrown if there is an error
     */
    public EntityInfoPage createElement(int elementIndex) throws KAONException {
        if (m_entity instanceof Property)
            return m_propertyInspector.createElement(elementIndex);
        else if (m_entity instanceof Concept)
            return m_conceptInspector.createElement(elementIndex);
        else if (m_entity instanceof Instance)
            return m_instanceInspector.createElement(elementIndex);
        else
            return null;
    }
}
