package de.fzi.wim.oimodeler.inspector;

import java.util.List;
import java.util.Iterator;
import java.awt.Color;
import java.awt.BorderLayout;
import java.awt.ComponentOrientation;
import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JComponent;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

import de.fzi.wim.guibase.menus.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.entityhierarchy.*;
import de.fzi.wim.oimodeler.conceptproperties.*;
import de.fzi.wim.oimodeler.selection.*;
import de.fzi.wim.oimodeler.lexicon.*;

/**
 * The inspector for concepts.
 */
public class ConceptInspector extends AbstractInspectorPanel implements OIModelListener {
    /** The currently shown concept. */
    protected Concept m_concept;
    /** The label showing the concept title. */
    protected JLabel m_conceptTitle;
    /** The tree of subconcepts. */
    protected OIModelTree m_subConcepts;
    /** The tree of superconcepts. */
    protected OIModelTree m_superConcepts;
    /** The table of properties from concept. */
    protected ConceptPropertiesTable m_propertiesFrom;
    /** The table of properties to concept. */
    protected ConceptPropertiesTable m_propertiesTo;
    /** The tree table for the concept instances. */
    protected ConceptInstancesTreeTable m_conceptInstances;
    /** The table for lexicon. */
    protected LexiconTable m_lexiconTable;
    /** Change visitor. */
    protected ChangeVisitor m_changeVisitor;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable                 the OI-modeler viewable
     */
    public ConceptInspector(OIModelerViewable oimodelerViewable) {
        super(oimodelerViewable);
        m_changeVisitor=new OIModelChangeVisitor();
        m_oimodelerViewable.getOIModel().addOIModelListener(this);
        m_conceptTitle=new JLabel();
        m_conceptTitle.setOpaque(true);
        m_conceptTitle.setBackground(Color.gray);
        m_conceptTitle.setForeground(Color.white);
        m_conceptTitle.setIcon(oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.concept"));
        m_conceptTitle.setIconTextGap(10);
        m_conceptTitle.setBorder(BorderFactory.createEmptyBorder(3,3,3,3));
        try {
            m_superConcepts=(OIModelTree)createElement(0);
            m_subConcepts=(OIModelTree)createElement(1);
            m_lexiconTable=(LexiconTable)createElement(2);
            m_propertiesFrom=(ConceptPropertiesTable)createElement(3);
            m_propertiesTo=(ConceptPropertiesTable)createElement(4);
            m_conceptInstances=(ConceptInstancesTreeTable)createElement(5);
        }
        catch (KAONException cantHappen) {
        }
        m_elements=new JComponent[] {
            createTitledScrollPane(m_superConcepts,"oimodeler.superConcepts"),
            createTitledScrollPane(m_subConcepts,"oimodeler.subConcepts"),
            createTitledScrollPane(m_lexiconTable,"oimodeler.lexicon"),
            createTitledScrollPane(m_propertiesFrom,"oimodeler.propertiesFrom"),
            createTitledScrollPane(m_propertiesTo,"oimodeler.propertiesTo"),
            createTitledScrollPane(m_conceptInstances,"oimodeler.conceptInstances"),
        };
        add(m_elementsPanel,BorderLayout.CENTER);
        add(m_conceptTitle,BorderLayout.NORTH);
        layoutVisibleElements();
    }
    /**
     * Creates an <code>OIModelTree</code> for super concepts. As well as the associated
     * <code>JTreePopupMenuManipulator</code> and <code>DoubleClickSelectionFocuser</code>.
     *
     * @return          the created <code>OIModelTree</code>
     */
    protected OIModelTree createSuperConcepts() {
        OIModelTree superConcepts=new OIModelTree(m_oimodelerViewable,new SuperConceptHierarchyNodeLoader(m_oimodelerViewable));
        superConcepts.setComponentOrientation(ComponentOrientation.RIGHT_TO_LEFT);
        superConcepts.setNodeIcon(m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.concept"));
        new JTreePopupMenuManipulator(superConcepts,m_oimodelerViewable.getModule().getAppDriver().createPopupMenu("popup.oimodeler.selection"));
        new DoubleClickSelectionFocuser(m_oimodelerViewable,superConcepts);
        return superConcepts;
    }
    /**
     * Creates an <code>OIModelTree</code> for sub concepts. As well as the associated
     * <code>JTreePopupMenuManipulator</code> and <code>DoubleClickSelectionFocuser</code>.
     *
     * @return          the created <code>OIModelTree</code>
     */
    protected OIModelTree createSubConcepts() {
        OIModelTree subConcepts=new OIModelTree(m_oimodelerViewable,new SubConceptHierarchyNodeLoader(m_oimodelerViewable));
        subConcepts.setNodeIcon(m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.concept"));
        new JTreePopupMenuManipulator(subConcepts,m_oimodelerViewable.getModule().getAppDriver().createPopupMenu("popup.oimodeler.selection"));
        new DoubleClickSelectionFocuser(m_oimodelerViewable,subConcepts);
        return subConcepts;
    }
    /**
     * Creates an <code>ConceptPropertiesTable</code> for "from" or "to"-properties. As well as the associated
     * <code>JTreePopupMenuManipulator</code> and <code>DoubleClickSelectionFocuser</code>.
     *
     * @param from      if <code>true</code> a <code>ConceptPropertiesTable</code> for "from"-properties will be created, a table for "to"-properties otherwise
     * @return          the created <code>ConceptPropertiesTable</code>
     */
    protected ConceptPropertiesTable createConceptPropertiesTable(boolean from) {
        ConceptPropertiesTable properties=new ConceptPropertiesTable(m_oimodelerViewable,from);
        new JTablePopupMenuManipulator(properties,m_oimodelerViewable.getModule().getAppDriver().createPopupMenu("popup.oimodeler.selection"));
        new DoubleClickSelectionFocuser(m_oimodelerViewable,properties);
        return properties;
    }
    /**
     * Disposes of the inspector.
     */
    public void dispose() {
        if (m_superConcepts!=null) {
            m_superConcepts.dispose();
            m_superConcepts=null;
        }
        if (m_subConcepts!=null) {
            m_subConcepts.dispose();
            m_subConcepts=null;
        }
        if (m_propertiesFrom!=null) {
            m_propertiesFrom.dispose();
            m_propertiesFrom=null;
        }
        if (m_propertiesTo!=null) {
            m_propertiesTo.dispose();
            m_propertiesTo=null;
        }
        if (m_conceptInstances!=null) {
            m_conceptInstances.dispose();
            m_conceptInstances=null;
        }
        if (m_lexiconTable!=null) {
            m_lexiconTable.dispose();
            m_lexiconTable=null;
        }
        if (m_oimodelerViewable!=null) {
            m_oimodelerViewable.getOIModel().removeOIModelListener(this);
            m_oimodelerViewable=null;
        }
    }
    /**
     * Creates the element for given index.
     *
     * @param elementIndex          the index of the element
     * @return                      the new element with given index
     * @throws KAONException        thrown if there is an error
     */
    public EntityInfoPage createElement(int elementIndex) throws KAONException {
        switch (elementIndex) {
        case 0:
            return showEntity(createSuperConcepts(),m_concept);
        case 1:
            return showEntity(createSubConcepts(),m_concept);
        case 2:
            return showEntity(new LexiconTable(m_oimodelerViewable),m_concept);
        case 3:
            return showEntity(createConceptPropertiesTable(true),m_concept);
        case 4:
            return showEntity(createConceptPropertiesTable(false),m_concept);
        case 5:
            {
                ConceptInstancesTreeTable element=new ConceptInstancesTreeTable(m_oimodelerViewable);
                if (m_concept==null)
                    element.showNothing();
                else
                    element.showEntityIndirectly(m_concept);
                return element;
            }
        default:
            throw new IllegalArgumentException("Illegal element index "+elementIndex);
        }
    }
    /**
     * Shows the given concept.
     *
     * @param concept                           the concept
     * @throws KAONException                    thrown if there is a problem with accessing the OI-model
     */
    public void showConcept(Concept concept) throws KAONException {
        m_concept=concept;
        m_conceptTitle.setText(concept.getURI());
        m_superConcepts.showEntity(concept);
        m_subConcepts.showEntity(concept);
        m_propertiesFrom.showEntity(concept);
        m_propertiesTo.showEntity(concept);
        m_conceptInstances.showEntityIndirectly(concept);
        m_lexiconTable.showEntity(concept);
    }
    /**
     * Makes sure nothing is shown.
     */
    public void showNothing() {
        m_concept=null;
        m_superConcepts.showNothing();
        m_subConcepts.showNothing();
        m_propertiesFrom.showNothing();
        m_propertiesTo.showNothing();
        m_conceptInstances.showNothing();
        m_lexiconTable.showNothing();
    }
    /**
     * Sets the language URI.
     *
     * @param languageURI                       the language URI
     * @throws KAONException                    thrown if there is an error
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        m_subConcepts.setLanguageURI(languageURI);
        m_superConcepts.setLanguageURI(languageURI);
        m_propertiesFrom.setLanguageURI(languageURI);
        m_propertiesTo.setLanguageURI(languageURI);
        m_conceptInstances.setLanguageURI(languageURI);
    }
    /**
     * Notifies this component that a request is being processed.
     *
     * @param beingProcessed                    <code>true</code> if OI-model is being processed
     */
    public void setBeingProcessed(boolean beingProcessed) {
        m_superConcepts.setEnabled(beingProcessed);
        m_subConcepts.setEnabled(beingProcessed);
        m_propertiesFrom.setEnabled(beingProcessed);
        m_propertiesTo.setEnabled(beingProcessed);
        m_conceptInstances.setEnabled(beingProcessed);
        m_lexiconTable.setEnabled(beingProcessed);
    }
    /**
     * Called when a bulk of change events is processed in the pool.
     *
     * @param oimodel                   OI-model that was changed
     * @param changeEvents              list of change events that occured
     */
    public void modelChanged(OIModel oimodel,List changeEvents) {
        try {
            Iterator iterator=changeEvents.iterator();
            while (iterator.hasNext()) {
                ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                changeEvent.accept(m_changeVisitor);
            }
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
    }
    /**
     * Called when model is refreshed.
     *
     * @param oimodel                   OI-model that was refreshed
     */
    public void modelRefreshed(OIModel oimodel) {
    }
    /**
     * Called when model is being deleted.
     *
     * @param oimodel                   OI-model that is being deleted
     */
    public void modelDeleted(OIModel oimodel) {
    }

    /**
     * The visitor for OI-model events.
     */
    protected class OIModelChangeVisitor extends NullChangeEventVisitor {
        public void visit(RemoveEntity event) {
            if (event.getEntity().equals(m_concept))
                showNothing();
        }
    }
}
