package de.fzi.wim.oimodeler.inspector;

import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JComponent;
import javax.swing.JScrollPane;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.guibase.dnd.*;

import de.fzi.wim.oimodeler.ui.*;

/**
 * Abstract base panel for the inspectors.
 */
public abstract class AbstractInspectorPanel extends JPanel {
    /** The OI-modeler module. */
    protected OIModelerViewable m_oimodelerViewable;
    /** The panel for items.*/
    protected JPanel m_elementsPanel;
    /** The elements in the panel. */
    protected JComponent[] m_elements;

    /**
     * Creates a new instance of AbstractInspectorPanel.
     *
     * @param oimodelerViewable                 the OI-modeler viewable
     */
    public AbstractInspectorPanel(OIModelerViewable oimodelerViewable) {
        super(new BorderLayout());
        m_oimodelerViewable=oimodelerViewable;
        m_elementsPanel=new JPanel(new GridBagLayout());
    }
    /**
     * Creates scrollpane with the title for the component.
     *
     * @param component                         the component
     * @param titleID                           the ID of the title, if <code>null</code> no title will be set
     * @return                                  the scrollpane
     */
    protected JScrollPane createTitledScrollPane(JComponent component,String titleID) {
        component.setAutoscrolls(true);
        JScrollPane scrollPane=new JScrollPane();
        scrollPane.setViewport(new BorderedViewport());
        scrollPane.setViewportView(component);
        if (titleID!=null) {
            String title=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase(titleID);
            scrollPane.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(),title));
        }
        scrollPane.putClientProperty("oimodeler.paneID",titleID);
        return scrollPane;
    }
    /**
     * Adjusts the layout of the elements panel.
     */
    protected void layoutVisibleElements() {
        m_elementsPanel.removeAll();
        GridBagConstraints gbc=new GridBagConstraints();
        gbc.weighty=100;
        gbc.weightx=100;
        gbc.fill=GridBagConstraints.BOTH;
        gbc.gridx=0;
        gbc.gridy=0;
        int numberOfVisibleItems=0;
        for (int i=0;i<m_elements.length;i++)
            if (m_elements[i].isVisible())
                numberOfVisibleItems++;
        int lineItemCount=numberOfVisibleItems/2;
        if(lineItemCount*2<numberOfVisibleItems){
            gbc.gridwidth=lineItemCount;
            lineItemCount++;
        }
        for (int i=0;i<m_elements.length;i++) {
            JComponent component=m_elements[i];
            if (component.isVisible()) {
                if(m_elementsPanel.getComponentCount()==lineItemCount) {
                    gbc.gridy++;
                    gbc.gridx=0;
                    if(lineItemCount*2>numberOfVisibleItems)
                        gbc.gridwidth++;
                }
                m_elementsPanel.add(component,gbc);
                gbc.gridx+=gbc.gridwidth;
            }
        }
        m_elementsPanel.revalidate();
    }
    /**
     * Called to determine if an element is shown or not.
     *
     * @param elementIndex          the index of the element
     * @return                      <code>true</code> if the given element is shown
     */
    public boolean isElementShown(int elementIndex) {
        return m_elements[elementIndex].isVisible();
    }
    /**
     * Called to set visibility of some element.
     *
     * @param elementIndex          the index of the element
     * @param isVisible             the status of the element
     */
    public void setElementVisibility(int elementIndex,boolean isVisible) {
        m_elements[elementIndex].setVisible(isVisible);
        layoutVisibleElements();
    }
    /**
     * Returns the names of the elements that the inspector can currently show.
     *
     * @return                      the names of the elements that the inspector can currently show
     */
    public String[] getAvailableElementNames() {
        String[] result=new String[m_elements.length];
        for (int i=0;i<m_elements.length;i++)
            result[i]=(String)m_elements[i].getClientProperty("oimodeler.paneID");
        return result;
    }
    /**
     * Creates the element for given index.
     *
     * @param elementIndex          the index of the element
     * @return                      the new element with given index
     * @throws KAONException        thrown if there is an error
     */
    public abstract EntityInfoPage createElement(int elementIndex) throws KAONException;
    /**
     * Shows given entity in the element.
     *
     * @param element               the element
     * @param entity                the entity
     * @return                      the element
     * @throws KAONException        thrown if there is an error
     */
    protected EntityInfoPage showEntity(EntityInfoPage element,Entity entity) throws KAONException {
        if (entity!=null)
            element.showEntity(entity);
        else
            element.showNothing();
        return element;
    }
}
