package de.fzi.wim.oimodeler.inference;

import java.io.IOException;
import java.util.Collection;
import java.awt.Component;
import java.awt.Insets;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Color;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.datatransfer.UnsupportedFlavorException;
import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JButton;
import javax.swing.JScrollPane;
import javax.swing.JViewport;
import javax.swing.plaf.TableUI;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.guibase.dnd.*;
import de.fzi.wim.guibase.localization.*;
import de.fzi.wim.guibase.selectionsource.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.sidebar.*;

/**
 * The pane for the clipboard.
 */
public class InferencePane extends JPanel {
    /** The OI-modeler viewable. */
    protected OIModelerViewable m_oimodelerViewable;
    /** The table for the inferenced properties. */
    protected InferenceTable m_inferenceTable;
    /** The currently shown entity. */
    protected Entity m_entity;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable                 the OI-modeler viewable
     */
    public InferencePane(OIModelerViewable oimodelerViewable) {
        super(new GridBagLayout());
        m_oimodelerViewable=oimodelerViewable;
        m_inferenceTable=new InferenceTable(oimodelerViewable);
        LocalizationManager localizationManager=oimodelerViewable.getModule().getAppDriver().getLocalizationManager();
        JButton refreshButton=localizationManager.getButton("oimodeler.refresh");
        //refreshButton.setIconTextGap(0);
        refreshButton.setMargin(new Insets(1,1,1,1));
        refreshButton.setToolTipText(localizationManager.getPhrase("oimodeler.refresh.inferences"));
        refreshButton.setIcon(localizationManager.getImageIcon("action.kaongui.refresh.icon"));
        refreshButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                reloadEntity();
            }
        });
        GridBagConstraints gbc=new GridBagConstraints();
        JScrollPane scrollPane=new JScrollPane();
        scrollPane.setViewport(new BorderedViewport());
        scrollPane.setViewportView(m_inferenceTable);
        //gbc.insets=new Insets(1,1,0,1);
        gbc.weightx=100;
        gbc.gridwidth=1;
        gbc.gridx=0;
        gbc.gridy=0;

        gbc.weighty=98;
        gbc.anchor=GridBagConstraints.CENTER;
        gbc.fill=GridBagConstraints.BOTH;
        add(scrollPane,gbc);

        gbc.weighty=2;
        gbc.anchor=GridBagConstraints.SOUTHEAST;
        gbc.fill=GridBagConstraints.NONE;
        gbc.gridy++;
        add(refreshButton,gbc);
    }
    /**
     * Disposes of the inference pane.
     */
    public void dispose() {
        if (m_inferenceTable!=null) {
            m_inferenceTable.dispose();
            m_inferenceTable=null;
        }
    }
    /**
     * Sets the current language URI.
     *
     * @param languageURI                       the language URI
     * @throws KAONException                    thrown if there is a problem
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        m_inferenceTable.setLanguageURI(languageURI);
        String label=m_entity.getLabel(languageURI);
        if (label==null) {
            label=m_entity.getURI();
            int hashPosition=label.indexOf('#');
            if (hashPosition!=-1)
                label=label.substring(hashPosition+1);
        }
        String message=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().format("oimodeler.sidebar.item.inference",new Object[] { label });
        setName(message);
        m_inferenceTable.setLanguageURI(languageURI);
    }
    /**
     * Shows the entity in the inference pane.
     *
     * @param entity            entity to be shown
     */
    public void showEntity(Entity entity) {
        m_entity=entity;
        reloadEntity();
    }
    /**
     * Reloads the entity.
     *
     */
    public void reloadEntity() {
        if (m_entity instanceof Concept)
            m_inferenceTable.showEntityIndirectly(m_entity);
        else if (m_entity instanceof Instance)
            m_inferenceTable.showEntityIndirectly(m_entity);
        else
            m_inferenceTable.showEntity(m_entity);
    }

    /**
     * Table showing the inferences.
     */
    protected class InferenceTable extends InferedConceptInstancesTreeTable {
        /** The drop target for the table. */
        protected InferenceTableDropTarget m_dropTarget;
        /** The pane owning this table. */
        protected InferencePane m_pane;

        public InferenceTable(OIModelerViewable viewable) {
            super(viewable);
            m_pane=InferencePane.this;
            m_dropTarget=new InferenceTableDropTarget();
        }
        /**
         * Overridden to set the color of the enclosing viewport.
         */
        protected void configureEnclosingScrollPane() {
            super.configureEnclosingScrollPane();
            Component parent=getParent();
            if (parent instanceof BorderedViewport)
                m_dropTarget.setBorderedViewport((BorderedViewport)parent);
        }
        /**
         * Updates the UI.
         */
        public void updateUI() {
            super.updateUI();
            Component parent=getParent();
            if (parent instanceof JViewport)
                parent.setBackground(getBackground());
        }
        /**
         * Overridden to set the non-dragging UI. Drag-and-drop and the UI don't work well together.
         *
         * @param ui                                            the UI (ignored)
         */
        public void setUI(TableUI ui) {
            super.setUI(new SelectionSourceDragSource.NoDragSelectionTableUI());
        }
        /**
         * Sets the current background color. Overridden to update the enclosing viewport's color as well.
         *
         * @param color                                         the color
         */
        public void setBackground(Color color) {
            super.setBackground(color);
            if (getParent() instanceof JViewport)
                getParent().setBackground(color);
        }

        /**
         * The drop target for the table.
         */
        protected class InferenceTableDropTarget extends JTableDropTarget {
            public InferenceTableDropTarget() {
                super(InferenceTable.this,null);
            }
            protected boolean acceptsDrag(DropTargetDragEvent dtde) {
                return dtde.isDataFlavorSupported(CollectionSelection.s_objectCollectionFlavor);
            }
            protected boolean processDrop(DropTargetDropEvent dtde) {
                Collection collection;
                try {
                    collection=(Collection)dtde.getTransferable().getTransferData(CollectionSelection.s_objectCollectionFlavor);
                }
                catch (IOException ignored) {
                    return false;
                }
                catch (UnsupportedFlavorException ignored) {
                    return false;
                }
                if (collection.size()==1) {
                    Object entity=collection.iterator().next();
                    if (entity instanceof Entity) {
                        InferenceTable.this.m_pane.showEntity((Entity)entity);
                    }
                    return true;
                }
                else
                    return false;
            }
        }
    }

    /**
     * The adapter for the sidebar.
     */
    public class SideBarAdapter implements OIModelerSideBar.SideBarElement {
        /**
         * Sets the current language URI.
         *
         * @param languageURI                       the language URI
         * @throws KAONException                    thrown if there is a problem
         */
        public void setLanguageURI(String languageURI) throws KAONException {
            InferencePane.this.setLanguageURI(languageURI);
        }
        /**
         * Returns the label of the element.
         *
         * @return                                  the name of the element
         */
        public String getLabel() {
            String name;
            try {
                name=m_entity.getLabel(m_oimodelerViewable.getLanguageURI());
                if (name==null)
                    name=m_entity.getURI();
            }
            catch (KAONException ignored) {
                name="<unkown entity>";
            }
            return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().format("oimodeler.sidebar.item.inference",new String[] { name });
        }
        /**
         * Returns the description of the element.
         *
         * @return                                  the description of the element
         */
        public String getDescription() {
            return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("oimodeler.sidebar.item.inference.description");
        }
        /**
         * Returns the icon of the element.
         *
         * @return                                  the icon of the element
         */
        public Icon getIcon() {
            return null;
        }
        /**
         * Returns the component.
         *
         * @return                                  the component
         */
        public JComponent getComponent() {
            return InferencePane.this;
        }
        /**
         * Called when the element is evicted from the sidebar.
         */
        public void removedFromSideBar() {
            dispose();
        }
    }
}
