package de.fzi.wim.oimodeler.inference;

import javax.swing.tree.TreeNode;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Node in the tree representing a property instance in the ontology.
 */
public class InferedPropertyInstanceNode extends InferedOIModelNode {
    /** The uncoupled property. */
    protected UncoupledEntity m_property;
    /** The uncoupled target instance (if the target it an instance). */
    protected UncoupledEntity m_targetinstance;
    /** The label of property instance. */
    protected String m_label;
    /** The target value label. */
    protected String m_targetValueLabel;
    /** Set to <code>true</code> if this property is an attribute. */
    protected boolean m_isAttribute;

    /**
     * Creates an adaptor for given property instance.
     *
     * @param parent                parent of this node
     * @param propertyInstance      property instance represented by this node
     * @param languageUri       language to be used
     * @throws KAONException        thrown if there is an error
     */
    public InferedPropertyInstanceNode(TreeNode parent, PropertyInstance propertyInstance, String languageUri) throws KAONException {
        super(parent);
        updatePropertyInstance(propertyInstance,languageUri);
    }
    /**
     * Creates an adaptor for given concept.
     *
     * @param parent                parent of this node
     * @param property              property this is an instance of
     * @param targetValue           target value of this instance
     * @param languageUri           language to be used
     * @throws KAONException        thrown if there is an error
     */
    public InferedPropertyInstanceNode(TreeNode parent, Property property, Object targetValue, String languageUri) throws KAONException {
        super(parent);
        updatePropertyInstance(property,targetValue,languageUri);
    }
    /**
     * Refreshes the property instance information.
     *
     * @param   propertyInstance    property instance to be uncoupled
     * @param   languageUri         uri to the language to be used
     * @throws KAONException        thrown if some error occured
     */
    protected void updatePropertyInstance(PropertyInstance propertyInstance, String languageUri) throws KAONException {
        updatePropertyInstance(propertyInstance.getProperty(),propertyInstance.getTargetValue(),languageUri);
    }
    /**
     * Refreshes the property instance information.
     *
     * @param property              property this is an instance of
     * @param targetValue           target value of this instance
     * @param languageUri           uri to the language to be used
     * @throws KAONException        thrown if some error occured
     */
    protected void updatePropertyInstance(Property property,Object targetValue,String languageUri) throws KAONException {
        m_property=new UncoupledEntity((Entity)property,languageUri);
        m_label=property.getLabel(languageUri);
        if (m_label==null)
            m_label=property.getURI();
        m_targetinstance=null;
        if (targetValue instanceof Instance) {
            m_targetinstance=new UncoupledEntity((Entity)targetValue,languageUri);
            m_targetValueLabel=m_targetinstance.getLabel();
        }
        else
            m_targetValueLabel=targetValue.toString();

        m_isAttribute=property.isAttribute();
    }
    /**
     * Returns the label of the property instance.
     *
     * @return                      the label of the property instance
     */
    public String getPropertyLabel() {
        return m_label;
    }
    /**
     * Returns the label of the target value.
     *
     * @return                      the label of the target value
     */
    public String getTargetValueLabel() {
        return m_targetValueLabel;
    }
    /**
     * Returns whether the node is an attribute node.
     *
     * @return                      <code>true</code> if the node is an attribute
     */
    public boolean isAttribute() {
        return m_isAttribute;
    }
    /**
     * Returns <code>false</code>.
     *
     * @return                      always <code>false</code>, since this node can't have children
     */
    public boolean getAllowsChildren() {
        return false;
    }
    /**
     * Returns the label of the node in the tree.
     *
     * @return                      label of the node in the tree
     */
    public String toString() {
        return m_label;
    }
    /**
     * Returns the uncoupled property instance for the node.
     *
     * @return                      the uncoupled property intance for the node
     *
     */
    public UncoupledEntity getEntity() {
        return m_property;
    }
    /**
     * Returns the uncoupled property for the node. (Returns the same entity as <code>getEntity()</code>.)
     *
     * @return                      the uncoupled property intance for the node
     *
     */
    public UncoupledEntity getProperty() {
        return getEntity();
    }
    /**
     * Returns the uncoupled target value instance for the node. (May be <code>null</code> if the target value is not an instance.)
     *
     * @return                      the target value instance for the node
     *
     */
    public UncoupledEntity getTargetValueInstance() {
        return m_targetinstance;
    }
}
