package de.fzi.wim.oimodeler.inference;

import javax.swing.tree.TreeNode;
import javax.swing.SwingUtilities;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.guibase.lazytreemodel.*;

import de.fzi.wim.oimodeler.ui.*;

/**
 * Node loader for some tree of the OI-model, without monitoring changes on the nodes and the model.
 */
public abstract class InferedOIModelNodeLoader implements LazyNodeLoader {
    /** The OI-modeler viewable. */
    protected OIModelerViewable m_oimodelerViewable;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable     the OI-modeler viewable
     */
    public InferedOIModelNodeLoader(OIModelerViewable oimodelerViewable) {
        m_oimodelerViewable=oimodelerViewable;
    }
    /**
     * Loads children of ontology nodes.
     *
     * @param model                 model that requested the load
     * @param node                  node whose children need to be loaded
     * @return                      array of loaded children (or {@link #LOAD_DEFERRED} indicating that load will be done later)
     */
    public TreeNode[] loadNodeChildren(LazyTreeModel model,LazyTreeNode node) {
        try {
            if ((m_oimodelerViewable.getOIModel().getCapabilities() & OIModel.CAPABILITY_SUPPORTS_OPTIMIZED_LOADING)!=0) {
                loadNodeChildrenAsync(model,node);
                return LOAD_DEFERRED;
            }
            else
                return loadNodeChildrenEx(model,node);
        }
        catch (InterruptedException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
            return LOAD_CANCELED;
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
            return LOAD_CANCELED;
        }
    }
    /**
     * Loads children of ontology nodes asynchronously.
     *
     * @param model                 model that requested the load
     * @param node                  node whose children need to be loaded
     * @throws InterruptedException thrown if loading cannot be performed
     */
    protected void loadNodeChildrenAsync(final LazyTreeModel model,final LazyTreeNode node) throws InterruptedException {
        m_oimodelerViewable.getModule().getAppDriver().getThreadPool().executeTask(new Runnable() {
            public void run() {
                try {
                    final TreeNode[] treeNodes;
                    synchronized (m_oimodelerViewable.getOIModel().getKAONConnection()) {
                        treeNodes=loadNodeChildrenEx(model,node);
                    }
                    SwingUtilities.invokeLater(new Runnable() {
                        public void run() {
                            model.notifyChildrenLoaded(node,treeNodes);
                        }
                    });
                }
                catch (final KAONException e) {
                    SwingUtilities.invokeLater(new Runnable() {
                        public void run() {
                            model.notifyChildrenLoaded(node,LOAD_CANCELED);
                            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
                        }
                    });
                }
            }
        });
    }
    /**
     * Loads children of ontology nodes, but throws exceptions if there is a problem.
     *
     * @param model                 model that requested the load
     * @param node                  node whose children need to be loaded
     * @return                      array of loaded children (or {@link #LOAD_DEFERRED} indicating that load will be done later)
     * @throws KAONException        the exception
     */
    protected abstract TreeNode[] loadNodeChildrenEx(LazyTreeModel model,LazyTreeNode node) throws KAONException;
    /**
     * Returns the position at which the supplied node should be inserted.
     *
     * @param node                  the node in whose children insertion is done
     * @param newNode               the new node
     * @return                      the position of the new node
     */
    public abstract int getInsertPosition(InferedOIModelNode node,InferedOIModelNode newNode);
}
