package de.fzi.wim.oimodeler.inference;

import java.util.Set;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import javax.swing.tree.TreeNode;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.guibase.lazytreemodel.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.viewfilter.*;

/**
 * Node loader for the entity hierarchy without monitoring changes.
 */
public abstract class InferedEntityHierarchyNodeLoader extends InferedOIModelNodeLoader {

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable     the OI-modeler viewable
     */
    public InferedEntityHierarchyNodeLoader(OIModelerViewable oimodelerViewable) {
        super(oimodelerViewable);
    }
    /**
     * Loads children of ontology nodes, but throws exceptions if there is a problem.
     *
     * @param model                 model that requested the load
     * @param node                  node whose children need to be loaded
     * @return                      array of loaded children (or {@link #LOAD_DEFERRED} indicating that load will be done later)
     * @throws KAONException        the exception
     */
    protected TreeNode[] loadNodeChildrenEx(LazyTreeModel model,LazyTreeNode node) throws KAONException {
        ViewFilter viewFilter=m_oimodelerViewable.getViewFilter();
        InferedOIModelTreeModel oimodelTreeModel=(InferedOIModelTreeModel)model;
        OIModel oimodel=m_oimodelerViewable.getOIModel();
        UncoupledEntity entity=((InferedEntityHierarchyNode)node).getEntity();
        Set entities=getRelatedEntities(entity,oimodel,oimodelTreeModel.getLanguageURI());
        oimodel.loadObjects(entities,getRelatedEntitiesLoadFlag());
        List children=new ArrayList(entities.size());
        Iterator elements=entities.iterator();
        while (elements.hasNext()) {
            Entity child=(Entity)elements.next();
            if (viewFilter.showEntity(child)) {
                InferedEntityHierarchyNode newNode=new InferedEntityHierarchyNode(node,child,oimodelTreeModel.getLanguageURI());
                children.add(newNode);
            }
        }
        TreeNode[] childrenAsArray=new TreeNode[children.size()];
        children.toArray(childrenAsArray);
        Arrays.sort(childrenAsArray,NodeLabelComparator.INSTANCE);
        return childrenAsArray;
    }
    /**
     * Returns the position at which the supplied node should be inserted.
     *
     * @param node                  the node in whose children insertion is done
     * @param newNode               the new node
     * @return                      the position of the new node
     */
    public int getInsertPosition(InferedOIModelNode node,InferedOIModelNode newNode) {
        TreeNode[] children=node.getChildren();
        for (int i=0;i<children.length;i++) {
            int result=NodeLabelComparator.INSTANCE.compare(children[i],newNode);
            if (result>=0)
                return i;
        }
        return children.length;
    }
    /**
     * Returns the index of the child containing given entity.
     *
     * @param node                  the node
     * @param childEntity           the child entity
     * @return                      the index of the child (or -1)
     */
    protected int getChildEntityIndex(InferedOIModelNode node,UncoupledEntity childEntity) {
        TreeNode[] children=node.getChildren();
        for (int i=0;i<children.length;i++) {
            InferedEntityHierarchyNode childNode=(InferedEntityHierarchyNode)children[i];
            if (childNode.getEntity().equals(childEntity))
                return i;
        }
        return -1;
    }
    /**
     * Returns the load flag for related entities.
     *
     * @return                      the load flag
     */
    protected abstract int getRelatedEntitiesLoadFlag();
    /**
     * Returns the set of related concepts.
     *
     * @param entity                the entity
     * @param oimodel               the oimodel to load from
     * @param languageUri           the language used for loading
     * @return                      the set of related entities
     * @throws KAONException        thrown if there is an error
     */
    protected abstract Set getRelatedEntities(UncoupledEntity entity,OIModel oimodel,String languageUri) throws KAONException;
    /**
     * The comparator for the nodes.
     */
    protected static class NodeLabelComparator implements Comparator {
        public static final Comparator INSTANCE=new NodeLabelComparator();

        public int compare(Object o1,Object o2) {
            InferedOIModelNode n1=(InferedOIModelNode)o1;
            InferedOIModelNode n2=(InferedOIModelNode)o2;
            String label1=n1.toString();
            String label2=n2.toString();
            return label1.compareToIgnoreCase(label2);
        }
    }
}
