package de.fzi.wim.oimodeler.inclusion;

import java.util.Set;
import java.util.List;
import java.util.Iterator;
import java.util.Enumeration;
import java.util.Arrays;
import java.util.Collections;
import javax.swing.tree.TreeNode;
import javax.swing.tree.DefaultTreeModel;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

import de.fzi.wim.oimodeler.ui.*;

/**
 * Tree model representing the inclusion hierarchy of the OI-models.
 */
public class OIModelInclusionTreeModel extends DefaultTreeModel implements OIModelListener {
    /** The OIModelerViewable. */
    protected OIModelerViewable m_oimodelerViewable;
    /** Change visitor. */
    protected ChangeVisitor m_changeVisitor;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable                 the OI-modeler viewable
     * @throws KAONException                    thrown of there is an error
     */
    public OIModelInclusionTreeModel(OIModelerViewable oimodelerViewable) throws KAONException {
        super(new OIModelNode(null,oimodelerViewable.getOIModel()));
        m_oimodelerViewable=oimodelerViewable;
        m_oimodelerViewable.getOIModel().addOIModelListener(this);
        m_changeVisitor=new OIModelChangeVisitor();
    }
    /**
     * Disposes of this model.
     */
    public void dispose() {
        if (m_oimodelerViewable!=null) {
            m_oimodelerViewable.getOIModel().removeOIModelListener(this);
            m_oimodelerViewable=null;
            setRoot(null);
        }
    }
    /**
     * Called when a bulk of change events is processed in the pool.
     *
     * @param oimodel                   OI-model that was changed
     * @param changeEvents              list of change events that occured
     */
    public void modelChanged(OIModel oimodel,List changeEvents) {
        try {
            Iterator iterator=changeEvents.iterator();
            while (iterator.hasNext()) {
                ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                changeEvent.accept(m_changeVisitor);
            }
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
    }
    /**
     * Called when model is refreshed.
     *
     * @param oimodel                   OI-model that was refreshed
     */
    public void modelRefreshed(OIModel oimodel) {
        try {
            setRoot(new OIModelNode(null,m_oimodelerViewable.getOIModel()));
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
    }
    /**
     * Called when model is being deleted.
     *
     * @param oimodel                   OI-model that is being deleted
     */
    public void modelDeleted(OIModel oimodel) {
    }


    /**
     * The node representing an OI-model.
     */
    public static class OIModelNode implements TreeNode {
        /** The OI-model. */
        protected OIModel m_oimodel;
        /** The logical URI of the model. */
        protected String m_logicalURI;
        /** The parent node. */
        protected OIModelNode m_parent;
        /** The array of children. */
        protected OIModelNode[] m_children;

        /**
         * Creates an instance of this class.
         *
         * @param parent                        the parent node
         * @param oimodel                       the OI-model
         * @throws KAONException                thrown of there is an error
         */
        public OIModelNode(OIModelNode parent,OIModel oimodel) throws KAONException {
            m_oimodel=oimodel;
            m_logicalURI=m_oimodel.getLogicalURI();
            m_parent=parent;
            Set includedOIModels=oimodel.getIncludedOIModels();
            m_children=new OIModelNode[includedOIModels.size()];
            Iterator iterator=includedOIModels.iterator();
            int index=0;
            while (iterator.hasNext()) {
                OIModel includedOIModel=(OIModel)iterator.next();
                m_children[index++]=new OIModelNode(this,includedOIModel);
            }
        }
        public Enumeration children() {
            return Collections.enumeration(Arrays.asList(m_children));
        }
        public boolean getAllowsChildren() {
            return true;
        }
        public TreeNode getChildAt(int childIndex) {
            return m_children[childIndex];
        }
        public int getChildCount() {
            return m_children.length;
        }
        public int getIndex(TreeNode node) {
            for (int i=0;i<m_children.length;i++)
                if (node==m_children[i])
                    return i;
            return -1;
        }
        public TreeNode getParent() {
            return m_parent;
        }
        public boolean isLeaf() {
            return getChildCount()==0;
        }
        public String toString() {
            return m_logicalURI;
        }
        public OIModel getOIModel() {
            return m_oimodel;
        }
        public OIModel getParentOIModel() {
            if (m_parent==null)
                return null;
            else
                return m_parent.m_oimodel;
        }
    }

    /**
     * The visitor for OI-model events.
     */
    protected class OIModelChangeVisitor extends NullChangeEventVisitor {
        public void visit(AddIncludedOIModel event) throws KAONException {
            addOIModel((OIModelNode)getRoot(),event.getOIModel(),event.getIncludedOIModel());
        }
        public void visit(RemoveIncludedOIModel event) throws KAONException {
            removeOIModel((OIModelNode)getRoot(),event.getOIModel(),event.getIncludedOIModel());
        }
        protected void addOIModel(OIModelNode oimodelNode,OIModel addToOIModel,OIModel oimodelToAdd) throws KAONException {
            if (oimodelNode.m_oimodel==addToOIModel) {
                OIModelNode newChild=new OIModelNode(oimodelNode,oimodelToAdd);
                int existingSize=oimodelNode.m_children.length;
                OIModelNode[] newChildren=new OIModelNode[existingSize+1];
                System.arraycopy(oimodelNode.m_children,0,newChildren,0,existingSize);
                newChildren[existingSize]=newChild;
                oimodelNode.m_children=newChildren;
                fireTreeNodesInserted(OIModelInclusionTreeModel.this,getPathToRoot(oimodelNode),new int[] { existingSize },new Object[] { newChild });
            }
            for (int i=0;i<oimodelNode.m_children.length;i++)
                addOIModel(oimodelNode.m_children[i],addToOIModel,oimodelToAdd);
        }
        protected void removeOIModel(OIModelNode oimodelNode,OIModel deleteFromOIModel,OIModel oimodelToDelete) throws KAONException {
            if (oimodelNode.m_oimodel==deleteFromOIModel) {
                for (int i=oimodelNode.m_children.length-1;i>=0;i--) {
                    OIModelNode deletedChild=oimodelNode.m_children[i];
                    if (deletedChild.m_oimodel==oimodelToDelete) {
                        int existingSize=oimodelNode.m_children.length;
                        OIModelNode[] newChildren=new OIModelNode[existingSize-1];
                        System.arraycopy(oimodelNode.m_children,0,newChildren,0,i);
                        System.arraycopy(oimodelNode.m_children,i+1,newChildren,i,existingSize-i-1);
                        oimodelNode.m_children=newChildren;
                        fireTreeNodesRemoved(OIModelInclusionTreeModel.this,getPathToRoot(oimodelNode),new int[] { i },new Object[] { deletedChild });
                    }
                }
            }
            for (int i=0;i<oimodelNode.m_children.length;i++)
                removeOIModel(oimodelNode.m_children[i],deleteFromOIModel,oimodelToDelete);
        }
    }
}
