package de.fzi.wim.oimodeler.entitylist;

import java.util.Set;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import javax.swing.Icon;
import javax.swing.table.AbstractTableModel;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

import de.fzi.wim.guibase.localization.*;

import de.fzi.wim.oimodeler.ui.*;

/**
 * The table model for displaying a list of entities.
 */
public class OIModelEntityTableModel extends AbstractTableModel implements OIModelListener {
    /** The OI-modeler viewable. */
    protected OIModelerViewable m_oimodelerViewable;
    /** The language URI. */
    protected String m_languageURI;
    /** The list of all elements in the table. */
    protected List m_elements;
    /** Change visitor. */
    protected ChangeVisitor m_changeVisitor;
    /** The icon for concpets. */
    protected Icon m_iconConcept;
    /** The icon for properties. */
    protected Icon m_iconProperty;
    /** The icon for instances. */
    protected Icon m_iconInstance;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable             the viewable
     */
    public OIModelEntityTableModel(OIModelerViewable oimodelerViewable) {
        m_oimodelerViewable=oimodelerViewable;
        m_elements=new ArrayList();
        m_changeVisitor=new OIModelChangeVisitor();
        m_oimodelerViewable.getOIModel().addOIModelListener(this);
        m_iconConcept=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.concept");
        m_iconProperty=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.property");
        m_iconInstance=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.instance");
    }
    /**
     * Disposes of this model.
     */
    public void dispose() {
        if (m_oimodelerViewable!=null) {
            m_oimodelerViewable.getOIModel().removeOIModelListener(this);
            m_oimodelerViewable=null;
            m_elements=null;
        }
    }
    /**
     * Shows nothing.
     */
    public void showNothing() {
        m_elements=new ArrayList();
        fireTableDataChanged();
    }
    /**
     * Returns the class of the column.
     *
     * @param columnIndex                   the index of the column
     * @return                              the column class
     */
    public Class getColumnClass(int columnIndex) {
        if (columnIndex==0)
            return Icon.class;
        else
            return String.class;
    }
    /**
     * Returns the number of columns.
     *
     * @return                              the number of columns
     */
    public int getColumnCount() {
        return 2;
    }
    /**
     * Returns the name of the column.
     *
     * @param columnIndex                   the name of the column
     * @return                              the column name
     */
    public String getColumnName(int columnIndex) {
        LocalizationManager localizationManager=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager();
        switch (columnIndex) {
        case 0:
            return " ";
        case 2:
        default:
            return localizationManager.getPhrase("oimodeler.searchResultName");
        }
    }
    /**
     * Returns the number of rows.
     *
     * @return                              the number of rows
     */
    public int getRowCount() {
        if (m_elements==null)
            return 1;
        else
            return m_elements.size();
    }
    /**
     * Returns the value at given position.
     *
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     * @return                              the value at given row and column
     */
    public Object getValueAt(int rowIndex,int columnIndex) {
        if (m_elements==null) {
            if (columnIndex==1)
                return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("oimodeler.loadingWait");
            else
                return null;
        }
        else {
            TableRow tableRow=(TableRow)m_elements.get(rowIndex);
            switch (columnIndex) {
            case 0:
                if (tableRow.m_entity instanceof Concept)
                    return m_iconConcept;
                else if (tableRow.m_entity instanceof Property)
                    return m_iconProperty;
                else
                    return m_iconInstance;
            case 1:
            default:
                return tableRow.m_value;
            }
        }
    }
    /**
     * Checks if the cell is editable.
     *
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     * @return                              <code>true</code> if the cell at given row and column is editable
     */
    public boolean isCellEditable(int rowIndex,int columnIndex) {
        return false;
    }
    /**
     * Sets the value at given position.
     *
     * @param aValue                        the value
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     */
    public void setValueAt(Object aValue,int rowIndex,int columnIndex) {
    }
    /**
     * Called when a bulk of change events is processed in the pool.
     *
     * @param oimodel                   OI-model that was changed
     * @param changeEvents              list of change events that occured
     */
    public void modelChanged(OIModel oimodel,List changeEvents) {
        if (m_elements!=null)
            try {
                Iterator iterator=changeEvents.iterator();
                while (iterator.hasNext()) {
                    ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                    changeEvent.accept(m_changeVisitor);
                }
                Set instancesWithChangedLexicalEntries=LexiconUtil.getInstancesWithChangedLexicalEntries(changeEvents);
                for (int i=m_elements.size()-1;i>=0;i--) {
                    TableRow tableRow=(TableRow)m_elements.get(i);
                    if (instancesWithChangedLexicalEntries.contains(tableRow.m_entity.getSpanningInstance())) {
                        tableRow.update();
                        fireTableRowsUpdated(i,i);
                    }
                }
            }
            catch (KAONException e) {
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
            }
    }
    /**
     * Called when model is refreshed.
     *
     * @param oimodel                   OI-model that was refreshed
     */
    public void modelRefreshed(OIModel oimodel) {
        showNothing();
    }
    /**
     * Called when model is being deleted.
     *
     * @param oimodel                   OI-model that is being deleted
     */
    public void modelDeleted(OIModel oimodel) {
    }
    /**
     * Returns the entity in given row.
     *
     * @param rowIndex                  the index of the row
     * @return                          the entity in given row
     */
    public Entity getEntity(int rowIndex) {
        if (m_elements!=null)
            return ((TableRow)m_elements.get(rowIndex)).m_entity;
        else
            return null;
    }
    /**
     * Returns the row for given entity.
     *
     * @param entity                    the entity
     * @return                          the row of given entity
     */
    public int getRowForEntity(Entity entity) {
        if (m_elements!=null)
            for (int i=m_elements.size()-1;i>=0;i--) {
                TableRow tableRow=(TableRow)m_elements.get(i);
                if (entity.equals(tableRow.m_entity))
                    return i;
            }
        return -1;
    }
    /**
     * Clears the model.
     */
    public void clearEntities() {
        m_elements=new ArrayList();
        fireTableDataChanged();
    }
    /**
     * Adds the collection of entities to the model.
     *
     * @param entities                  the entities
     * @throws KAONException            thrown if there is an error
     */
    public void addEntities(Collection entities) throws KAONException {
        boolean changed=false;
        Iterator iterator=entities.iterator();
        while (iterator.hasNext()) {
            Object object=iterator.next();
            if (object instanceof Entity) {
                Entity entity=(Entity)object;
                if (getRowForEntity(entity)==-1) {
                    m_elements.add(new TableRow(entity));
                    changed=true;
                }
            }
        }
        if (changed) {
            Collections.sort(m_elements,RowComparator.INSTANCE);
            fireTableDataChanged();
        }
    }
    /**
     * Removes specified entities from the model.
     *
     * @param entities                  the entities
     */
    public void removeEntities(Collection entities) {
        Iterator iterator=entities.iterator();
        while (iterator.hasNext()) {
            Object object=iterator.next();
            if (object instanceof Entity) {
                Entity entity=(Entity)object;
                int row=getRowForEntity(entity);
                if (row!=-1) {
                    m_elements.remove(row);
                    fireTableRowsDeleted(row,row);
                }
            }
        }
    }
    /**
     * Updates the language in the model.
     *
     * @param languageURI               the language URI
     * @throws KAONException            thrown if there is an error
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        m_languageURI=languageURI;
        Iterator iterator=m_elements.iterator();
        while (iterator.hasNext()) {
            TableRow tableRow=(TableRow)iterator.next();
            tableRow.update();
        }
        fireTableDataChanged();
    }

    /**
     * The row in the table.
     */
    public class TableRow {
        protected Entity m_entity;
        protected String m_value;

        public TableRow(Entity entity) throws KAONException {
            m_entity=entity;
            update();
        }
        public void update() throws KAONException {
            m_value=m_entity.getLabel(m_languageURI);
            if (m_value==null)
                m_value=m_entity.getURI();
        }
    }

    /**
     * The comparator for the rows.
     */
    protected static class RowComparator implements Comparator {
        public static final Comparator INSTANCE=new RowComparator();
        public int compare(Object o1,Object o2) {
            TableRow r1=(TableRow)o1;
            TableRow r2=(TableRow)o2;
            int compare=r1.m_value.compareToIgnoreCase(r2.m_value);
            if (compare!=0)
                return compare;
            int index1=getIndex(r1.m_entity);
            int index2=getIndex(r2.m_entity);
            if (index1<index2)
                return -1;
            else if (index1==index2)
                return 0;
            else
                return 1;
        }
        protected int getIndex(Entity entity) {
            if (entity instanceof Concept)
                return 0;
            else if (entity instanceof Property)
                return 1;
            else
                return 2;
        }
    }

    /**
     * The visitor for OI-model events.
     */
    protected class OIModelChangeVisitor extends NullChangeEventVisitor {
        public void visit(RemoveEntity event) throws KAONException {
            String entityURI=event.getEntity().getURI();
            for (int i=m_elements.size()-1;i>=0;i--) {
                TableRow tableRow=(TableRow)m_elements.get(i);
                if (entityURI.equals(tableRow.m_entity.getURI())) {
                    m_elements.remove(i);
                    fireTableRowsDeleted(i,i);
                }
            }
        }
    }
}
