package de.fzi.wim.oimodeler.entityhierarchy;

import java.util.Set;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import javax.swing.tree.TreeNode;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

import de.fzi.wim.guibase.lazytreemodel.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.viewfilter.*;

/**
 * Node loader for parent concept hierarchy.
 */
public class TransitivePropertiesHierarchyNodeLoader extends OIModelNodeLoader {
    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable     the OI-modeler viewable
     */
    public TransitivePropertiesHierarchyNodeLoader(OIModelerViewable oimodelerViewable) {
        super(oimodelerViewable);
    }
    /**
     * Returns the position at which the supplied node should be inserted.
     *
     * @param node                  the node in whose children insertion is done
     * @param newNode               the new node
     * @return                      the position of the new node
     */
    public int getInsertPosition(OIModelNode node,OIModelNode newNode) {
        TreeNode[] children=node.getChildren();
        for (int i=0;i<children.length;i++) {
            int result=NodeLabelComparator.INSTANCE.compare(children[i],newNode);
            if (result>=0)
                return i;
        }
        return children.length;
    }
    /**
     * Searches the branch upwards to find anode containing a property. This property is returend.
     *
     * @param node                  the node with which to start the search
     * @return                      found property
     */
    private Property propertyFromNode(OIModelNode node) {
        Property property=null;
        OIModelNode parent=(OIModelNode)node.getParent();
        while ((parent!=null)&&(property==null)) {
            if (parent instanceof EntityHierarchyNode)
                if (((EntityHierarchyNode)parent).getEntity() instanceof Property)
                    property=(Property)((EntityHierarchyNode)parent).getEntity();
            parent=(OIModelNode)parent.getParent();
        }
        return property;
    }

    /**
     * Examines the given event and returns the new node created as the result of the event
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the new child node (or <code>null</code>)
     */
    public OIModelNode getAddedChildNode(OIModelNode node,ChangeEvent changeEvent) {
        try {
            Entity entity=((EntityHierarchyNode)node).getEntity();
            Entity childEntity=null;

            if ((entity instanceof Instance) && (changeEvent instanceof SetPropertyTransitive) && node.getParent()==null) {
                SetPropertyTransitive setPropertyTransitive=(SetPropertyTransitive)changeEvent;
                if (setPropertyTransitive.getOldTransitive()==false && setPropertyTransitive.getTransitive()==true) {
                    Set values=((Instance)entity).getFromPropertyValues(setPropertyTransitive.getProperty());
                    if (!values.isEmpty())
                        childEntity=setPropertyTransitive.getProperty();
                }
            }
            else if ((entity instanceof Instance) && (changeEvent instanceof SetPropertyInstanceValue) && node.getParent()!=null) {
                // only instances not being the root of the tree have instances as children
                // for instances other than root find the transitive property being expanded in this branch of the tree
                Property property=propertyFromNode(node);
                if (property!=null) {
                    // now having found the property we need to check if the changed property instance matches with the given node
                    SetPropertyInstanceValue setPropertyInstanceValue=(SetPropertyInstanceValue)changeEvent;
                    PropertyInstance changedpropertyinstance=setPropertyInstanceValue.getPropertyInstance();
                    if (property.equals(changedpropertyinstance.getProperty())) {
                        if (entity.equals(changedpropertyinstance.getSourceInstance())) {
                            Object targetvalue=changedpropertyinstance.getTargetValue();
                            if (targetvalue instanceof Instance)
                                childEntity=(Entity)targetvalue;
                        }
                    }
                }
            }
            else if ((entity instanceof Instance) && (changeEvent instanceof AddPropertyInstance) && node.getParent()!=null) {
                // only instances not being the root of the tree have instances as children
                // for instances other than root find the transitive property being expanded in this branch of the tree
                Property property=propertyFromNode(node);
                if (property!=null) {
                    // now having found the property we need to check if the changed property instance matches with the given node
                    AddPropertyInstance addPropertyInstance=(AddPropertyInstance)changeEvent;
                    PropertyInstance addedpropertyinstance=addPropertyInstance.getPropertyInstance();
                    if (property.equals(addedpropertyinstance.getProperty())) {
                        if (entity.equals(addedpropertyinstance.getSourceInstance())) {
                            Object targetvalue=addedpropertyinstance.getTargetValue();
                            if (targetvalue instanceof Instance)
                                childEntity=(Entity)targetvalue;
                        }
                    }
                }
            }
            else if ((entity instanceof Instance) && (changeEvent instanceof AddPropertyInstance) && node.getParent()==null) {
                // eventually add a property
                AddPropertyInstance addPropertyInstance=(AddPropertyInstance)changeEvent;
                PropertyInstance addedpropertyinstance=addPropertyInstance.getPropertyInstance();
                Property property=addedpropertyinstance.getProperty();
                if (property.isTransitive()) {
                    Set values=((Instance)entity).getFromPropertyValues(property);
                    if (values.size()==1) {
                        childEntity=property;
                    }
                }
            }
            else if ((entity instanceof Property) && (changeEvent instanceof AddPropertyInstance) ) {
                if (((Property)entity).isTransitive()) {
                    // first we need to find the instance owning this property (should be the root of the tree)
                    EntityHierarchyNode parent=(EntityHierarchyNode)node.getParent();
                    Entity parententity=parent.getEntity();
                    if (parententity instanceof Instance && parent.getParent()==null) {
                        AddPropertyInstance addPropertyInstance=(AddPropertyInstance)changeEvent;
                        PropertyInstance addedpropertyinstance=addPropertyInstance.getPropertyInstance();
                        if (((Property)entity).equals(addedpropertyinstance.getProperty())) {
                            if (parententity.equals(addedpropertyinstance.getSourceInstance())) {
                                Object targetvalue=addedpropertyinstance.getTargetValue();
                                if (targetvalue instanceof Instance)
                                    childEntity=(Entity)targetvalue;
                            }
                        }
                    }
                }
            }
            else if ((entity instanceof Property) && (changeEvent instanceof SetPropertyInstanceValue)) {
                if (((Property)entity).isTransitive()) {
                    // first we need to find the instance owning this property (should be the parent in the tree)
                    EntityHierarchyNode parent=(EntityHierarchyNode)node.getParent();
                    Entity parententity=parent.getEntity();
                    if (parententity instanceof Instance && parent.getParent()==null) {
                        SetPropertyInstanceValue setPropertyInstanceValue=(SetPropertyInstanceValue)changeEvent;
                        PropertyInstance changedpropertyinstance=setPropertyInstanceValue.getPropertyInstance();
                        if (((Property)entity).equals(changedpropertyinstance.getProperty())) {
                            if (parententity.equals(changedpropertyinstance.getSourceInstance())) {
                                Object targetvalue=changedpropertyinstance.getTargetValue();
                                if (targetvalue instanceof Instance)
                                    childEntity=(Entity)targetvalue;
                            }
                        }
                    }
                }
            }

            if (childEntity!=null && m_oimodelerViewable.getViewFilter().showEntity(childEntity))
                return new EntityHierarchyNode(node,childEntity);
        }
        catch (KAONException ignored) {
        }
        return null;
    }
    /**
     * Returns the index of the child containing given entity.
     *
     * @param node                  the node
     * @param childEntity           the child entity
     * @return                      the index of the child (or -1)
     */
    protected int getChildEntityIndex(OIModelNode node,Entity childEntity) {
        TreeNode[] children=node.getChildren();
        for (int i=0;i<children.length;i++) {
            EntityHierarchyNode childNode=(EntityHierarchyNode)children[i];
            if (childNode.getEntity().equals(childEntity))
                return i;
        }
        return -1;
    }
    /**
     * Examines the given event and returns the index of the child that must be removed for the event.
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the index of the child that must be removed for the event (or -1)
     */
    public int getRemovedChildIndex(OIModelNode node,ChangeEvent changeEvent) {
        Entity entity=((EntityHierarchyNode)node).getEntity();

        try {

            if ((entity instanceof Instance) && (changeEvent instanceof SetPropertyTransitive) && node.getParent()==null) {
                // check if a property that formerly has been transitive been set to not being transitive anymore
                SetPropertyTransitive setPropertyTransitive=(SetPropertyTransitive)changeEvent;
                if (setPropertyTransitive.getOldTransitive()==true && setPropertyTransitive.getTransitive()==false) {
                    Set values=((Instance)entity).getFromPropertyValues(setPropertyTransitive.getProperty());
                    // remove the property from the tree if there are any instances of this property going out from the instance
                    if (!values.isEmpty())
                        return getChildEntityIndex(node,(Entity)setPropertyTransitive.getProperty());
                }
            }
            else if ((entity instanceof Instance) && (changeEvent instanceof SetPropertyInstanceValue) && node.getParent()!=null) {
                // only instances not being the root of the tree have instances as children
                // for instances other than root check if the given instance is source of the transitive property
                Property property=propertyFromNode(node);
                if (property!=null) {
                    // now having found the property we need to check if the changed property instance matches with the given node
                    SetPropertyInstanceValue setPropertyInstanceValue=(SetPropertyInstanceValue)changeEvent;
                    PropertyInstance changedpropertyinstance=setPropertyInstanceValue.getPropertyInstance();
                    if (property.equals(changedpropertyinstance.getProperty())) {
                        if (entity.equals(changedpropertyinstance.getSourceInstance())) {
                            Object targetvalue=changedpropertyinstance.getTargetValue();
                            if (targetvalue instanceof Instance)
                                return getChildEntityIndex(node,(Entity)targetvalue);
                        }
                    }
                }
            }
            else if ((entity instanceof Instance) && (changeEvent instanceof RemovePropertyInstance) && node.getParent()!=null) {
                // only instances not being the root of the tree have instances as children
                // for instances other than root check if the given instance is source of the transitive property
                Property property=propertyFromNode(node);
                if (property!=null) {
                    // now having found the property we need to check if the changed property instance matches with the given node
                    RemovePropertyInstance removePropertyInstanceValue=(RemovePropertyInstance)changeEvent;
                    PropertyInstance removedpropertyinstance=removePropertyInstanceValue.getPropertyInstance();
                    if (property.equals(removedpropertyinstance.getProperty())) {
                        if (entity.equals(removedpropertyinstance.getSourceInstance())) {
                            Object targetvalue=removedpropertyinstance.getTargetValue();
                            if (targetvalue instanceof Instance)
                                return getChildEntityIndex(node,(Entity)targetvalue);
                        }
                    }
                }
            }
            else if ((entity instanceof Instance) && (changeEvent instanceof RemovePropertyInstance) && node.getParent()==null) {
                // eventually a property node needs to be removed
                RemovePropertyInstance removePropertyInstanceValue=(RemovePropertyInstance)changeEvent;
                PropertyInstance removedpropertyinstance=removePropertyInstanceValue.getPropertyInstance();
                Property property=removedpropertyinstance.getProperty();
                if (property.isTransitive()) {
                    Set values=((Instance)entity).getFromPropertyValues(property);
                    if (values.size()==0) {
                        return getChildEntityIndex(node,property);
                    }
                }
            }
            else if ((entity instanceof Property) && (changeEvent instanceof RemovePropertyInstance) ) {
                if (((Property)entity).isTransitive()) {
                    // first we need to find the instance owning this property (should be the root of the tree)
                    EntityHierarchyNode parent=(EntityHierarchyNode)node.getParent();
                    Entity parententity=parent.getEntity();
                    if (parententity instanceof Instance && parent.getParent()==null) {
                        RemovePropertyInstance removePropertyInstance=(RemovePropertyInstance)changeEvent;
                        PropertyInstance removedpropertyinstance=removePropertyInstance.getPropertyInstance();
                        if (((Property)entity).equals(removedpropertyinstance.getProperty())) {
                            if (parententity.equals(removedpropertyinstance.getSourceInstance())) {
                                Object targetvalue=removedpropertyinstance.getTargetValue();
                                if (targetvalue instanceof Instance)
                                    return getChildEntityIndex(node,(Entity)targetvalue);
                            }
                        }
                    }
                }
            }
            else if ((entity instanceof Property) && (changeEvent instanceof SetPropertyInstanceValue)) {
                if (((Property)entity).isTransitive()) {
                    // first we need to find the instance owning this property (should be the parent in the tree)
                    EntityHierarchyNode parent=(EntityHierarchyNode)node.getParent();
                    Entity parententity=parent.getEntity();
                    if (parententity instanceof Instance && parent.getParent()==null) {
                        SetPropertyInstanceValue setPropertyInstanceValue=(SetPropertyInstanceValue)changeEvent;
                        PropertyInstance changedpropertyinstance=setPropertyInstanceValue.getPropertyInstance();
                        if (((Property)entity).equals(changedpropertyinstance.getProperty())) {
                            if (parententity.equals(changedpropertyinstance.getSourceInstance())) {
                                Object targetvalue=changedpropertyinstance.getTargetValue();
                                if (targetvalue instanceof Instance)
                                    return getChildEntityIndex(node,(Entity)targetvalue);
                            }
                        }
                    }
                }
            }
        }
        catch (KAONException ignored) {
        }

        return -1;
    }
    /**
     * Examines the given event and returns the index of the child that was changed by the event.
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the index of the child that was changed by the event (or -1)
     */
    public int getChangedChildIndex(OIModelNode node,ChangeEvent changeEvent) {
        return -1;
    }
    /**
     * Examines the given event and returns <code>true</code> if the lexicon of the node changed.
     *
     * @param node                                  the node
     * @param instancesWithChangedLexicalEntries    the set of instances whose lexical entries changed
     * @return                                      <code>true</code> if the lexicon of the node changed
     */
    public boolean getLexiconChanged(OIModelNode node,Set instancesWithChangedLexicalEntries) {
        try {
            EntityHierarchyNode entityHierarchyNode=(EntityHierarchyNode)node;
            return instancesWithChangedLexicalEntries.contains(entityHierarchyNode.getEntity().getSpanningInstance());
        }
        catch (KAONException e) {
            return false;
        }
    }
    /**
     * Returns the event for associating nodes. If entities cannot be associated, <code>null</code>
     * is returned.
     *
     * @param superNode             the node
     * @param object                the object
     * @return                      the event for associating nodes
     */
    public ChangeEvent getAssociationEvent(OIModelNode superNode,Object object) {
        try {
            if (superNode.getParent()!=null && object instanceof Instance) {
                // only do something if it's not the root node having been hit
                Entity subEntity=(Entity)object;
                Entity superEntity=((EntityHierarchyNode)superNode).getEntity();
                if (!superEntity.equals(subEntity) && m_oimodelerViewable.getViewFilter().showEntity(subEntity)) {
                    if (superEntity instanceof Instance) {
                        // make the sub entity the target, super entity the source of a property instance being expanded in the current branch of the tree
                        // find the property first
                        Property property=propertyFromNode(superNode);
                        if (property!=null) {
                            return new AddPropertyInstance(m_oimodelerViewable.getActiveOIModel(),null,property,(Instance)superEntity,subEntity);
                        }
                    }
                    else if (superEntity instanceof Property) {
                        // make the sub entity the target, the instance of the root node the source of a property instance of type of the super entity property
                        Instance rootInstance=(Instance)((EntityHierarchyNode)superNode.getParent()).getEntity();
                        return new AddPropertyInstance(m_oimodelerViewable.getActiveOIModel(),null,(Property)superEntity,rootInstance,subEntity);
                    }
                }
            }
        }
        catch (KAONException ignored) {
        }
        return null;
    }
    /**
     * Loads children of ontology nodes.
     *
     * @param model                 model that requested the load
     * @param node                  node whose children need to be loaded
     * @return                      array of loaded children (or {@link #LOAD_DEFERRED} indicating that load will be done later)
     * @throws KAONException        thrown if there is a problem
     */
    protected TreeNode[] loadNodeChildrenEx(LazyTreeModel model,LazyTreeNode node) throws KAONException {
        Entity entity=((EntityHierarchyNode)node).getEntity();
        List children=new ArrayList();
        ViewFilter viewFilter=m_oimodelerViewable.getViewFilter();
        OIModelTreeModel oimodelTreeModel=(OIModelTreeModel)model;
        if (entity instanceof Instance) {
            // we have an instance entity
            if (node.getParent()==null) {
                // since it is the root instance, add all transitive properties to tree
                Iterator elements=((Instance)entity).getFromPropertyInstances().iterator();
                while (elements.hasNext()) {
                    PropertyInstance propertyInstance=(PropertyInstance)elements.next();
                    Property property=propertyInstance.getProperty();
                    if (property.isTransitive() && viewFilter.showEntity(property) && !children.contains(property))
                        children.add(property);
                }
            }
            else {
                // for instances other than root find the transitive property being expanded in this branch of the tree
                Property property=propertyFromNode((OIModelNode)node);
                if (property!=null) {
                    //  property has been found -> get all transitive instances
                    Iterator elements=((Instance)entity).getFromPropertyInstances().iterator();
                    while (elements.hasNext()) {
                        PropertyInstance propertyInstance=(PropertyInstance)elements.next();
                        if (property.equals(propertyInstance.getProperty())) {
                            Object targetValue=propertyInstance.getTargetValue();
                            if (targetValue instanceof Instance && viewFilter.showEntity((Entity)targetValue))
                                children.add(targetValue);
                        }
                    }
                }
            }
        }
        else if (entity instanceof Property) {
            // first we need to find the instance owning this property (should be the parent in the tree)
            EntityHierarchyNode parent=(EntityHierarchyNode)node.getParent();
            Instance parentInstance=(Instance)parent.getEntity();
            // get all property instances owned by this instance
            Iterator elements=parentInstance.getFromPropertyInstances().iterator();
            while (elements.hasNext()) {
                PropertyInstance propertyInstance=(PropertyInstance)elements.next();
                if (((Property)entity).equals(propertyInstance.getProperty())) {
                    Object targetValue=propertyInstance.getTargetValue();
                    if (targetValue instanceof Instance && viewFilter.showEntity((Entity)targetValue))
                        children.add(targetValue);
                }
            }
        }
        entity.getOIModel().loadObjects(children,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_INSTANCE_BASICS | OIModel.LOAD_LEXICON);
        TreeNode[] childrenAsArray=new TreeNode[children.size()];
        for (int i=0;i<children.size();i++) {
            EntityHierarchyNode newNode=new EntityHierarchyNode(node,(Entity)children.get(i));
            newNode.setLanguageURI(oimodelTreeModel.getLanguageURI());
            childrenAsArray[i]=newNode;
        }
        Arrays.sort(childrenAsArray,NodeLabelComparator.INSTANCE);
        return childrenAsArray;
    }

    /**
     * The comparator for the nodes.
     */
    protected static class NodeLabelComparator implements Comparator {
        public static final Comparator INSTANCE=new NodeLabelComparator();

        public int compare(Object o1,Object o2) {
            OIModelNode n1=(OIModelNode)o1;
            OIModelNode n2=(OIModelNode)o2;
            String label1=n1.toString();
            String label2=n2.toString();
            return label1.compareToIgnoreCase(label2);
        }
    }
}
