package de.fzi.wim.oimodeler.entityhierarchy;

import java.util.List;
import java.util.Set;
import java.util.Collections;
import javax.swing.tree.TreeNode;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

import de.fzi.wim.guibase.lazytreemodel.*;

import de.fzi.wim.oimodeler.ui.*;

/**
 * Tree model showing the OI-model elements.
 */
public class OIModelTreeModel extends LazyTreeModel implements OIModelListener {
    /** OI-model being wrapped. */
    protected OIModel m_oimodel;
    /** The language URI. */
    protected String m_languageURI;

    /**
     * Creates a tree model adapter.
     *
     * @param oimodelerViewable     the OI-modeler viewable
     * @param loader                the loader for nodes
     */
    public OIModelTreeModel(OIModelerViewable oimodelerViewable,OIModelNodeLoader loader) {
        super(null,loader,new DefaultNoChildrenNode(oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("oimodeler.loadingWait"),null));
        m_languageURI=oimodelerViewable.getLanguageURI();
        m_oimodel=oimodelerViewable.getOIModel();
        setAsksAllowsChildren(true);
        m_oimodel.addOIModelListener(this);
    }
    /**
     * Destroys this tree model.
     */
    public void dispose() {
        showNothing();
        if (m_oimodel!=null) {
            m_oimodel.removeOIModelListener(this);
            m_oimodel=null;
        }
    }
    /**
     * Returns the OI-model.
     *
     * @return                      the OI-model
     */
    public OIModel getOIModel() {
        return m_oimodel;
    }
    /**
     * Sets the current language URI.
     *
     * @param languageURI           the languageURI
     * @throws KAONException        thrown if there is an error
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        m_languageURI=languageURI;
        if (getRoot()!=null)
            setLanguageURI((OIModelNode)getRoot());
    }
    /**
     * Updates the node's language URI.
     *
     * @param node                  the node
     * @throws KAONException        thrown if there is an error
     */
    protected void setLanguageURI(OIModelNode node) throws KAONException {
        node.setLanguageURI(m_languageURI);
        fireTreeNodesChanged(this,getParentPath(node),null,null);
        if (node.getNodeState()==OIModelNode.LOADED) {
            TreeNode[] children=node.getChildren();
            for (int i=0;i<children.length;i++)
                setLanguageURI((OIModelNode)children[i]);
        }
    }
    /**
     * returns the current language URI.
     *
     * @return                      the current languageURI
     */
    public String getLanguageURI() {
        return m_languageURI;
    }
    /**
     * Returns the current entity.
     *
     * @return                      the entity shown
     */
    public Entity getCurrentEntity() {
        if (getRoot()==null)
            return null;
        else
            return ((EntityHierarchyNode)getRoot()).getEntity();
    }
    /**
     * Makes sure nothing is shown in the model.
     */
    public void showNothing() {
        setRoot(null);
    }
    /**
     * Shows given entity in the model.
     *
     * @param entity                the entity shown
     * @throws KAONException        thrown if the entity cannot be shown
     */
    public void showEntity(Entity entity) throws KAONException {
        showNode(new EntityHierarchyNode(null,entity));
    }
    /**
     * Shows given node as the root.
     *
     * @param root                  the node to show in the root
     * @throws KAONException        thrown if the entity cannot be shown
     */
    public void showNode(OIModelNode root) throws KAONException {
        root.setLanguageURI(m_languageURI);
        setRoot(root);
    }
    /**
     * Returns an array of objects representing a path to given node.
     *
     * @param node                  the node for which the parent path was requested
     * @return                      array of objects representing a path to given node
     */
    public TreeNode[] getParentPath(TreeNode node) {
        int pathLength=0;
        TreeNode current=node;
        while (current!=null) {
            current=current.getParent();
            pathLength++;
        }
        TreeNode[] result=new TreeNode[pathLength];
        current=node;
        while (current!=null) {
            result[--pathLength]=current;
            current=current.getParent();
        }
        return result;
    }
    /**
     * Called when a of change events is processed in the pool.
     *
     * @param oimodel                   OI-model that was changed
     * @param changeEvents              list of change events that occured
     */
    public void modelChanged(OIModel oimodel,List changeEvents) {
        if (getRoot()!=null) {
            Set instancesWithChangedLexicalEntries;
            try {
                instancesWithChangedLexicalEntries=LexiconUtil.getInstancesWithChangedLexicalEntries(changeEvents);
            }
            catch (KAONException ignored) {
                instancesWithChangedLexicalEntries=Collections.EMPTY_SET;
            }
            ChangeEvent[] changeEventsArray=new ChangeEvent[changeEvents.size()];
            changeEvents.toArray(changeEventsArray);
            checkSubTreeForEvent((OIModelNode)getRoot(),changeEventsArray,instancesWithChangedLexicalEntries);
        }
    }
    /**
     * Checks supplied node and all its children.
     *
     * @param node                                      node being checked
     * @param changeEvents                              array of change events
     * @param instancesWithChangedLexicalEntries        the set of instances whose lexical entries have changed
     */
    protected void checkSubTreeForEvent(OIModelNode node,ChangeEvent[] changeEvents,Set instancesWithChangedLexicalEntries) {
        OIModelNodeLoader loader=(OIModelNodeLoader)getLazyNodeLoader(node);
        TreeNode[] nodePath=null;
        if (loader.getLexiconChanged(node,instancesWithChangedLexicalEntries)) {
            if (nodePath==null)
                nodePath=getParentPath(node);
            try {
                node.setLanguageURI(m_languageURI);
            }
            catch (KAONException ignored) {
            }
            fireTreeNodesChanged(this,nodePath,null,null);
        }
        for (int i=0;i<changeEvents.length;i++) {
            if (node.getNodeState()==OIModelNode.LOADED) {
                OIModelNode newNode=loader.getAddedChildNode(node,changeEvents[i]);
                if (newNode!=null) {
                    if (nodePath==null)
                        nodePath=getParentPath(node);
                    try {
                        newNode.setLanguageURI(m_languageURI);
                    }
                    catch (KAONException ignored) {
                    }
                    int insertPosition=loader.getInsertPosition(node,newNode);
                    TreeNode[] nodeChildren=node.getChildren();
                    int numberOfChildren=nodeChildren.length;
                    TreeNode[] newChildren=new TreeNode[numberOfChildren+1];
                    if (insertPosition>0)
                        System.arraycopy(nodeChildren,0,newChildren,0,insertPosition);
                    newChildren[insertPosition]=newNode;
                    if (insertPosition<numberOfChildren)
                        System.arraycopy(nodeChildren,insertPosition,newChildren,insertPosition+1,numberOfChildren-insertPosition);
                    nodeChildren=newChildren;
                    node.setChildrenLoaded(nodeChildren);
                    fireTreeNodesInserted(this,nodePath,new int[] { insertPosition },new Object[] { newNode });
                }
                int removedIndex=loader.getRemovedChildIndex(node,changeEvents[i]);
                if (removedIndex!=-1) {
                    if (nodePath==null)
                        nodePath=getParentPath(node);
                    TreeNode[] nodeChildren=node.getChildren();
                    OIModelNode removedNode=(OIModelNode)nodeChildren[removedIndex];
                    int numberOfChildren=nodeChildren.length;
                    TreeNode[] newChildren=new TreeNode[numberOfChildren-1];
                    if (removedIndex>0)
                        System.arraycopy(nodeChildren,0,newChildren,0,removedIndex);
                    if (removedIndex<numberOfChildren)
                        System.arraycopy(nodeChildren,removedIndex+1,newChildren,removedIndex,numberOfChildren-removedIndex-1);
                    nodeChildren=newChildren;
                    node.setChildrenLoaded(nodeChildren);
                    fireTreeNodesRemoved(this,nodePath,new int[] { removedIndex },new Object[] { removedNode });
                }
                int changedIndex=loader.getChangedChildIndex(node,changeEvents[i]);
                if (changedIndex!=-1) {
                    if (nodePath==null)
                        nodePath=getParentPath(node);
                    TreeNode[] nodeChildren=node.getChildren();
                    OIModelNode changedNode=(OIModelNode)nodeChildren[changedIndex];
                    try {
                        changedNode.setLanguageURI(m_languageURI);
                    }
                    catch (KAONException ignored) {
                    }
                    fireTreeNodesChanged(this,nodePath,new int[] { changedIndex },new Object[] { changedNode });
                }
            }
        }
        if (node.getNodeState()==OIModelNode.LOADED) {
            TreeNode[] nodeChildren=node.getChildren();
            int childCount=nodeChildren.length;
            for (int i=0;i<childCount;i++)
                checkSubTreeForEvent((OIModelNode)nodeChildren[i],changeEvents,instancesWithChangedLexicalEntries);
        }
    }
    /**
     * Called when model is refreshed.
     *
     * @param oimodel               the OI-model that was refreshed
     */
    public void modelRefreshed(OIModel oimodel) {
        refresh();
    }
    /**
     * Called when model is being deleted.
     *
     * @param oimodel                   OI-model that is being deleted
     */
    public void modelDeleted(OIModel oimodel) {
    }
    /**
     * Refreshes this model.
     */
    protected void refresh() {
        OIModelNode root=(OIModelNode)getRoot();
        if (root!=null) {
            root.unloadChildren();
            fireTreeStructureChanged(this,new Object[] { root },null,null);
        }
    }
}
