package de.fzi.wim.oimodeler.entityhierarchy;

import java.util.Set;
import javax.swing.tree.TreeNode;
import javax.swing.SwingUtilities;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

import de.fzi.wim.guibase.lazytreemodel.*;

import de.fzi.wim.oimodeler.ui.*;

/**
 * Node loader for the some tree in the OI-model.
 */
public abstract class OIModelNodeLoader implements LazyNodeLoader {
    /** The OI-modeler viewable. */
    protected OIModelerViewable m_oimodelerViewable;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable     the OI-modeler viewable
     */
    public OIModelNodeLoader(OIModelerViewable oimodelerViewable) {
        m_oimodelerViewable=oimodelerViewable;
    }
    /**
     * Loads children of ontology nodes.
     *
     * @param model                 model that requested the load
     * @param node                  node whose children need to be loaded
     * @return                      array of loaded children (or {@link #LOAD_DEFERRED} indicating that load will be done later)
     */
    public TreeNode[] loadNodeChildren(LazyTreeModel model,LazyTreeNode node) {
        try {
            OIModelTreeModel oimodelTreeModel=(OIModelTreeModel)model;
            if ((oimodelTreeModel.getOIModel().getCapabilities() & OIModel.CAPABILITY_SUPPORTS_OPTIMIZED_LOADING)!=0) {
                loadNodeChildrenAsync(model,node);
                return LOAD_DEFERRED;
            }
            else
                return loadNodeChildrenEx(model,node);
        }
        catch (InterruptedException e) {
            if (node instanceof EntityHierarchyNode)
                m_oimodelerViewable.errorLoadingEntity(((EntityHierarchyNode)node).getEntity(),e);
            else
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
            return LOAD_CANCELED;
        }
        catch (KAONException e) {
            if (node instanceof EntityHierarchyNode)
                m_oimodelerViewable.errorLoadingEntity(((EntityHierarchyNode)node).getEntity(),e);
            else
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
            return LOAD_CANCELED;
        }
    }
    /**
     * Loads children of ontology nodes asynchronously.
     *
     * @param model                 model that requested the load
     * @param node                  node whose children need to be loaded
     * @throws InterruptedException thrown if loading cannot be performed
     */
    protected void loadNodeChildrenAsync(final LazyTreeModel model,final LazyTreeNode node) throws InterruptedException {
        m_oimodelerViewable.getModule().getAppDriver().getThreadPool().executeTask(new Runnable() {
            public void run() {
                try {
                    OIModelTreeModel oimodelTreeModel=(OIModelTreeModel)model;
                    final TreeNode[] treeNodes;
                    synchronized (oimodelTreeModel.getOIModel().getKAONConnection()) {
                        treeNodes=loadNodeChildrenEx(model,node);
                    }
                    SwingUtilities.invokeLater(new Runnable() {
                        public void run() {
                            model.notifyChildrenLoaded(node,treeNodes);
                        }
                    });
                }
                catch (final KAONException e) {
                    SwingUtilities.invokeLater(new Runnable() {
                        public void run() {
                            model.notifyChildrenLoaded(node,LOAD_CANCELED);
                            if (node instanceof EntityHierarchyNode)
                                m_oimodelerViewable.errorLoadingEntity(((EntityHierarchyNode)node).getEntity(),e);
                            else
                                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
                        }
                    });
                }
            }
        });
    }
    /**
     * Loads children of ontology nodes, but throws exceptions if there is a problem.
     *
     * @param model                 model that requested the load
     * @param node                  node whose children need to be loaded
     * @return                      array of loaded children (or {@link #LOAD_DEFERRED} indicating that load will be done later)
     * @throws KAONException        the exception
     */
    protected abstract TreeNode[] loadNodeChildrenEx(LazyTreeModel model,LazyTreeNode node) throws KAONException;
    /**
     * Returns the position at which the supplied node should be inserted.
     *
     * @param node                  the node in whose children insertion is done
     * @param newNode               the new node
     * @return                      the position of the new node
     */
    public abstract int getInsertPosition(OIModelNode node,OIModelNode newNode);
    /**
     * Examines the given event and returns the new node created as the result of the event
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the new child node (or <code>null</code>)
     */
    public abstract OIModelNode getAddedChildNode(OIModelNode node,ChangeEvent changeEvent);
    /**
     * Examines the given event and returns the index of the child that must be removed for the event.
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the index of the child that must be removed for the event (or -1)
     */
    public abstract int getRemovedChildIndex(OIModelNode node,ChangeEvent changeEvent);
    /**
     * Examines the given event and returns the index of the child that was changed by the event.
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the index of the child that was changed by the event (or -1)
     */
    public abstract int getChangedChildIndex(OIModelNode node,ChangeEvent changeEvent);
    /**
     * Examines the given event and returns <code>true</code> if the lexicon of the node changed.
     *
     * @param node                                  the node
     * @param instancesWithChangedLexicalEntries    the set of instances whose lexical entries changed
     * @return                                      <code>true</code> if the lexicon of the node changed
     */
    public abstract boolean getLexiconChanged(OIModelNode node,Set instancesWithChangedLexicalEntries);
    /**
     * Returns the event for associating nodes with objects. If association can't be created, <code>null</code>
     * is returned.
     *
     * @param node                  the node
     * @param object                the object
     * @return                      the event for associating nodes
     */
    public abstract ChangeEvent getAssociationEvent(OIModelNode node,Object object);
}
