package de.fzi.wim.oimodeler.entityhierarchy;

import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.util.Comparator;
import javax.swing.tree.TreeNode;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

import de.fzi.wim.guibase.lazytreemodel.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.viewfilter.*;

/**
 * Node loader for the properties of an instance.
 */
public class InstancePropertiesNodeLoader extends OIModelNodeLoader {

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable     the OI-modeler viewable
     */
    public InstancePropertiesNodeLoader(OIModelerViewable oimodelerViewable) {
        super(oimodelerViewable);
    }
    /**
     * Loads children of ontology nodes, but throws exceptions if there is a problem.
     *
     * @param model                 model that requested the load
     * @param node                  node whose children need to be loaded
     * @return                      array of loaded children (or {@link #LOAD_DEFERRED} indicating that load will be done later)
     * @throws KAONException        the exception
     */
    protected TreeNode[] loadNodeChildrenEx(LazyTreeModel model,LazyTreeNode node) throws KAONException {
        ViewFilter viewFilter=m_oimodelerViewable.getViewFilter();
        OIModelTreeModel oimodelTreeModel=(OIModelTreeModel)model;
        Instance instance=(Instance)((EntityHierarchyNode)node).getEntity();
        OIModel oimodel=instance.getOIModel();
        Set propertyInstances=instance.getFromPropertyInstances();
        Set relatedObjectsToLoad=new HashSet();
        Iterator elements=propertyInstances.iterator();
        while (elements.hasNext()) {
            PropertyInstance propertyInstance=(PropertyInstance)elements.next();
            relatedObjectsToLoad.add(propertyInstance.getProperty());
            Object targetValue=propertyInstance.getTargetValue();
            if (targetValue instanceof Instance)
                relatedObjectsToLoad.add(targetValue);
        }
        oimodel.loadObjects(relatedObjectsToLoad,OIModel.LOAD_INSTANCE_BASICS | OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_LEXICON);
        List children=new ArrayList(propertyInstances.size());
        elements=propertyInstances.iterator();
        while (elements.hasNext()) {
            PropertyInstance propertyInstance=(PropertyInstance)elements.next();
            if (viewFilter.showPropertyInstance(propertyInstance)) {
                PropertyInstanceNode newNode=new PropertyInstanceNode(node,propertyInstance);
                newNode.setChildrenLoaded(PropertyInstanceNode.EMPTY_CHILDREN);
                newNode.setLanguageURI(oimodelTreeModel.getLanguageURI());
                children.add(newNode);
            }
        }
        TreeNode[] childrenAsArray=new TreeNode[children.size()];
        children.toArray(childrenAsArray);
        return childrenAsArray;
    }
    /**
     * Returns the position at which the supplied node should be inserted.
     *
     * @param node                  the node in whose children insertion is done
     * @param newNode               the new node
     * @return                      the position of the new node
     */
    public int getInsertPosition(OIModelNode node,OIModelNode newNode) {
        TreeNode[] children=node.getChildren();
        for (int i=0;i<children.length;i++) {
            int result=PropertyInstanceNodeComparator.INSTANCE.compare(children[i],newNode);
            if (result>=0)
                return i;
        }
        return children.length;
    }
    /**
     * Examines the given event and returns the new node created as the result of the event
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the new child node (or <code>null</code>)
     */
    public OIModelNode getAddedChildNode(OIModelNode node,ChangeEvent changeEvent) {
        try {
            if ((changeEvent instanceof AddPropertyInstance) && (node instanceof EntityHierarchyNode)) {
                EntityHierarchyNode entityHierarchyNode=(EntityHierarchyNode)node;
                AddPropertyInstance addPropertyInstance=(AddPropertyInstance)changeEvent;
                PropertyInstance propertyInstance=addPropertyInstance.getPropertyInstance();
                if (entityHierarchyNode.getEntity().equals(propertyInstance.getSourceInstance()) && m_oimodelerViewable.getViewFilter().showPropertyInstance(propertyInstance))
                    return new PropertyInstanceNode(node,propertyInstance);
            }
        }
        catch (KAONException ignored) {
        }
        return null;
    }
    /**
     * Examines the given event and returns the index of the child that must be removed for the event.
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the index of the child that must be removed for the event (or -1)
     */
    public int getRemovedChildIndex(OIModelNode node,ChangeEvent changeEvent) {
        try {
            if ((changeEvent instanceof RemovePropertyInstance) && (node instanceof EntityHierarchyNode)) {
                EntityHierarchyNode entityHierarchyNode=(EntityHierarchyNode)node;
                RemovePropertyInstance removePropertyInstance=(RemovePropertyInstance)changeEvent;
                PropertyInstance propertyInstance=removePropertyInstance.getPropertyInstance();
                if (entityHierarchyNode.getEntity().equals(propertyInstance.getSourceInstance())) {
                    TreeNode[] children=node.getChildren();
                    for (int i=0;i<children.length;i++) {
                        PropertyInstanceNode childNode=(PropertyInstanceNode)children[i];
                        if (childNode.getPropertyInstance().equals(propertyInstance))
                            return i;
                    }
                }
            }
        }
        catch (KAONException ignored) {
        }
        return -1;
    }
    /**
     * Examines the given event and returns the index of the child that was changed by the event.
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the index of the child that was changed by the event (or -1)
     */
    public int getChangedChildIndex(OIModelNode node,ChangeEvent changeEvent) {
        try {
            if ((changeEvent instanceof ChangePropertyInstanceValue) && (node instanceof EntityHierarchyNode)) {
                EntityHierarchyNode entityHierarchyNode=(EntityHierarchyNode)node;
                ChangePropertyInstanceValue changePropertyInstanceValue=(ChangePropertyInstanceValue)changeEvent;
                PropertyInstance propertyInstance=changePropertyInstanceValue.getPropertyInstance();
                if (entityHierarchyNode.getEntity().equals(propertyInstance.getSourceInstance())) {
                    TreeNode[] children=node.getChildren();
                    for (int i=0;i<children.length;i++) {
                        PropertyInstanceNode childNode=(PropertyInstanceNode)children[i];
                        if (childNode.getPropertyInstance().equals(propertyInstance)) {
                            childNode.setNewTargetValue(changePropertyInstanceValue.getNewTargetValue());
                            return i;
                        }
                    }
                }
            }
        }
        catch (KAONException ignored) {
        }
        return -1;
    }
    /**
     * Examines the given event and returns <code>true</code> if the lexicon of the node changed.
     *
     * @param node                                  the node
     * @param instancesWithChangedLexicalEntries    the set of instances whose lexical entries changed
     * @return                                      <code>true</code> if the lexicon of the node changed
     */
    public boolean getLexiconChanged(OIModelNode node,Set instancesWithChangedLexicalEntries) {
        try {
            EntityHierarchyNode entityHierarchyNode=(EntityHierarchyNode)node;
            return instancesWithChangedLexicalEntries.contains(entityHierarchyNode.getEntity().getSpanningInstance());
        }
        catch (KAONException e) {
            return false;
        }
    }
    /**
     * Returns the event for associating nodes with objects. If association can't be created, <code>null</code>
     * is returned.
     *
     * @param node                  the node
     * @param object                the object
     * @return                      the event for associating nodes
     */
    public ChangeEvent getAssociationEvent(OIModelNode node,Object object) {
        return null;
    }

    /**
     * The comparator for the nodes.
     */
    protected static class PropertyInstanceNodeComparator implements Comparator {
        public static final Comparator INSTANCE=new PropertyInstanceNodeComparator();

        public int compare(Object o1,Object o2) {
            PropertyInstanceNode n1=(PropertyInstanceNode)o1;
            PropertyInstanceNode n2=(PropertyInstanceNode)o2;
            int result=n1.getPropertyLabel().compareToIgnoreCase(n2.getPropertyLabel());
            if (result==0)
                result=n1.getTargetValueLabel().compareToIgnoreCase(n2.getTargetValueLabel());
            return result;
        }
    }
}
