package de.fzi.wim.oimodeler.entityhierarchy;

import java.util.Set;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import javax.swing.tree.TreeNode;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

import de.fzi.wim.guibase.lazytreemodel.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.viewfilter.*;

/**
 * Node loader for the entity hierarchy
 */
public abstract class EntityHierarchyNodeLoader extends OIModelNodeLoader {

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable     the OI-modeler viewable
     */
    public EntityHierarchyNodeLoader(OIModelerViewable oimodelerViewable) {
        super(oimodelerViewable);
    }
    /**
     * Loads children of ontology nodes, but throws exceptions if there is a problem.
     *
     * @param model                 model that requested the load
     * @param node                  node whose children need to be loaded
     * @return                      array of loaded children (or {@link #LOAD_DEFERRED} indicating that load will be done later)
     * @throws KAONException        the exception
     */
    protected TreeNode[] loadNodeChildrenEx(LazyTreeModel model,LazyTreeNode node) throws KAONException {
        ViewFilter viewFilter=m_oimodelerViewable.getViewFilter();
        OIModelTreeModel oimodelTreeModel=(OIModelTreeModel)model;
        Entity entity=((EntityHierarchyNode)node).getEntity();
        OIModel oimodel=entity.getOIModel();
        Set entities=getRelatedEntities(entity);
        oimodel.loadObjects(entities,getRelatedEntitiesLoadFlag());
        List children=new ArrayList(entities.size());
        Iterator elements=entities.iterator();
        while (elements.hasNext()) {
            Entity child=(Entity)elements.next();
            if (viewFilter.showEntity(child)) {
                EntityHierarchyNode newNode=new EntityHierarchyNode(node,child);
                newNode.setLanguageURI(oimodelTreeModel.getLanguageURI());
                children.add(newNode);
            }
        }
        TreeNode[] childrenAsArray=new TreeNode[children.size()];
        children.toArray(childrenAsArray);
        Arrays.sort(childrenAsArray,NodeLabelComparator.INSTANCE);
        return childrenAsArray;
    }
    /**
     * Returns the position at which the supplied node should be inserted.
     *
     * @param node                  the node in whose children insertion is done
     * @param newNode               the new node
     * @return                      the position of the new node
     */
    public int getInsertPosition(OIModelNode node,OIModelNode newNode) {
        TreeNode[] children=node.getChildren();
        for (int i=0;i<children.length;i++) {
            int result=NodeLabelComparator.INSTANCE.compare(children[i],newNode);
            if (result>=0)
                return i;
        }
        return children.length;
    }
    /**
     * Examines the given event and returns the new node created as the result of the event
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the new child node (or <code>null</code>)
     */
    public OIModelNode getAddedChildNode(OIModelNode node,ChangeEvent changeEvent) {
        try {
            Entity childEntity=getAddedChild(((EntityHierarchyNode)node).getEntity(),changeEvent);
            if (childEntity!=null && m_oimodelerViewable.getViewFilter().showEntity(childEntity))
                return new EntityHierarchyNode(node,childEntity);
        }
        catch (KAONException ignored) {
        }
        return null;
    }
    /**
     * Returns the index of the child containing given entity.
     *
     * @param node                  the node
     * @param childEntity           the child entity
     * @return                      the index of the child (or -1)
     */
    protected int getChildEntityIndex(OIModelNode node,Entity childEntity) {
        TreeNode[] children=node.getChildren();
        for (int i=0;i<children.length;i++) {
            EntityHierarchyNode childNode=(EntityHierarchyNode)children[i];
            if (childNode.getEntity().equals(childEntity))
                return i;
        }
        return -1;
    }
    /**
     * Examines the given event and returns the index of the child that must be removed for the event.
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the index of the child that must be removed for the event (or -1)
     */
    public int getRemovedChildIndex(OIModelNode node,ChangeEvent changeEvent) {
        Entity removedEntity=getRemovedChild(((EntityHierarchyNode)node).getEntity(),changeEvent);
        if (removedEntity!=null)
            return getChildEntityIndex(node,removedEntity);
        else
            return -1;
    }
    /**
     * Examines the given event and returns the index of the child that was changed by the event.
     *
     * @param node                  the node
     * @param changeEvent           the event
     * @return                      the index of the child that was changed by the event (or -1)
     */
    public int getChangedChildIndex(OIModelNode node,ChangeEvent changeEvent) {
        return -1;
    }
    /**
     * Examines the given event and returns <code>true</code> if the lexicon of the node changed.
     *
     * @param node                                  the node
     * @param instancesWithChangedLexicalEntries    the set of instances whose lexical entries changed
     * @return                                      <code>true</code> if the lexicon of the node changed
     */
    public boolean getLexiconChanged(OIModelNode node,Set instancesWithChangedLexicalEntries) {
        try {
            EntityHierarchyNode entityHierarchyNode=(EntityHierarchyNode)node;
            return instancesWithChangedLexicalEntries.contains(entityHierarchyNode.getEntity().getSpanningInstance());
        }
        catch (KAONException e) {
            return false;
        }
    }
    /**
     * Returns the event for associating nodes. If entities cannot be associated, <code>null</code>
     * is returned.
     *
     * @param superNode             the node
     * @param object                the object
     * @return                      the event for associating nodes
     */
    public ChangeEvent getAssociationEvent(OIModelNode superNode,Object object) {
        try {
            if (object instanceof Entity) {
                Entity subEntity=(Entity)object;
                Entity superEntity=((EntityHierarchyNode)superNode).getEntity();
                if (!superEntity.equals(subEntity) && m_oimodelerViewable.getViewFilter().showEntity(subEntity))
                    return getEntityAssociationEvent(superEntity,subEntity);
            }
        }
        catch (KAONException ignored) {
        }
        return null;
    }
    /**
     * Returns the load flag for related entities.
     *
     * @return                      the load flag
     */
    protected abstract int getRelatedEntitiesLoadFlag();
    /**
     * Returns the set of related concepts.
     *
     * @param entity                the entity
     * @return                      the set of related entities
     * @throws KAONException        thrown if there is an error
     */
    protected abstract Set getRelatedEntities(Entity entity) throws KAONException;
    /**
     * Examines the given event and returns the entity that must be added to the given node as child. If no node
     * should be generated for this event, <code>null</code> is returned.
     *
     * @param entity                the entity
     * @param changeEvent           the event
     * @return                      the entity that should be added as child (or <code>null</code> if no entity should be added)
     */
    public abstract Entity getAddedChild(Entity entity,ChangeEvent changeEvent);
    /**
     * Examines the given event and returns the entity that must be removed from the given node as child. If no node
     * should be removed for this event, <code>null</code> is returned.
     *
     * @param entity                the entity
     * @param changeEvent           the event
     * @return                      the entity that should be removed as child (or <code>null</code> if no entity should be removed)
     */
    public abstract Entity getRemovedChild(Entity entity,ChangeEvent changeEvent);
    /**
     * Returns the event for associating entities. If entities cannot be associated, <code>null</code>
     * is returned.
     *
     * @param superEntity           the superentity
     * @param subEntity             the subentity
     * @return                      the event for associating entityes
     */
    public abstract ChangeEvent getEntityAssociationEvent(Entity superEntity,Entity subEntity);

    /**
     * The comparator for the nodes.
     */
    protected static class NodeLabelComparator implements Comparator {
        public static final Comparator INSTANCE=new NodeLabelComparator();

        public int compare(Object o1,Object o2) {
            OIModelNode n1=(OIModelNode)o1;
            OIModelNode n2=(OIModelNode)o2;
            String label1=n1.toString();
            String label2=n2.toString();
            return label1.compareToIgnoreCase(label2);
        }
    }
}
