package de.fzi.wim.oimodeler.entityhierarchy;

import java.util.Set;
import java.util.Collections;
import javax.swing.Icon;
import javax.swing.tree.TreeNode;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

import de.fzi.wim.guibase.treetable.*;
import de.fzi.wim.guibase.lazytreemodel.*;

import de.fzi.wim.oimodeler.ui.*;

/**
 * Tree table model showing the concept instances.
 */
public class ConceptInstancesTreeTableModel extends OIModelTreeModel implements TreeTableModel {
    /** The viewable. */
    protected OIModelerViewable m_oimodelerViewable;
    /** The loader for instance properties. */
    protected OIModelNodeLoader m_instancePropertiesLoader;
    /** The null-loader. */
    protected OIModelNodeLoader m_nullNodeLoader;
    /** The icon for instance. */
    protected Icon m_instance;

    /**
     * Creates a tree model adapter.
     *
     * @param oimodelerViewable     the OI-modeler viewable
     */
    public ConceptInstancesTreeTableModel(OIModelerViewable oimodelerViewable) {
        super(oimodelerViewable,new ConceptInstancesNodeLoader(oimodelerViewable));
        m_oimodelerViewable=oimodelerViewable;
        m_instancePropertiesLoader=new InstancePropertiesNodeLoader(m_oimodelerViewable);
        m_nullNodeLoader=new NullNodeLoader(m_oimodelerViewable);
        m_instance=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.instance");
    }
    /**
     * Shows given entity in the model indirectly, meaning that another node is added that wraps the entity.
     *
     * @param entity                the entity shown
     * @throws KAONException        thrown if the entity cannot be shown
     */
    public void showEntityIndirectly(Entity entity) throws KAONException {
        showNode(new EntityIndirectionRootNode(entity));
    }
    /**
     * Returns the current entity.
     *
     * @return                      the entity shown
     */
    public Entity getCurrentEntity() {
        if (getRoot()==null)
            return null;
        else if (getRoot() instanceof EntityIndirectionRootNode)
            return ((EntityHierarchyNode)((EntityIndirectionRootNode)getRoot()).getChildAt(0)).getEntity();
        else
            return ((EntityHierarchyNode)getRoot()).getEntity();
    }
    /**
     * Returns a node loader for given node.
     *
     * @param node                          node for which loader is requested
     * @return                              loader capable of loading children of supplied node
     */
    public LazyNodeLoader getLazyNodeLoader(LazyTreeNode node) {
        if (node instanceof EntityHierarchyNode) {
            Entity entity=((EntityHierarchyNode)node).getEntity();
            if (entity instanceof Concept)
                return m_loader;
            else if (entity instanceof Instance)
                return m_instancePropertiesLoader;
        }
        return m_nullNodeLoader;
    }
    /**
     * Returns the number of columns in the model.
     *
     * @return                      number of columsn in the model
     */
    public int getColumnCount() {
        return 3;
    }
    /**
     * Returns the name of the column with given index.
     *
     * @param column                index of column for which name is requested
     * @return                      name of column with given index
     */
    public String getColumnName(int column) {
        switch (column) {
        case 0:
            return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("oimodeler.entityName");
        case 1:
            return " ";
        case 2:
        default:
            return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("oimodeler.value");
        }
    }
    /**
     * Returns the class of the column with given index.
     *
     * @param column                index of column for which class is requested
     * @return                      class of column with given index
     */
    public Class getColumnClass(int column) {
        if (column==1)
            return Icon.class;
        else
            return String.class;
    }
    /**
     * Returns the value from an object at given column.
     *
     * @param node                  node whose value needs to be read
     * @param column                column whose value is requested
     * @return                      value of the column for given node
     */
    public Object getValueAt(Object node,int column) {
        if (node instanceof EntityHierarchyNode) {
            EntityHierarchyNode entityHierarchyNode=(EntityHierarchyNode)node;
            if (column==0)
                return entityHierarchyNode.toString();
        }
        else if (node instanceof PropertyInstanceNode) {
            PropertyInstanceNode propertyInstanceNode=(PropertyInstanceNode)node;
            if (column==0)
                return propertyInstanceNode.getPropertyLabel();
            else if (column==1) {
                if (!propertyInstanceNode.isAttribute())
                    return m_instance;
            }
            else
                return propertyInstanceNode.getTargetValueLabel();
        }
        else if (node instanceof DefaultNoChildrenNode) {
            if (column==0)
                return ((DefaultNoChildrenNode)node).toString();
        }
        return null;
    }
    /**
     * Checks whether cell for given node is editable.
     *
     * @param node                  node that needs to be tested
     * @param column                column that needs to be tested
     * @return                      <code>true</code> if the cell displaying the value is editable
     */
    public boolean isCellEditable(Object node,int column) {
        if (node instanceof PropertyInstanceNode && column==2)
            return ((PropertyInstanceNode)node).isAttribute();
        return false;
    }
    /**
     * Sets the value into a node at given column.
     *
     * @param aValue                value that needs to be set
     * @param node                  node whose value needs to be set
     * @param column                column whose value needs to be set
     */
    public void setValueAt(Object aValue,Object node,int column) {
        try {
            if (node instanceof PropertyInstanceNode && column==2) {
                PropertyInstanceNode propertyInstanceNode=(PropertyInstanceNode)node;
                if (propertyInstanceNode.isAttribute()) {
                    String newValue=aValue==null ? "" : aValue.toString();
                    if (!newValue.equals(propertyInstanceNode.getTargetValueLabel())) {
                        PropertyInstance propertyInstance=propertyInstanceNode.getPropertyInstance();
                        m_oimodelerViewable.changeOIModel(Collections.singletonList(new ChangePropertyInstanceValue(propertyInstance.getSourceOIModel(),null,propertyInstance,newValue)));
                    }
                }
            }
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
    }

    /**
     * The null node loader.
     */
    protected static class NullNodeLoader extends OIModelNodeLoader {
        public NullNodeLoader(OIModelerViewable oimodelerViewable) {
            super(oimodelerViewable);
        }
        protected TreeNode[] loadNodeChildrenEx(LazyTreeModel model,LazyTreeNode node) {
            return AbstractLazyTreeNode.EMPTY_CHILDREN;
        }
        public int getInsertPosition(OIModelNode node,OIModelNode newNode) {
            return node.getChildCount();
        }
        public OIModelNode getAddedChildNode(OIModelNode node,ChangeEvent changeEvent) {
            return null;
        }
        public int getRemovedChildIndex(OIModelNode node,ChangeEvent changeEvent) {
            return -1;
        }
        public int getChangedChildIndex(OIModelNode node,ChangeEvent changeEvent) {
            return -1;
        }
        public boolean getLexiconChanged(OIModelNode node,Set instancesWithChangedLexicalEntries) {
            try {
                if (node instanceof PropertyInstanceNode) {
                    PropertyInstance propertyInstance=((PropertyInstanceNode)node).getPropertyInstance();
                    if (instancesWithChangedLexicalEntries.contains(propertyInstance.getProperty().getSpanningInstance()))
                        return true;
                    else if (instancesWithChangedLexicalEntries.contains(propertyInstance.getSourceInstance().getSpanningInstance()))
                        return true;
                    else if ((propertyInstance.getTargetValue() instanceof Instance) && instancesWithChangedLexicalEntries.contains(((Instance)propertyInstance.getTargetValue()).getSpanningInstance()))
                        return true;
                }
            }
            catch (KAONException e) {
            }
            return false;
        }
        public ChangeEvent getAssociationEvent(OIModelNode node,Object object) {
            return null;
        }
    }

    /**
     * The root node. This class adds another level of hierarchy in the tree in order to
     * prevent the tree from expanding the node by default.
     */
    protected class EntityIndirectionRootNode extends OIModelNode {
        public EntityIndirectionRootNode(Entity entity) throws KAONException {
            super(null);
            EntityHierarchyNode node=new EntityHierarchyNode(this,entity);
            node.setLanguageURI(m_languageURI);
            m_children=new TreeNode[] { node };
        }
        public void setLanguageURI(String languageURI) {
        }
        public Object getObject() {
            return null;
        }
    }
}
