package de.fzi.wim.oimodeler.conceptproperties;

import java.util.Set;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collections;
import java.util.Comparator;
import javax.swing.Icon;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

import de.fzi.wim.guibase.localization.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.viewfilter.*;

/**
 * The table model for displaying the property list.
 */
public class ConceptPropertiesTableModel extends AbstractTableModel implements OIModelListener {
    /** The OI-modeler viewable. */
    protected OIModelerViewable m_oimodelerViewable;
    /** Set to <code>true</code> if properties from the concept are shown. */
    protected boolean m_showPropertiesFrom;
    /** The concept for which the list is shown. */
    protected Concept m_concept;
    /** The list of all elements in the table. */
    protected List m_elements;
    /** The URI of the language. */
    protected String m_languageURI;
    /** Change visitor. */
    protected ChangeVisitor m_changeVisitor;
    /** The icon for inherited property. */
    protected Icon m_inheritedProperty;
    /** The icon for own property. */
    protected Icon m_ownProperty;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable             the viewable
     * @param showPropertiesFrom            <code>true</code> if this model will show properties from a concept
     */
    public ConceptPropertiesTableModel(OIModelerViewable oimodelerViewable,boolean showPropertiesFrom) {
        m_languageURI=oimodelerViewable.getLanguageURI();
        m_oimodelerViewable=oimodelerViewable;
        m_showPropertiesFrom=showPropertiesFrom;
        m_elements=Collections.EMPTY_LIST;
        m_changeVisitor=new OIModelChangeVisitor();
        m_oimodelerViewable.getOIModel().addOIModelListener(this);
        m_inheritedProperty=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.inherited_property");
        m_ownProperty=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.property");
    }
    /**
     * Disposes of this model.
     */
    public void dispose() {
        if (m_oimodelerViewable!=null) {
            m_oimodelerViewable.getOIModel().removeOIModelListener(this);
            m_oimodelerViewable=null;
            m_concept=null;
            m_elements.clear();
        }
    }
    /**
     * Returns <code>true</code> if this model shows properties from a concept.
     *
     * @return                              <code>true</code> if this model shows properties from a concept
     */
    public boolean getShowPropertiesFrom() {
        return m_showPropertiesFrom;
    }
    /**
     * Sets the language URI.
     *
     * @param languageURI                   the language URI
     * @throws KAONException                thrown if there is an error
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        m_languageURI=languageURI;
        Iterator iterator=m_elements.iterator();
        while (iterator.hasNext()) {
            TableRow tableRow=(TableRow)iterator.next();
            tableRow.update();
        }
        Collections.sort(m_elements,RowComparator.INSTANCE);
        fireTableDataChanged();
    }
    /**
     * Returns the concept.
     *
     * @return                              the concept for which the list is shown
     */
    public Concept getConcept() {
        return m_concept;
    }
    /**
     * Sets the concept.
     *
     * @param concept                       the concept for which the list is shown
     */
    public void setConcept(final Concept concept) {
        try {
            m_concept=concept;
            if (m_concept==null)
                m_elements=Collections.EMPTY_LIST;
            else if ((m_concept.getOIModel().getCapabilities() & OIModel.CAPABILITY_SUPPORTS_OPTIMIZED_LOADING)!=0) {
                m_elements=null;
                m_oimodelerViewable.getModule().getAppDriver().getThreadPool().executeTask(new Runnable() {
                    public void run() {
                        try {
                            final List elements=loadModel(concept);
                            SwingUtilities.invokeLater(new Runnable() {
                                public void run() {
                                    // if we are still supposed to load this element
                                    if (m_concept==concept) {
                                        m_elements=elements;
                                        fireTableDataChanged();
                                    }
                                }
                            });
                        }
                        catch (final KAONException e) {
                            SwingUtilities.invokeLater(new Runnable() {
                                public void run() {
                                    // if we are still supposed to load this element
                                    if (m_concept==concept) {
                                        m_elements=Collections.EMPTY_LIST;
                                        fireTableDataChanged();
                                    }
                                    m_oimodelerViewable.errorLoadingEntity(m_concept,e);
                                }
                            });
                        }
                    }
                });
            }
            else
                m_elements=loadModel(m_concept);
        }
        catch (KAONException e) {
            m_elements=Collections.EMPTY_LIST;
            m_oimodelerViewable.errorLoadingEntity(m_concept,e);
        }
        catch (InterruptedException e) {
            m_elements=Collections.EMPTY_LIST;
            m_oimodelerViewable.errorLoadingEntity(m_concept,e);
        }
        fireTableDataChanged();
    }
    /**
     * Loads the model.
     *
     * @param concept                       the concept
     * @return                              the new array of model elements
     * @throws KAONException                thrown if there is an error
     */
    protected List loadModel(Concept concept) throws KAONException {
        ViewFilter viewFilter=m_oimodelerViewable.getViewFilter();
        List elements=new ArrayList();
        synchronized (concept.getOIModel().getKAONConnection()) {
            Set allProperties=m_showPropertiesFrom ? concept.getAllPropertiesFromConcept() : concept.getAllPropertiesToConcept();
            Set ownProperties=m_showPropertiesFrom ? concept.getPropertiesFromConcept() : concept.getPropertiesToConcept();
            concept.getOIModel().loadObjects(allProperties,OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_PROPERTY_DOMAINS | OIModel.LOAD_LEXICON);
            Iterator iterator=allProperties.iterator();
            while (iterator.hasNext()) {
                Property property=(Property)iterator.next();
                if (viewFilter.showEntity(property)) {
                    TableRow tableRow=new TableRow(property,!ownProperties.contains(property));
                    elements.add(tableRow);
                    tableRow.update();
                }
            }
        }
        Collections.sort(elements,RowComparator.INSTANCE);
        return elements;
    }
    /**
     * Shows nothing.
     */
    public void showNothing() {
        setConcept(null);
    }
    /**
     * Returns the class of the column.
     *
     * @param columnIndex                   the index of the column
     * @return                              the column class
     */
    public Class getColumnClass(int columnIndex) {
        if (columnIndex==0)
            return Icon.class;
        else if (columnIndex==1)
            return String.class;
        else
            return Integer.class;
    }
    /**
     * Returns the number of columns.
     *
     * @return                              the number of columns
     */
    public int getColumnCount() {
        return m_showPropertiesFrom ? 4 : 2;
    }
    /**
     * Returns the name of the column.
     *
     * @param columnIndex                   the name of the column
     * @return                              the column name
     */
    public String getColumnName(int columnIndex) {
        LocalizationManager localizationManager=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager();
        switch (columnIndex) {
        case 0:
            return " ";
        case 1:
            return localizationManager.getPhrase("oimodeler.propertyName");
        case 2:
            return localizationManager.getPhrase("oimodeler.minimumCardinality");
        case 3:
        default:
            return localizationManager.getPhrase("oimodeler.maximumCardinality");
        }
    }
    /**
     * Returns the number of rows.
     *
     * @return                              the number of rows
     */
    public int getRowCount() {
        if (m_elements==null)
            return 1;
        else
            return m_elements.size();
    }
    /**
     * Returns the value at given position.
     *
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     * @return                              the value at given row and column
     */
    public Object getValueAt(int rowIndex,int columnIndex) {
        if (m_elements==null) {
            if (columnIndex==1)
                return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("oimodeler.loadingWait");
            else
                return "";
        }
        else {
            TableRow tableRow=(TableRow)m_elements.get(rowIndex);
            switch (columnIndex) {
            case 0:
                return tableRow.m_isInherited ? m_inheritedProperty : m_ownProperty;
            case 1:
                return tableRow.m_label;
            case 2:
                return tableRow.m_minimumCardinality;
            case 3:
            default:
                if (tableRow.m_maximumCardinality.intValue()==Integer.MAX_VALUE)
                    return null;
                else
                    return tableRow.m_maximumCardinality;
            }
        }
    }
    /**
     * Checks if the cell is editable.
     *
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     * @return                              <code>true</code> if the cell at given row and column is editable
     */
    public boolean isCellEditable(int rowIndex,int columnIndex) {
        return m_elements!=null && columnIndex>1;
    }
    /**
     * Sets the value at given position.
     *
     * @param aValue                        the value
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     */
    public void setValueAt(Object aValue,int rowIndex,int columnIndex) {
        try {
            TableRow tableRow=(TableRow)m_elements.get(rowIndex);
            if (columnIndex==2) {
                int value;
                if (aValue instanceof Number)
                    value=((Number)aValue).intValue();
                else
                    value=0;
                m_oimodelerViewable.changeOIModel(Collections.singletonList(new SetMinimumCardinality(tableRow.m_property.getSourceOIModel(),null,tableRow.m_property,m_concept,value)));
            }
            else if (columnIndex==3) {
                int value;
                if (aValue instanceof Number)
                    value=((Number)aValue).intValue();
                else
                    value=Integer.MAX_VALUE;
                m_oimodelerViewable.changeOIModel(Collections.singletonList(new SetMaximumCardinality(tableRow.m_property.getSourceOIModel(),null,tableRow.m_property,m_concept,value)));
            }
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
    }
    /**
     * Called when a bulk of change events is processed in the pool.
     *
     * @param oimodel                   OI-model that was changed
     * @param changeEvents              list of change events that occured
     */
    public void modelChanged(OIModel oimodel,List changeEvents) {
        if (m_concept!=null && m_elements!=null)
            try {
                Iterator iterator=changeEvents.iterator();
                while (iterator.hasNext()) {
                    ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                    changeEvent.accept(m_changeVisitor);
                }
                Set instancesWithChangedLexicalEntries=LexiconUtil.getInstancesWithChangedLexicalEntries(changeEvents);
                for (int i=m_elements.size()-1;i>=0;i--) {
                    TableRow tableRow=(TableRow)m_elements.get(i);
                    if (instancesWithChangedLexicalEntries.contains(tableRow.m_property.getSpanningInstance())) {
                        tableRow.update();
                        fireTableRowsUpdated(i,i);
                    }
                }
            }
            catch (KAONException e) {
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
            }
    }
    /**
     * Called when model is refreshed.
     *
     * @param oimodel                   OI-model that was refreshed
     */
    public void modelRefreshed(OIModel oimodel) {
        if (m_concept!=null)
            setConcept(m_concept);
    }
    /**
     * Called when model is being deleted.
     *
     * @param oimodel                   OI-model that is being deleted
     */
    public void modelDeleted(OIModel oimodel) {
    }
    /**
     * Locates the table row index for given property.
     *
     * @param property                  the property
     * @return                          the table row index for given property (-1 if row doesn't exist)
     */
    public int getTableRowIndex(Property property) {
        for (int i=0;i<m_elements.size();i++) {
            TableRow tableRow=(TableRow)m_elements.get(i);
            if (tableRow.m_property.equals(property))
                return i;
        }
        return -1;
    }
    /**
     * Inserts a row into the table.
     *
     * @param tableRow                  the row object
     */
    protected void insertTableRow(TableRow tableRow) {
        for (int i=0;i<m_elements.size();i++) {
            TableRow current=(TableRow)m_elements.get(i);
            int result=RowComparator.INSTANCE.compare(current,tableRow);
            if (result>=0) {
                m_elements.add(i,tableRow);
                fireTableRowsInserted(i,i);
                return;
            }
        }
        int index=m_elements.size();
        m_elements.add(tableRow);
        fireTableRowsInserted(index,index);
    }
    /**
     * Returns the property in given row.
     *
     * @param rowIndex                  the index of the row
     * @return                          the property in given row
     */
    public Property getProperty(int rowIndex) {
        return ((TableRow)m_elements.get(rowIndex)).m_property;
    }

    /**
     * The row in the table.
     */
    protected class TableRow {
        protected Property m_property;
        protected boolean m_isInherited;
        protected String m_label;
        protected Integer m_minimumCardinality;
        protected Integer m_maximumCardinality;

        public TableRow(Property property,boolean isInherited) {
            m_property=property;
            m_isInherited=isInherited;
        }
        public void update() throws KAONException {
            m_label=m_property.getLabel(m_languageURI);
            if (m_label==null)
                m_label=m_property.getURI();
            if (m_showPropertiesFrom) {
                m_minimumCardinality=new Integer(m_property.getMinimumCardinality(m_concept));
                m_maximumCardinality=new Integer(m_property.getMaximumCardinality(m_concept));
            }
        }
    }

    /**
     * The comparator for the rows.
     */
    protected static class RowComparator implements Comparator {
        public static final Comparator INSTANCE=new RowComparator();
        public int compare(Object o1,Object o2) {
            TableRow r1=(TableRow)o1;
            TableRow r2=(TableRow)o2;
            if (r1.m_isInherited && !r2.m_isInherited)
                return -1;
            if (!r1.m_isInherited && r2.m_isInherited)
                return 1;
            return r1.m_label.compareToIgnoreCase(r2.m_label);
        }
    }

    /**
     * The visitor for OI-model events.
     */
    protected class OIModelChangeVisitor extends NullChangeEventVisitor {
        public void visit(AddPropertyDomain event) throws KAONException {
            if (m_showPropertiesFrom && m_concept.equals(event.getConcept())) {
                TableRow tableRow=new TableRow(event.getProperty(),false);
                tableRow.update();
                insertTableRow(tableRow);
            }
        }
        public void visit(RemovePropertyDomain event) {
            if (m_showPropertiesFrom && m_concept.equals(event.getConcept())) {
                int index=getTableRowIndex(event.getProperty());
                if (index!=-1) {
                    m_elements.remove(index);
                    fireTableRowsDeleted(index,index);
                }
            }
        }
        public void visit(AddPropertyRange event) throws KAONException {
            if (!m_showPropertiesFrom && m_concept.equals(event.getConcept())) {
                TableRow tableRow=new TableRow(event.getProperty(),false);
                tableRow.update();
                insertTableRow(tableRow);
            }
        }
        public void visit(RemovePropertyRange event) {
            if (!m_showPropertiesFrom && m_concept.equals(event.getConcept())) {
                int index=getTableRowIndex(event.getProperty());
                if (index!=-1) {
                    m_elements.remove(index);
                    fireTableRowsDeleted(index,index);
                }
            }
        }
        public void visit(SetMinimumCardinality event) throws KAONException {
            if (m_showPropertiesFrom && m_concept.equals(event.getConcept())) {
                int tableRowIndex=getTableRowIndex(event.getProperty());
                if (tableRowIndex!=-1) {
                    TableRow tableRow=(TableRow)m_elements.get(tableRowIndex);
                    tableRow.update();
                    fireTableRowsUpdated(tableRowIndex,tableRowIndex);
                }
            }
        }
        public void visit(SetMaximumCardinality event) throws KAONException {
            if (m_showPropertiesFrom && m_concept.equals(event.getConcept())) {
                int tableRowIndex=getTableRowIndex(event.getProperty());
                if (tableRowIndex!=-1) {
                    TableRow tableRow=(TableRow)m_elements.get(tableRowIndex);
                    tableRow.update();
                    fireTableRowsUpdated(tableRowIndex,tableRowIndex);
                }
            }
        }
    }
}
