package de.fzi.wim.oimodeler.commands;

import java.util.List;
import java.lang.reflect.InvocationTargetException;
import javax.swing.SwingUtilities;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.evolutionlog.*;

import de.fzi.wim.guibase.command.*;
import de.fzi.wim.oimodeler.ui.*;

/**
 * When executed, this command changes the OI-model.
 */
public class ChangeOIModelCmd implements Command {
    /** OI-model to change. */
    protected OIModel m_oimodel;
    /** List of changes to apply. */
    protected List m_changeList;
    /** Evolution strategy to apply. */
    protected DependentEvolutionStrategy m_dependentEvolutionStrategy;
    /** The evolution logs for the connection. */
    protected KAONConnectionEvolutionLogs m_kaonConnectionEvolutionLogs;
    /** The viewable to disable while the command is applied. */
    protected OIModelerViewable m_oimodelerViewable;
    /** The type of the logging request. */
    protected int m_loggingType;

    /**
     * Creates an instance of this class.
     *
     * @param oimodel                           OI-model to change
     * @param changeList                        list of changes
     * @param dependentEvolutionStrategy        the evolution strategy to apply (may be <code>null</code>)
     * @param kaonConnectionEvolutionLogs       the information about the evolution logs
     * @param loggingType                       the type of the logging request
     * @param oimodelerViewable                 the viewable disable
     */
    public ChangeOIModelCmd(OIModel oimodel,List changeList,DependentEvolutionStrategy dependentEvolutionStrategy,KAONConnectionEvolutionLogs kaonConnectionEvolutionLogs,int loggingType,OIModelerViewable oimodelerViewable) {
        m_oimodel=oimodel;
        m_changeList=changeList;
        m_dependentEvolutionStrategy=dependentEvolutionStrategy;
        m_kaonConnectionEvolutionLogs=kaonConnectionEvolutionLogs;
        m_loggingType=loggingType;
        m_oimodelerViewable=oimodelerViewable;
    }
    /**
     * Returns <code>true</code>, since this command is executed asynchronously.
     *
     * @return                          <code>true</code>, since this command is executed asynchronously
     */
    public boolean executeAsynchronously() {
        return true;
    }
    /**
     * Executes this command.
     *
     * @throws CommandException         thrown if the command cannof be executed
     */
    public void execute() throws CommandException {
        commandWillBeExecuted();
        try {
            if (m_dependentEvolutionStrategy!=null) {
                m_changeList=m_dependentEvolutionStrategy.computeRequestedChanges(m_changeList);
                if (!doEvolutionDetailsDialog())
                    throw new CommandAbortedException("Evolution dialog not confirmed.");
                m_dependentEvolutionStrategy=null;
            }
            m_oimodel.getKAONConnection().applyChanges(m_changeList);
            m_kaonConnectionEvolutionLogs.logChanges(m_changeList,m_loggingType);
        }
        catch (KAONException e) {
            throw new CommandException("Cannot execute command",e);
        }
        finally {
            commandExecuted();
        }
    }
    /**
     * Called before command is executed.
     *
     * @throws CommandException         thrown if command cannot be executed
     */
    protected void commandWillBeExecuted() throws CommandException {
        if (SwingUtilities.isEventDispatchThread())
            m_oimodelerViewable.startOIModelProcessing();
        else
            try {
                SwingUtilities.invokeAndWait(new Runnable() {
                    public void run() {
                        m_oimodelerViewable.startOIModelProcessing();
                    }
                });
            }
            catch (InvocationTargetException ignored) {
            }
            catch (InterruptedException error) {
                throw new CommandException("Thread interrupted",error);
            }
    }
    /**
     * Called after command is executed.
     */
    protected void commandExecuted() {
        if (SwingUtilities.isEventDispatchThread())
            m_oimodelerViewable.finishOIModelProcessing();
        else
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    m_oimodelerViewable.finishOIModelProcessing();
                }
            });
    }
    /**
     * Executes the evolution dialog.
     *
     * @return                          <code>true</code> if the evolution has been confirmed
     */
    protected boolean doEvolutionDetailsDialog() {
        if (SwingUtilities.isEventDispatchThread())
            return doEvolutionDetailsDialogMainThread();
        else
            try {
                final boolean[] result=new boolean[1];
                SwingUtilities.invokeAndWait(new Runnable() {
                    public void run() {
                        result[0]=doEvolutionDetailsDialogMainThread();
                    }
                });
                return result[0];
            }
            catch (InterruptedException e) {
                return false;
            }
            catch (InvocationTargetException e) {
                return false;
            }
    }
    /**
     * Executes the evolution dialog on the main thread.
     *
     * @return                          <code>true</code> if the evolution has been confirmed
     */
    protected boolean doEvolutionDetailsDialogMainThread() {
        try {
            if (m_oimodelerViewable.getModule().getAppDriver().getConfiguration().getBoolean("oimodeler.showEvolution",true)) {
                EvolutionDetailsDlg evolutionDetailsDlg=new EvolutionDetailsDlg(m_oimodelerViewable,m_changeList);
                evolutionDetailsDlg.setVisible(true);
                if (!evolutionDetailsDlg.isConfirmed())
                    return false;
            }
            return true;
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
            return false;
        }
    }
}
