package de.fzi.wim.kaonportal.tree;

import java.util.Map;
import java.util.HashMap;

/**
 * A tree for showing information in the GUI.
 */
public abstract class Tree {
    /** The root node. */
    protected TreeNode m_root;
    /** The ID of the last node. */
    protected int m_lastNodeID;
    /** The map opf nodes indexed by their IDs. */
    protected Map m_nodesByIDs;

    /**
     * Creates an instance of this class.
     */
    public Tree() {
        m_nodesByIDs=new HashMap();
        m_lastNodeID=1;
    }
    /**
     * Refreshes the tree.
     *
     * @throws TreeException                    thrown if there is an error
     */
    public synchronized void refresh() throws TreeException {
        m_lastNodeID=1;
        m_nodesByIDs.clear();
        m_root=createRootNode();
        m_nodesByIDs.put(m_root.getID(),m_root);
        setExpanded(m_root,true);
    }
    /**
     * Returns a node with given ID.
     *
     * @param id                                the ID of the node
     * @return                                  the node with given ID (or <code>null</code> if the node with given ID doesn't exist)
     */
    public synchronized TreeNode getNode(String id) {
        return (TreeNode)m_nodesByIDs.get(id);
    }
    /**
     * Sets the expansion state of the node.
     *
     * @param treeNode                          the tree node
     * @param expanded                          the expansion state of the node
     */
    public synchronized void setExpanded(TreeNode treeNode,boolean expanded) {
        treeNode.setExpanded(expanded);
    }
    /**
     * Sets the expansion state of the node.
     *
     * @param id                                the id of the node
     * @param expanded                          the expansion state of the node
     */
    public synchronized void setExpanded(String id,boolean expanded) {
        TreeNode treeNode=getNode(id);
        if (treeNode!=null)
            setExpanded(treeNode,expanded);
    }
    /**
     * Returns the root node.
     *
     * @return                                  the root node
     */
    public synchronized TreeNode getRoot() {
        return m_root;
    }
    /**
     * Returns the children of given node.
     *
     * @param treeNode                          the node
     * @return                                  the children of the node
     * @throws TreeException                    thrown if there is an error
     */
    public synchronized TreeNode[] getNodeChildren(TreeNode treeNode) throws TreeException {
        if (!treeNode.getChildrenLoaded()) {
            TreeNode[] children=loadNodeChildren(treeNode);
            for (int i=0;i<children.length;i++)
                m_nodesByIDs.put(children[i].getID(),children[i]);
            treeNode.setNodeChildren(children);
        }
        return treeNode.getChildren();
    }
    /**
     * Returns the child node with given index.
     *
     * @param treeNode                          the node
     * @param index                             the index
     * @return                                  the node with given index
     * @throws TreeException                    thrown if there is an error
     */
    public synchronized TreeNode getChild(TreeNode treeNode,int index) throws TreeException {
        return getNodeChildren(treeNode)[index];
    }
    /**
     * Checks whether supplied node is leaf node.
     *
     * @param treeNode                          the node
     * @return                                  <code>true</code> if the node is leaf
     */
    public synchronized boolean isLeaf(TreeNode treeNode) {
        return treeNode.isLeaf();
    }
    /**
     * Creates the root node.
     *
     * @return                                  the root node
     * @throws TreeException                    thrown if there is an error
     */
    protected abstract TreeNode createRootNode() throws TreeException;
    /**
     * Loads the children of the node.
     *
     * @param node                              the node to load
     * @return                                  the children of the node
     * @throws TreeException                    thrown if there is an error loading node's children
     */
    protected abstract TreeNode[] loadNodeChildren(TreeNode node) throws TreeException;
}
