package de.fzi.wim.kaonportal.tree;

import java.util.Set;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

import de.fzi.wim.kaonportal.*;

/**
 * A tree showing the entity hierarchy.
 */
public class EntityHierarchyTree extends Tree {
    /** The OI-model. */
    protected OIModel m_oimodel;

    /**
     * Creates an instance of this class.
     */
    public EntityHierarchyTree() {
    }
    /**
     * Sets the ontology access.
     *
     * @param ontologyAccess                    the ontology access
     */
    public synchronized void setOntologyAccess(OntologyAccess ontologyAccess) {
        m_oimodel=ontologyAccess.getOIModel();
    }
    /**
     * Creates the root node.
     *
     * @return                                  the root node
     */
    protected TreeNode createRootNode() {
        return new EntityTreeNode(m_lastNodeID++,KAONVocabularyAdaptor.INSTANCE.getRoot());
    }
    /**
     * Loads the children of the node.
     *
     * @param node                              the node to load
     * @return                                  the children of the node
     * @throws TreeException                    thrown if there is an error loading node's children
     */
    protected TreeNode[] loadNodeChildren(TreeNode node) throws TreeException {
        try {
            EntityTreeNode entityTreeNode=(EntityTreeNode)node;
            Concept concept=m_oimodel.getConcept(entityTreeNode.getEntityURI());
            Set subConcepts=concept.getSubConcepts();
            m_oimodel.loadObjects(subConcepts,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_LEXICON);
            int numberOfChildren=subConcepts.size();
            if (KAONVocabularyAdaptor.INSTANCE.getRoot().equals(concept.getURI())) {
                numberOfChildren=0;
                Iterator iterator=subConcepts.iterator();
                while (iterator.hasNext()) {
                    Concept subConcept=(Concept)iterator.next();
                    if (!subConcept.getURI().startsWith(KAONConnection.LEXICAL_OIMODEL_URI))
                        numberOfChildren++;
                }
            }
            TreeNode[] children=new TreeNode[numberOfChildren];
            int index=0;
            Iterator iterator=subConcepts.iterator();
            while (iterator.hasNext()) {
                Concept subConcept=(Concept)iterator.next();
                if (!subConcept.getURI().startsWith(KAONConnection.LEXICAL_OIMODEL_URI)) {
                    EntityTreeNode newNode=new EntityTreeNode(m_lastNodeID++,subConcept.getURI());
                    newNode.initialize(subConcept);
                    children[index++]=newNode;
                }
            }
            return children;
        }
        catch (KAONException e) {
            throw new TreeException("Error accessing the ontology.",e);
        }
    }

    /**
     * The node containig an entity.
     */
    public static class EntityTreeNode extends TreeNode {
        /** The URI of the entity in this node. */
        protected String m_entityURI;
        /** The labels indexed by the language. */
        protected Map m_labelsByLanguage;

        /**
         * Creates an instance of this class.
         *
         * @param id                            the ID of the node
         * @param entityURI                     the entity URI
         */
        public EntityTreeNode(int id,String entityURI) {
            super(String.valueOf(id));
            m_entityURI=entityURI;
            m_labelsByLanguage=new HashMap();
        }
        /**
         * Initializes this node.
         *
         * @param entity                        the entity
         * @throws KAONException                thrown if there is an error
         */
        public void initialize(Entity entity) throws KAONException {
            Iterator iterator=entity.getLexicalEntries().iterator();
            while (iterator.hasNext()) {
                LexicalEntry lexicalEntry=(LexicalEntry)iterator.next();
                if (KAONVocabularyAdaptor.INSTANCE.getKAONLabel().equals(lexicalEntry.getTypeURI())) {
                    String languageURI=lexicalEntry.getInLanguage();
                    String label=lexicalEntry.getValue();
                    m_labelsByLanguage.put(languageURI,label);
                }
            }
        }
        /**
         * Returns the URI of the entity in this node.
         *
         * @return                              the entity URI
         */
        public String getEntityURI() {
            return m_entityURI;
        }
        /**
         * Returns the label in given language.
         *
         * @param languageURI                   the URI of the language
         * @return                              the label in given URI
         */
        public String getLabel(String languageURI) {
            String label=(String)m_labelsByLanguage.get(languageURI);
            if (label==null)
                label=m_entityURI;
            return label;
        }
    }
}
