package de.fzi.wim.kaonportal.tags;

import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import java.util.List;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.jsp.tagext.VariableInfo;

import de.fzi.wim.kaonportal.multilingual.Language;
import de.fzi.wim.kaonportal.OntologyFormatter;
import de.fzi.wim.kaonportal.OntologyInfo;
import de.fzi.wim.kaonportal.logging.EventLogger;
import de.fzi.wim.kaonportal.logging.NullEventLogger;

/**
 * This tag class initiates an ontology query supplied with the <code>query</code>
 * attribute. This list will be assigned to a surrounding tag which implements
 * <code>IteratorConsumer</code> or to the scripting variable with name
 * of <code>results</code>
 *
 * @see de.fzi.wim.kaonportal.tags.SearchTag.SearchTagTEI
 * @version 08-05-2002
 * @author Tammo Riedinger
 */
public class SearchTag extends AbstractObjectAccess {

    /** property declaration for tag attribute: results. */
    private String results;

    /** property declaration for tag attribute: query. */
    private String query;

    /**
     * This TagExtraInfo class serves as handler for the scripting variables
     * of the <code>SearchTag</code> class.
     *
     * @see de.fzi.wim.kaonportal.tags.SearchTag
     * @version 25-06-2003
     * @author Tammo Riedinger, Valentin Zacharias
     */
    public static class SearchTagTEI extends AbstractTEI {

        /**
         *
         * Creates a new <code>SearchTagTEI</code> object
         * and registers the variables supported be the <code>SearchTag</code>
         *
         * @throws JspException will be thrown when some error occures while creating the object
         */
        public SearchTagTEI() throws JspException{
            super();

            // register variables to export
            addVariable( "results", "java.util.Iterator", true, VariableInfo.NESTED );
        }
    }

    /**
     * Bean method to return the <code>query</code> parameter.
     * The <code>searchstring</code> parameter from the request will be returned.
     *
     * @return String - returns the <code>query</code> parameter
     */
    public String getQuery() {
        return query;
    }

    /**
     * Bean method to set the <code>query</code> parameter.
     *
     * @param value new value of <code>query</code> parameter
     */
    public void setQuery(String value) {
        query = value;
    }

    /**
     * Either returns the query parameter of decodes the session.
     *
     * @return String - the search string
     */
    protected String getSearchString() {
        String searchString=getQuery();
        if (searchString==null)
            searchString=(String)pageContext.getSession().getAttribute("currentSearchString");
        if (searchString==null)
            searchString="";
        return searchString;
    }

    /**
     * Bean method to return the <code>results</code> parameter.
     *
     * @return String - returns the <code>results</code> parameter
     */
    public String getResults() {
        return results;
    }

    /**
     * Bean method to set the <code>results</code> parameter.
     *
     * @param value new value of <code>results</code> parameter
     */
    public void setResults(String value) {
        results = value;
    }


    /**
     * This method is called after the JSP engine finished processing the tag.
     *
     * @return EVAL_BODY_BUFFERED if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     */
    public int doStartTag() {
        return EVAL_BODY_BUFFERED;
    }

    /**
     * Performs the search accoring to the <code>query</code> attribute and copies
     * the list of results either to an enclosing tag which implements the <code>IteratorConsumer</code> interface
     * or to the exported scripting variable. If no results are returned,
     * a language independent string will be added to the list, indicating to the
     * viewer that nothing has been found.
     *
     * This method is called after the JSP engine finished processing the tag.
     *
     * @return EVAL_PAGE if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     * @throws JspException will be thrown when some error occures
     */
    public int doEndTag() throws JspException {
        IteratorConsumer consumer = (IteratorConsumer)findAncestorWithClass(this, IteratorConsumer.class);

        try {
            OntologyFormatter format=getOntologyFormatter();
            String searchString=getSearchString();
            Collection searchResults=format.getOntologyAccess().getMatchingEntities(format.getLanguageURI(),searchString);
            OntologyInfo ontologyInfo=(OntologyInfo)pageContext.getSession().getAttribute("ontologyinfo");
            EventLogger eventLogger;
            if (ontologyInfo==null)
                eventLogger=NullEventLogger.INSTANCE;
            else
                eventLogger=ontologyInfo.getEventLogger();
            eventLogger.logSearchResults((HttpServletRequest)pageContext.getRequest(),searchString,searchResults);
            Iterator it=format.enumNamed(searchResults);
            if (!it.hasNext()) {
                Language lelem=(Language) pageContext.getSession().getAttribute("language");
                List noResults=new ArrayList();
                noResults.add(lelem.getPhrase("search_noresult_key"));
                it=noResults.iterator();
            }
            if (getResults()!= null) {
                pageContext.setAttribute(getResults(),it);
            }
            else {
                consumer.addIterator(it);
            }
        }
        catch (Exception e) {
            throw new JspException(e);
        }

        return EVAL_PAGE;
    }

    /**
     * This method is called after the JSP engine processes the body content of the tag.
     * If the tag's bodyContent is set to "empty," then this method
     * will not be called.
     *
     * @return EVAL_BODY_TAG if the JSP engine should evaluate the tag body again, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doAfterBody() throws JspException {
        try {
            JspWriter out = getPreviousOut();
            BodyContent bodyContent = getBodyContent();

            bodyContent.writeOut(out);
            bodyContent.clearBody();

        } catch (Exception ex) {
            throw new JspException("error in SearchTag: " + ex);
        }

        return SKIP_BODY;
    }
}
