package de.fzi.wim.kaonportal.tags;

import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import javax.servlet.jsp.tagext.VariableInfo;

import de.fzi.wim.kaonportal.OntologyFormatter;
import edu.unika.aifb.kaon.api.KAONException;

/**
 * This tag class retrieves a list of all root concepts from the ontology.
 * This list will be assigned to a surrounding tag which implements
 * <code>IteratorConsumer</code> or to the scripting variable with name
 * of <code>rootconcepts</code>
 *
 * @see de.fzi.wim.kaonportal.tags.RootConceptsTag.RootConceptsTagTEI
 * @version 08-05-2002
 * @author Tammo Riedinger
 */
public class RootConceptsTag extends AbstractObjectAccess {

    /** property declaration for tag attribute: rootconcepts. */
    private String rootconcepts;

    /**
     * This TagExtraInfo class serves as handler for the scripting variables
     * of the <code>RootConceptsTag</code> class.
     *
     * @see de.fzi.wim.kaonportal.tags.RootConceptsTag
     * @author Tammo Riedinger
     */
    public static class RootConceptsTagTEI extends AbstractTEI {

        /**
         *
         * Creates a new <code>RootConceptsTagTEI</code> object
         * and registers the variables supported be the <code>RootConceptsTag</code>
         *
         * @throws JspException will be thrown when some error occures while creating the object
         */
        public RootConceptsTagTEI() throws JspException{
            super();

            // register variables to export
            addVariable( "rootconcepts", "java.util.Iterator", true, VariableInfo.NESTED );
        }
    }

    /**
     * Bean method to return the <code>rootconcepts</code> parameter.
     *
     * @return String - returns the <code>rootconcepts</code> parameter
     */
    public String getRootconcepts() {
        return rootconcepts;
    }

    /**
     * Bean method to set the <code>rootconcepts</code> parameter.
     *
     * @param value new value of <code>rootconcepts</code> parameter
     */
    public void setRootconcepts(String value) {
        rootconcepts = value;
    }

    /**
     * This method copies the root concepts either to a scripting variable
     * or a surrounding tag which implements the <code>IteratorConsumer</code> interface.
     *
     * It is called when the JSP engine encounters the start tag,
     * after the attributes are processed.
     * Scripting variables (if any) have their values set here.
     *
     * @return EVAL_BODY_BUFFERED if the JSP engine should evaluate the tag body, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doStartTag() throws JspException {
        IteratorConsumer consumer = (IteratorConsumer)findAncestorWithClass(this, IteratorConsumer.class);

        try {
            OntologyFormatter formatter = getOntologyFormatter();
            if ( getRootconcepts() != null ) {
                pageContext.setAttribute( getRootconcepts(), formatter.getRootConceptsEnum() );
            }
            else {
                consumer.addIterator( formatter.getRootConceptsEnum() );
            }
        }
        catch (KAONException e) {
            throw new JspException(e);
        }

        return EVAL_BODY_BUFFERED;
    }

    /**
     * This method is called after the JSP engine finished processing the tag.
     *
     * @return EVAL_PAGE if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     */
    public int doEndTag() {
        return EVAL_PAGE;
    }

    /**
     * This method is called after the JSP engine processes the body content of the tag.
     * If the tag's bodyContent is set to "empty," then this method
     * will not be called.
     *
     * @return EVAL_BODY_TAG if the JSP engine should evaluate the tag body again, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doAfterBody() throws JspException {
        try {
            JspWriter out = getPreviousOut();
            BodyContent bodyContent = getBodyContent();

            bodyContent.writeOut(out);
            bodyContent.clearBody();

        } catch (Exception ex) {
            throw new JspException("error in RootConceptsTag: " + ex);
        }

        return SKIP_BODY;
    }
}
