package de.fzi.wim.kaonportal.tags;

import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import java.io.IOException;
import java.util.Iterator;

import de.fzi.wim.kaonportal.OntologyFormatter;

/**
 * This tag class is used to print the list of related instances.
 * Depending on the attribute <code>tableexists</code> a new table (its width set to
 * attribute <code>width</code> if specified) will be created and the attributes will be
 * inserted in that table. The columns can also be printed without creating a new table.
 * Optionally css classes for the relation type and the related instances can
 * be declared with the attributes <code>relationtypeclass</code> and <code>instancesclass</code>.
 * The elements that should be printed can either be set with the <code>addIterator</code> method
 * or can be supplied in a scripting variable of type <code>Iterator</code> with the name specified
 * in the <code>iteratorname</code> attribute.
 * @version 08-05-2002
 * @author Tammo Riedinger
 */
public class RelatedInstancesTableTag extends AbstractIteratorConsumer {

    /** property declaration for tag attribute: width. */
    private String width;

    /** property declaration for tag attribute: relationtypeclass. */
    private String relationtypeclass;

    /** property declaration for tag attribute: instancesclass. */
    private String instancesclass;

    /** property declaration for tag attribute: tableexists. */
    private String tableexists;

    /**
     * Bean method to return the <code>width</code> parameter.
     *
     * @return String - returns the <code>width</code> parameter
     */
    public String getWidth() {
        return width;
    }

    /**
     * Bean method to set the <code>width</code> parameter.
     *
     * @param value new value of <code>width</code>
     */
    public void setWidth(String value) {
        width = value;
    }

    /**
     * Bean method to return the <code>relationtypeclass</code> parameter.
     *
     * @return String - returns the <code>relationtypeclass</code> parameter
     */
    public String getRelationtypeclass() {
        return relationtypeclass;
    }

    /**
     * Bean method to set the <code>relationtypeclass</code> parameter.
     *
     * @param value new value of <code>relationtypeclass</code>
     */
    public void setRelationtypeclass(String value) {
        relationtypeclass = value;
    }

    /**
     * Bean method to return the <code>instancesclass</code> parameter.
     *
     * @return String - returns the <code>instancesclass</code> parameter
     */
    public String getInstancesclass() {
        return instancesclass;
    }

    /**
     * Bean method to set the <code>instancesclass</code> parameter.
     *
     * @param value new value of <code>instancesclass</code>
     */
    public void setInstancesclass(String value) {
        instancesclass = value;
    }

    /**
     * Bean method to return the <code>tableexists</code> parameter.
     *
     * @return String - returns the <code>tableexists</code> parameter
     */
    public String getTableexists() {
        return tableexists;
    }

    /**
     * Bean method to set the <code>tableexists</code> parameter.
     *
     * @param value new value of <code>tableexists</code>
     */
    public void setTableexists(String value) {
        tableexists = value;
    }

    /**
     * This method prints out the table and all instances contained in the supplied Iterator.
     *
     * @param strOut buffer to receive the output
     * @param iRelated Iterator of <code>OntologyFormatter.Tupel</code> objects containing
     * relations and the related instances
     */
    private void printRelatedTable( StringBuffer strOut, Iterator iRelated )  {

        // for each element in the iterator
        while ( iRelated.hasNext() ) {

            OntologyFormatter.Tupel current = (OntologyFormatter.Tupel) iRelated.next();

            if ( current != null ) {

                // add a new row
                strOut.append( "<tr>" );

                // add a new column for the relation
                strOut.append( "<td " );

                // eventually set the css class for the relation
                if ( getRelationtypeclass() != null ) {
                    strOut.append( "class=\"" + getRelationtypeclass() + "\"" );
                }

                // calculate the width of the column
                strOut.append(" >");

                // print the attribute name
                strOut.append( (String)current.a );

                // close the column
                strOut.append("</td>");

                // close the row
                strOut.append( "</tr>" );

                // append a rows with two columns of the instances associated
                // with this relation ( use NColumnTableTag class )
                NColumnTableTag ncoltable = new NColumnTableTag();
                ncoltable.setNumcolumns(2);
                ncoltable.setElementclass(getInstancesclass());
                ncoltable.printToNColunms(strOut, (Iterator)current.b );
            }
        }
    }

    /**
     * Depending on the <code>tableexists</code> attribute a new table will
     * be created. The width of the table is set to the <code>width</code> attribute.
     *
     * This method is called when the JSP engine encounters the start tag,
     * after the attributes are processed.
     * Scripting variables (if any) have their values set here.
     *
     * @return EVAL_BODY_BUFFERED if the JSP engine should evaluate the tag body, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doStartTag() throws JspException {
        // check if we have read the iterator from the page context
        if ( getIteratorname() != null ) {
            Iterator it = (Iterator) pageContext.getAttribute( getIteratorname() );

            if ( it != null ) {
                addIterator( it );
            }
        }

        // create the table if necessary
        StringBuffer strOut = new StringBuffer();

        if (!"true".equals(getTableexists())) {
            strOut.append( "<table" );

            if ( width != null ) {
                strOut.append( " width=\"" + getWidth() + "\"" );
            }

            strOut.append( ">" );
        }

        try {
            pageContext.getOut().print( strOut.toString() );
        }
        catch (IOException e) {
            throw new JspException(e);
        }

        return EVAL_BODY_BUFFERED;
    }

    /**
     * Depending on the supplied parameters the attributes of the object will be queried
     * and printed out.
     *
     * The method is called after the JSP engine finished processing the tag.
     *
     * @return EVAL_PAGE if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     * @throws JspException will be thrown when some error occures
     */
    public int doEndTag() throws JspException {
        StringBuffer strOut = new StringBuffer();

        printRelatedTable( strOut, getIterator() );
        // a table might have been created
        if (!"true".equals(getTableexists())) {
            // close table
            strOut.append( "</table>" );
        }

        try {
            pageContext.getOut().print( strOut.toString() );
        }
        catch (IOException e) {
            throw new JspException(e);
        }

        return EVAL_PAGE;
    }

    /**
     * This method is called after the JSP engine processes the body content of the tag.
     * If the tag's bodyContent is set to "empty," then this method
     * will not be called.
     *
     * @return EVAL_BODY_TAG if the JSP engine should evaluate the tag body again, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doAfterBody() throws JspException {
        try {
            JspWriter out = getPreviousOut();
            BodyContent bodyContent = getBodyContent();

            bodyContent.writeOut(out);
            bodyContent.clearBody();

        } catch (Exception ex) {
            throw new JspException("error in RelatedInstancesTableTag: " + ex);
        }

        return SKIP_BODY;
    }
}
