package de.fzi.wim.kaonportal.tags;

import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import java.util.Iterator;
import javax.servlet.jsp.tagext.VariableInfo;

import de.fzi.wim.kaonportal.OntologyFormatter;

import edu.unika.aifb.kaon.api.KAONException;
/**
 * This tag class retrieves <code>Entity</code> objects with a specified relation to
 * the current object. The current object will either be retrieved from the page context
 * or from the supplied uri and type. According to the <code>relation</code>
 * attribute a list of entities which have the specified relation to the loaded
 * object will be retrieved from the onotlogy. This list will be assigned to a
 * surrounding <code>entitylist</code> tag or exported as an <code>Iterator</code>
 * to a variable with name <code>relatedentities</code> if specified.<br>
 * Supported properties (values for <code>relation</code> attribute) according to the object types:<br><br>
 * <table border="1">
 *  <tr>
 *    <th>Objects of type <code>Concept</code></th>
 *    <th>Objects of type <code>Instance</code></th>
 *    <th>Objects of type <code>Property</code></th>
 *  </tr>
 *  <tr>
 *    <td><i>superconcept</i></td>
 *    <td><i>parentconcept</i></td>
 *    <td><i>domain</i></td>
 *  </tr>
 *  <tr>
 *    <td><i>subconcept</i></td>
 *    <td><i>similar</i></td>
 *    <td></td>
 *  </tr>
 *  <tr>
 *    <td><i>attribute</i></td>
 *    <td></td>
 *    <td></td>
 *  </tr>
 *  <tr>
 *    <td><i>incoming</i></td>
 *    <td></td>
 *    <td></td>
 *  </tr>
 *  <tr>
 *    <td><i>outgoing</i></td>
 *    <td></td>
 *    <td></td>
 *  </tr>
 *  <tr>
 *    <td><i>instance</i></td>
 *    <td></td>
 *    <td></td>
 *  </tr>
 * </table><br>
 * Other combinations are not supported.
 *
 * @see de.fzi.wim.kaonportal.tags.RelatedEntitiesTag.RelatedEntitiesTagTEI
 * @version 08-05-2002
 * @author Tammo Riedinger
 */
public class RelatedEntitiesTag extends AbstractObjectAccess {
    /** property declaration for tag attribute: relatedentities. */
    private String relatedentities;

    /**
     * This TagExtraInfo class serves as handler for the scripting variables
     * of the <code>RelatedEntitiesTag</code> class.
     *
     * @see de.fzi.wim.kaonportal.tags.RelatedEntitiesTag
     * @author Tammo Riedinger
     */
    public static class RelatedEntitiesTagTEI extends AbstractTEI {

        /**
         *
         * Creates a new <code>RelatedEntitiesTagTEI</code> object
         * and registers the variables supported be the <code>RelatedEntitiesTag</code>
         *
         * @throws JspException will be thrown when some error occures while creating the object
         */
        public RelatedEntitiesTagTEI() throws JspException{
            super();

            // register variables to export
            addVariable( "relatedentities", "java.util.Iterator", true, VariableInfo.NESTED );
        }
    }

    /**
     * Bean method to return the <code>relatedentities</code> parameter.
     *
     * @return String - returns the <code>relatedentities</code> parameter
     */
    public String getRelatedentities() {
        return relatedentities;
    }

    /**
     * Bean method to set the <code>relatedentities</code> parameter.
     *
     * @param value new value of <code>relatedentities</code> parameter
     */
    public void setRelatedentities(String value) {
        relatedentities = value;
    }

    /** Contains the type of the relation of the related entity to the current object. */
    protected String relation;

    /**
     * Bean method to return the <code>relation</code> parameter.
     *
     * @return String - returns the <code>relation</code> parameter
     */
    public String getRelation() {
        return relation;
    }

    /**
     * Bean method to set the <code>relation</code> parameter.
     *
     * @param value new value of <code>relation</code> parameter
     */
    public void setRelation(String value) {
        relation = value;
    }

    /**
     * This method retrieves a list of entities having the relation which is supplied
     * with the <code>relation</code> attribute to the specified object. This list
     * will either be exported as a scripting variable with the name specified by the
     * <code>relatedentities</code> attribute or copied to an enclosing tag that implements the <code>IteratorConsumer</code>
     * interface. If no results are returned a language independent string will be added to the list, indicating to the
     * viewer that nothing has been found.
     *
     * This method is called when the JSP engine encounters the start tag,
     * after the attributes are processed.
     * Scripting variables (if any) have their values set here.
     *
     * @return EVAL_BODY_BUFFERED if the JSP engine should evaluate the tag body, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doStartTag() throws JspException {
        IteratorConsumer consumer = (IteratorConsumer)findAncestorWithClass(this, IteratorConsumer.class);

        OntologyFormatter format = getOntologyFormatter();

        try {
            Iterator it = null;
            if ("superconcepts".equals(relation)) {
                it = format.getSuperConceptsEnum(getConcept());
            }
            else if ("subconcepts".equals(relation)) {
                it = format.getSubConceptsEnum(getConcept());
            }
            else if ("concept_attributes".equals(relation)) {
                it = format.getAttributesEnum(getConcept());
            }
            else if ("concept_incoming_properties".equals(relation)) {
                it = format.getIncomingPropertiesEnum(getConcept());
            }
            else if ("concept_outgoing_properties_no_attributes".equals(relation)) {
                it = format.getOutgoingPropertiesNoAttributesEnum(getConcept());
            }
            else if ("instance_attributes".equals(relation)) {
                it = format.getInstanceAttributes(getInstance());
            }
            else if ("all_instance_attributes".equals(relation)) {
                it = format.getAllInstanceAttributes(getInstance());
            }
            else if ("instance_outgoing_properties_no_attributes".equals(relation)) {
                it = format.getFromRelatedEnum(getInstance());
            }
            else if ("all_instance_outgoing_properties_no_attributes".equals(relation)) {
                it = format.getAllFromRelatedEnum(getInstance());
            }
            else if ("instance".equals(relation)) {
                it = format.getInstancesEnum(getConcept());
            }
            else if ("domain".equals(relation)) {
                it = format.getDomain(getProperty());
            }
            else if ("range".equals(relation)) {
                it = format.getRange(getProperty());
            }
            else if ("parentconcept".equals(relation)) {
                it = format.getConceptsEnum(getInstance());
            }
            else if ("similar".equals(relation)) {
                it = format.getSimilarEnum(getInstance());
            }
            else
                throw new JspException("Unknown relation "+getRelation());

            if (getRelatedentities()!=null) {
                pageContext.setAttribute(getRelatedentities(), it );
            }
            else if (consumer!=null) {
                consumer.addIterator(it);
            }

        }
        catch (KAONException e) {
            throw new JspException(e);
        }

        return EVAL_BODY_BUFFERED;
    }

    /**
     * This method is called after the JSP engine finished processing the tag.
     *
     * @return EVAL_PAGE if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     */
    public int doEndTag() {
        return EVAL_PAGE;
    }

    /**
     * This method is called after the JSP engine processes the body content of the tag.
     * If the tag's bodyContent is set to "empty," then this method
     * will not be called.
     *
     * @return EVAL_BODY_TAG if the JSP engine should evaluate the tag body again, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doAfterBody() throws JspException {
        try {
            JspWriter out = getPreviousOut();
            BodyContent bodyContent = getBodyContent();

            bodyContent.writeOut(out);
            bodyContent.clearBody();

        } catch (Exception ex) {
            throw new JspException("error in RelatedEntitiesTag: " + ex);
        }

        return SKIP_BODY;
    }
}
