package de.fzi.wim.kaonportal.tags;

import javax.servlet.http.HttpSession;
import javax.servlet.jsp.tagext.BodyTagSupport;
import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import javax.servlet.jsp.tagext.VariableInfo;
import java.io.IOException;

import de.fzi.wim.kaonportal.multilingual.Language;

/**
 * With this tag class a phrase specified by the <code>key</code> attribute
 * can be retrieved according to the currently activated language. With the
 * <code>output</code> attribute a scripting variable with the scope of the phrase tag
 * can be created. Inside the body of the phrase tag that variable can be referenced
 * by the name specified in the <code>output</code> attribute.
 *
 * @see de.fzi.wim.kaonportal.tags.PhraseTag.PhraseTagTEI
 * @version 08-05-2002
 * @author Tammo Riedinger
 */
public class PhraseTag extends BodyTagSupport {

    /** property declaration for tag attribute: key. */
    private String key;

    /** property declaration for tag attribute: output. */
    private String output;

    /**
     * This TagExtraInfo class serves as handler for the scripting variables
     * of the <code>PhraseTag</code> class.
     *
     * @see de.fzi.wim.kaonportal.tags.PhraseTag
     * @author Tammo Riedinger
     */
    public static class PhraseTagTEI extends AbstractTEI {

        /**
         *
         * Creates a new <code>PhraseTagTEI</code> object
         * and registers the variables supported be the <code>PhraseTag</code>
         *
         * @throws JspException will be thrown when some error occures while creating the object
         */
        public PhraseTagTEI() throws JspException{
            super();

            // register variables to export
            addVariable( "output", "java.lang.String", true, VariableInfo.NESTED );
        }
    }

    /**
     * Bean method to return the <code>key</code> parameter.
     *
     * @return String - returns the <code>key</code> parameter
     */
    public String getKey() {
        return key;
    }

    /**
     * Bean method to set the <code>key</code> parameter.
     *
     * @param value new value of <code>key</code> parameter
     */
    public void setKey(String value) {
        key = value;
    }

    /**
     * Bean method to return the <code>output</code> parameter.
     *
     * @return String - returns the <code>output</code> parameter
     */
    public String getOutput() {
        return output;
    }

    /**
     * Bean method to set the <code>output</code> parameter.
     *
     * @param value new value of <code>output</code> parameter
     */
    public void setOutput(String value) {
        output = value;
    }


    /**
     * This method looks up the phrase specified by the <code>key</code> attribute using
     * the current language. It the either prints the phrase or copies it to
     * the scripting variable with the name specified in the <code>output</code> attribute.
     *
     * The method is called when the JSP engine encounters the start tag,
     * after the attributes are processed.
     * Scripting variables (if any) have their values set here.
     *
     * @return EVAL_BODY_BUFFERED if the JSP engine should evaluate the tag body, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doStartTag() throws JspException {
        String value = "";
        if ( getKey() != null ) {
            HttpSession session = pageContext.getSession();
            Language lelem = (Language) session.getAttribute("language");
            value = lelem.getPhrase(getKey());
        }

        // print the phrase to the servlet or copy it to the scripting variable
        if ( getOutput() != null ) {
            pageContext.setAttribute( getOutput(), value );
        }
        else {
            try {
                pageContext.getOut().print( value );
            }
            catch (IOException e) {
                throw new JspException(e);
            }
        }

        return EVAL_BODY_BUFFERED;
    }

    /**
     * This method is called after the JSP engine finished processing the tag.
     *
     * @return EVAL_PAGE if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     */
    public int doEndTag() {
        return EVAL_PAGE;
    }

    /**
     *
     * This method is called after the JSP engine processes the body content of the tag.
     * If the tag's bodyContent is set to "empty," then this method
     * will not be called.
     *
     * @return EVAL_BODY_TAG if the JSP engine should evaluate the tag body again, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doAfterBody() throws JspException {
        try {
            JspWriter out = getPreviousOut();
            BodyContent bodyContent = getBodyContent();

            bodyContent.writeOut(out);
            bodyContent.clearBody();

        } catch (Exception ex) {
            throw new JspException("error in PhraseTag: " + ex);
        }

        return SKIP_BODY;
    }
}
