package de.fzi.wim.kaonportal.tags;

import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import java.io.IOException;
import java.util.Iterator;

/**
 * This tag class is used to print a list of elements in n columns ( html table columns ).
 * Depending on the attribute <code>tableexists</code> a new table (its width set to
 * attribute <code>width</code> if specified) will be created and the columns will be
 * inserted in that table. The columns can also be printed without creating a new table.
 * Optionally a css class for the elements can be declared with the attribute <code>elementclass</code>.
 * If the attribute <code>additionalelementclass</code> is set to a second css class then
 * every second line of the table will be formatted with that css class.
 * The elements that should be printed can either be set with the <code>addIterator</code> method
 * or can be supplied in a scripting variable of type <code>Iterator</code> with the name specified
 * in the <code>iteratorname</code> attribute.
 * @version 08-05-2002
 * @author Tammo Riedinger
 */
public class NColumnTableTag extends AbstractIteratorConsumer {
    /** property declaration for tag attribute: numcolumns. */
    private int numcolumns;

    /** property declaration for tag attribute: width. */
    private String width;

    /** property declaration for tag attribute: elementclass. */
    private String elementclass;

    /** property declaration for tag attribute: additionalelementclass. */
    private String additionalelementclass;

    /** property declaration for tag attribute: tableexists. */
    private String tableexists;

    /**
     * Bean method to return the <code>numcolumns</code> parameter.
     *
     * @return String - returns the <code>numcolumns</code> parameter
     */
    public int getNumcolumns() {
        // at least one column must be printed
        if ( numcolumns <= 0 ) {
            numcolumns = 1;
        }

        return numcolumns;
    }

    /**
     * Bean method to set the <code>numcolumns</code> parameter.
     *
     * @param value new value of <code>numcolumns</code>
     */
    public void setNumcolumns(int value) {
        numcolumns = value;
    }

    /**
     * Bean method to return the <code>width</code> parameter.
     *
     * @return String - returns the <code>width</code> parameter
     */
    public String getWidth() {
        return width;
    }

    /**
     * Bean method to set the <code>width</code> parameter.
     *
     * @param value new value of <code>width</code>
     */
    public void setWidth(String value) {
        width = value;
    }

    /**
     * Bean method to return the <code>elementclass</code> parameter.
     *
     * @return String - returns the <code>elementclass</code> parameter
     */
    public String getElementclass() {
        return elementclass;
    }

    /**
     * Bean method to set the <code>elementclass</code> parameter.
     *
     * @param value new value of <code>elementclass</code>
     */
    public void setElementclass(String value) {
        elementclass = value;
    }

    /**
     * Bean method to return the <code>additionalelementclass</code> parameter.
     *
     * @return String - returns the <code>additionalelementclass</code> parameter
     */
    public String getAdditionalelementclass() {
        return additionalelementclass;
    }

    /**
     * Bean method to set the <code>additionalelementclass</code> parameter.
     *
     * @param value new value of <code>additionalelementclass</code>
     */
    public void setAdditionalelementclass(String value) {
        additionalelementclass = value;
    }

    /**
     * Bean method to return the <code>tableexists</code> parameter.
     *
     * @return String - returns the <code>tableexists</code> parameter
     */
    public String getTableexists() {
        return tableexists;
    }

    /**
     * Bean method to set the <code>tableexists</code> parameter.
     *
     * @param value new value of <code>tableexists</code>
     */
    public void setTableexists(String value) {
        tableexists = value;
    }

    /**
     * Depending on the <code>numColumns</code> parameter the elements of the iterator
     * will be printed out in the specified number of columns. They will be inserted
     * rowwise.
     *
     * @param strOut string buffer which will receive the generated output
     * @param it list to insert values from
     */
    public void printToNColunms(StringBuffer strOut, Iterator it) {

        // add all elements of list to the table
        int colcount = getNumcolumns();
        // switch the element class
        int iRowCount = 0;

        if ( it != null ) {
            // for each element in the iterator
            while ( it.hasNext() ) {

                Object obj = it.next();
                // which is not null
                if ( obj != null ) {

                    if ( colcount == getNumcolumns() ) {
                        // open a new row if there are still numcolumns columns to be filled
                        strOut.append("<tr>");
                        iRowCount++;
                    }

                    // add a new column
                    strOut.append( "<td" );

                    // eventually set the css class
                    if ( ( getAdditionalelementclass() != null ) && ( ( iRowCount % 2 ) == 0 ) ) {
                        strOut.append( " class=\"" + getAdditionalelementclass() + "\"" );
                    }
                    else if ( getElementclass() != null ) {
                        strOut.append( " class=\"" + getElementclass() + "\"" );
                    }

                    // calculate the width of the column
                    strOut.append(" width=\"" + (100/getNumcolumns())  + "%\">");

                    // print the element
                    strOut.append( obj.toString() );

                    // close the column
                    strOut.append("</td>");

                    // handle the row closing
                    colcount--;
                    if ( colcount == 0 ) {
                        // close the row after numcolumns columns have been filled
                        strOut.append("</tr>");
                        colcount = getNumcolumns();
                    }
                }
            }

            // the element might have been written into the last column in a row
            if ( colcount < getNumcolumns() ) {
                // close a previously opened row
                strOut.append("</tr>");
            }
        }
    }

    /**
     * Depending on the <code>tableexists</code> attribute a new table will
     * be created. The width of the table is set to the <code>width</code> attribute.
     *
     * This method is called when the JSP engine encounters the start tag,
     * after the attributes are processed.
     * Scripting variables (if any) have their values set here.
     *
     * @return EVAL_BODY_BUFFERED if the JSP engine should evaluate the tag body, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doStartTag() throws JspException {
        // check if we have read the iterator from the page context
        if ( getIteratorname() != null ) {
            Iterator it = (Iterator) pageContext.getAttribute( getIteratorname() );

            if ( it != null ) {
                addIterator( it );
            }
        }

        // create the table
        StringBuffer strOut = new StringBuffer();

        if (!"true".equals(getTableexists())) {
            strOut.append( "<table" );

            if ( width != null ) {
                strOut.append( " width=\"" + getWidth() + "\"" );
            }

            strOut.append( ">" );
        }

        try {
            pageContext.getOut().print( strOut.toString() );
        }
        catch (IOException e) {
            throw new JspException(e);
        }

        return EVAL_BODY_BUFFERED;
    }

    /**
     * Depending on the <code>numcolumns</code> attribute the elements of the list
     * will be printed out in the specified number of columns. They will be inserted
     * rowwise.
     *
     * This method is called after the JSP engine finished processing the tag.
     *
     * @return EVAL_PAGE if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     * @throws JspException will be thrown when some error occures
     */
    public int doEndTag() throws JspException {
        StringBuffer strOut = new StringBuffer();

        printToNColunms( strOut, getIterator() );
        // a table might have been created
        if (!"true".equals(getTableexists())) {
            // close table
            strOut.append( "</table>" );
        }

        try {
            pageContext.getOut().print( strOut.toString() );
        }
        catch (IOException e) {
            throw new JspException(e);
        }

        return EVAL_PAGE;
    }

    /**
     *
     * This method is called after the JSP engine processes the body content of the tag.
     * If the tag's bodyContent is set to "empty," then this method
     * will not be called.
     *
     * @return EVAL_BODY_TAG if the JSP engine should evaluate the tag body again, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doAfterBody() throws JspException {
        try {
            JspWriter out = getPreviousOut();
            BodyContent bodyContent = getBodyContent();

            bodyContent.writeOut(out);
            bodyContent.clearBody();

        } catch (Exception ex) {
            throw new JspException("error in NColumnTableTag: " + ex);
        }

        return SKIP_BODY;
    }
}
