package de.fzi.wim.kaonportal.tags;

import javax.servlet.http.HttpSession;
import javax.servlet.jsp.tagext.BodyTagSupport;
import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.io.UnsupportedEncodingException;
import javax.servlet.jsp.tagext.VariableInfo;

import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;

import de.fzi.wim.kaonportal.multilingual.Language;
import de.fzi.wim.kaonportal.OntologyStore;
import de.fzi.wim.kaonportal.OntologyInfo;

/**
 * This tag class enables the dynamic change of the shown ontology.
 * It either prints out a list of uri's for all ontologies loaded from the configuration file,
 * assign the list to a surrounding tag which implements <code>IteratorConsumer</code> or to
 * exported to a scripting variable with the name specified in the <code>ontologylinks</code> parameter.
 * The displayed names for the ontologies will be retrieved from the <code>LanguageLoader</code>
 * The list will only be printed if the tag is neither nested in a <code>IteratorConsumer</code> tag
 * nor when a name is specified in the <code>ontologylinks</code> parameter.
 *
 * @see de.fzi.wim.kaonportal.configuration.ConfigurationReader
 * @see de.fzi.wim.kaonportal.tags.ChangeOntologyTag.ChangeOntologyTagTEI
 * @version 1.0
 * @author Tammo Riedinger
 */
public class ChangeOntologyTag extends BodyTagSupport {
    /** property declaration for tag attribute: ontologylinks. */
    private String ontologylinks;

    /**
     * This TagExtraInfo class serves as handler for the scripting variables
     * of the <code>ChangeOntologyTag</code> class.
     *
     * @see de.fzi.wim.kaonportal.tags.ChangeOntologyTag
     * @author Tammo Riedinger
     */
    public static class ChangeOntologyTagTEI extends AbstractTEI {

        /**
         *
         * Creates a new <code>ChangeOntologyTagTEI</code> object
         * and registers the variables supported be the <code>ChangeOntologyTag</code>
         *
         * @throws JspException will be thrown when some error occures while creating the object
         */
        public ChangeOntologyTagTEI() throws JspException{
            super();

            // register variables to export
            addVariable( "ontologylinks", "java.util.Iterator", true, VariableInfo.NESTED );
        }
    }

    /**
     * Bean method to return the <code>ontologylinks</code> parameter.
     *
     * @return String - returns the <code>ontologylinks</code> parameter
     */
    public String getOntologylinks() {
        return ontologylinks;
    }

    /**
     * Bean method to set the <code>ontologylinks</code> parameter.
     *
     * @param value new value of <code>ontologylinks</code> parameter
     */
    public void setOntologylinks(String value) {
        ontologylinks = value;
    }

    /**
     * This function creates an entry for the provided ontology in the list of ontologies.
     * It will create a link for all ontologies but the current one.
     *
     * @param ontology contains the ontology to be switched to
     * @param currentOntology contains the current ontology
     * @param strQuery contains the query to use as <code>href</code>
     * @param curLan contains the current language
     * @return String - returns the link
     */
    private String formatEntry( OntologyInfo ontology, OntologyInfo currentOntology, String strQuery, Language curLan) {
        if (ontology==currentOntology) {
            return ontology.getName();
        }
        else {
            StringBuffer buffy = new StringBuffer();

            // create the link to this ontology
            buffy.append( "<a href=\"" );

            try {
                buffy.append( ((HttpServletResponse)pageContext.getResponse()).encodeURL( strQuery + "cmd_changeontology=true&uri=" + URLEncoder.encode( ontology.getPhysicalURI(), "UTF-8" ) ) );
            }
            catch (UnsupportedEncodingException cantHappen) {
                buffy.append( "error" );
            }

            buffy.append( "\">" );
            buffy.append( curLan.getPhrase( ontology.getName() ) );
            buffy.append( "</a>" );

            return buffy.toString();
        }
    }

    /**
     * This method generates a list of links to the <code>dispatcher</code> servlet
     * with the parameters indicating to switch to the desired ontology.
     * The currently active ontology will be included in the list not as a link but only as a label.
     *
     * This method is called when the JSP engine encounters the start tag,
     * after the attributes are processed.
     * Scripting variables (if any) have their values set here.
     *
     * @return EVAL_BODY_BUFFERED if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     * @throws JspException will be thrown when some error occures
     */
    public int doStartTag() throws JspException {
        IteratorConsumer consumer = (IteratorConsumer)findAncestorWithClass(this, IteratorConsumer.class);

        try {
            HttpSession session = pageContext.getSession();
            OntologyStore store=(OntologyStore) pageContext.getServletContext().getAttribute("ontologystore");
            Language language = (Language) session.getAttribute("language");
            OntologyInfo currentOntology = (OntologyInfo)  session.getAttribute("ontologyinfo");
            List lOntos = new ArrayList();

            Iterator iontos = store.getOntologies().iterator();
            while (iontos.hasNext()) {
                OntologyInfo ontology=(OntologyInfo)iontos.next();
                lOntos.add( formatEntry( ontology, currentOntology, "dispatcher?", language) );
            }

            if ( getOntologylinks() != null ) {
                pageContext.setAttribute( getOntologylinks(), lOntos.iterator() );
            }

            if ( consumer != null ) {
                consumer.addIterator( lOntos.iterator() );
            }

        }
        catch (Exception e) {
            throw new JspException( e.getMessage() );
        }

        return EVAL_BODY_BUFFERED;
    }

    /**
     * This method is called after the JSP engine finished processing the tag.
     *
     * @return EVAL_PAGE if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     */
    public int doEndTag() {
        return EVAL_PAGE;
    }

    /**
     * This method is called after the JSP engine processes the body content of the tag.
     * If the tag's bodyContent is set to "empty," then this method
     * will not be called.
     *
     * @return EVAL_BODY_TAG if the JSP engine should evaluate the tag body again, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doAfterBody() throws JspException {
        try {
            JspWriter out = getPreviousOut();
            BodyContent bodyContent = getBodyContent();

            bodyContent.writeOut(out);
            bodyContent.clearBody();

        } catch (Exception ex) {
            throw new JspException("error in SearchTag: " + ex);
        }

        return SKIP_BODY;
    }
}
