package de.fzi.wim.kaonportal.tags;

import javax.servlet.http.HttpServletResponse;
import javax.servlet.jsp.tagext.BodyTagSupport;
import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import javax.servlet.jsp.tagext.VariableInfo;
import java.util.Locale;
import java.util.Iterator;

import de.fzi.wim.kaonportal.multilingual.Language;
import de.fzi.wim.kaonportal.multilingual.LanguageLoader;

/**
 * This tag class enables the dynamic change of the currently shown language.
 * It either prints out a list of uri's for all languages found in the language definition file,
 * assign the list to a surrounding tag which implements <code>IteratorConsumer</code> or to
 * exported to a scripting variable with the name specified in the <code>languagelinks</code> parameter.
 * The displayed names of the language will be retrieved from the <code>LanguageLoader</code>.
 * The list will only be printed if the tag is neither nested in a <code>IteratorConsumer</code> tag
 * nor when a name is specified in the <code>languagelinks</code> parameter.
 *
 * @see de.fzi.wim.kaonportal.tags.ChangeLanguageTag.ChangeLanguageTagTEI
  * @see de.fzi.wim.kaonportal.multilingual.Language
 * @see de.fzi.wim.kaonportal.multilingual.LanguageLoader
 * @version 08-05-2002
 * @author Tammo Riedinger
 */
public class ChangeLanguageTag extends BodyTagSupport {
    /** property declaration for tag attribute: languagelinks. */
    private String languagelinks;

    /**
     * This TagExtraInfo class serves as handler for the scripting variables
     * of the <code>ChangeLanguageTag</code> class.
     *
     * @see de.fzi.wim.kaonportal.tags.ChangeLanguageTag
     * @author Tammo Riedinger
     */
    public static class ChangeLanguageTagTEI extends AbstractTEI {

        /**
         *
         * Creates a new <code>ChangeLanguageTagTEI</code> object
         * and registers the variables supported be the <code>ChangeLanguageTag</code>
         *
         * @throws JspException will be thrown when some error occures while creating the object
         */
         public ChangeLanguageTagTEI() throws JspException{
            super();

            // register variables to export
            addVariable( "languagelinks", "java.util.Iterator", true, VariableInfo.NESTED );
        }
    }

    /**
     * Bean method to return the <code>languagelinks</code> parameter.
     *
     * @return String - returns the <code>languagelinks</code> parameter
     */
    public String getLanguagelinks() {
        return languagelinks;
    }

    /**
     * Bean method to set the <code>languagelinks</code> parameter.
     *
     * @param value new value of <code>languagelinks</code> parameter
     */
    public void setLanguagelinks(String value) {
        languagelinks = value;
    }

    /**
     * This function creates an entry for the provided language in the list of languages.
     * It will create a link for all languages but the current one or those which visible state
     * is set to <code>false</code>.
     *
     * @param lLan contains the language to be switched to
     * @param lCurrent contains the current locale
     * @param lanload landuage loader
     * @param strQuery contains the query to use as <code>href</code>
     * @return String - returns the link
     */
    private String formatEntry( Locale lLan, Locale lCurrent, LanguageLoader lanload, String strQuery ) {
        String value=null;
        // append the display name of the language in that language
        Language language=lanload.getLanguage(lLan);
        if (language!=null)
            value=language.getDisplayName(lLan);
        if (value==null)
            value=lLan.getDisplayName(lLan);
        if (value==null)
            value=lLan.toString();

        if (lLan.equals(lCurrent)) {
            return value;
        }
        else {
            // create the link to this language
            StringBuffer buffer = new StringBuffer();
            String strhref ="";

            strhref = strQuery + "cmd_changelanguage=true&language=" + lLan.getLanguage();
            if (lLan.getCountry() != null  && lLan.getCountry().length() > 0 ) {
                strhref = strhref + "&country=" + lLan.getCountry() ;
            }

            strhref = ((HttpServletResponse)pageContext.getResponse()).encodeURL( strhref );

            buffer.append("<a href=\"");
            buffer.append( strhref );
            buffer.append("\">");
            buffer.append(value);
            buffer.append("</a>");

            return buffer.toString();
        }
    }

    /**
     * This method generates a list of links to the <code>dispatcher</code> servlet
     * with the parameters indicating to switch to the desired language.
     * The parameters are <code>cmd_changelanguage</code>, <code>language</code> and <code>country</code>.
     * The currently active language will be included in the list not as a link but only as a label.
     *
     * This method is called when the JSP engine encounters the start tag,
     * after the attributes are processed.
     * Scripting variables (if any) have their values set here.
     *
     * @return EVAL_BODY_BUFFERED if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     * @throws JspException will be thrown when some error occures
     */
    public int doStartTag() throws JspException {
        IteratorConsumer consumer = (IteratorConsumer)findAncestorWithClass(this, IteratorConsumer.class);

        try {
            LanguageLoader lanload = (LanguageLoader) pageContext.getServletContext().getAttribute("languageloader");
            Locale lCurrent = ((Language) pageContext.getSession().getAttribute("language")).getLocale();
            java.util.ArrayList lLanguagelinks = new java.util.ArrayList();

            for ( Iterator lans = lanload.getLanguages(); lans.hasNext(); ) {
                Language lelem = (Language) lans.next();
                if (lelem.getVisible()) {
                    Locale llan = lelem.getLocale();
                    lLanguagelinks.add( formatEntry( llan, lCurrent, lanload, "dispatcher?" ) );
                }
            }

            boolean bPrinted=false;
            if ( getLanguagelinks() != null ) {
                pageContext.setAttribute( getLanguagelinks(), lLanguagelinks.iterator() );
                bPrinted=true;
            }

            if ( consumer != null ) {
                consumer.addIterator( lLanguagelinks.iterator() );
                bPrinted=true;
            }

            if ( !bPrinted ) {
                StringBuffer strLan = new StringBuffer();
                boolean first=true;
                for ( Iterator lans = lLanguagelinks.iterator(); lans.hasNext(); ) {
                    if (first)
                        first=false;
                    else
                        strLan.append("<br>");

                    strLan.append( (String) lans.next() );
                }

                pageContext.getOut().print( strLan.toString() );
            }
        }
        catch (Exception e) {
            throw new JspException( e.getMessage() );
        }

        return EVAL_BODY_BUFFERED;
    }

    /**
     * This method is called after the JSP engine finished processing the tag.
     *
     * @return EVAL_PAGE if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     */
    public int doEndTag() {
        return EVAL_PAGE;
    }

    /**
     * This method is called after the JSP engine processes the body content of the tag.
     * If the tag's bodyContent is set to "empty," then this method
     * will not be called.
     *
     * @return EVAL_BODY_TAG if the JSP engine should evaluate the tag body again, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doAfterBody() throws JspException {
        try {
            JspWriter out = getPreviousOut();
            BodyContent bodyContent = getBodyContent();

            bodyContent.writeOut(out);
            bodyContent.clearBody();

        } catch (Exception ex) {
            throw new JspException("error in SearchTag: " + ex);
        }

        return SKIP_BODY;
    }
}
