package de.fzi.wim.kaonportal.tags;

import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import java.io.IOException;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.fzi.wim.kaonportal.OntologyFormatter;

/**
 * This tag class is used to print the list of attributes
 * either into an existing or a new table. The list can be retrieved either from
 * an enclosed tag that uses the <code>addIterator</code> method or from a variable
 * of the <code>pageContext</code> with name <code>iteratorname</code>.
 * Depending on the attribute <code>tableexists</code> a new table (its width set to
 * attribute <code>width</code> if specified) will be created and the attributes will be
 * inserted in that table. The columns can also be printed without creating a new table.
 * Optionally css classes for the attribute names and values can
 * be declared with the attributes <code>nameclass</code> and <code>valueclass</code>.
 * @version 08-05-2002
 * @author Tammo Riedinger
 */
public class AttributeTableTag extends AbstractIteratorConsumer {
	
	/** a pattern that should recognizes eMail adresses (and not too much else) */
	private static Pattern eMailPatten = Pattern.compile("[^@\\s]+@[\\S]+\\.[\\S]*\\s*");
	
	/** a pattern that should recognize url's (and not too much else) */
	private static Pattern urlPattern = Pattern.compile("http://[\\S]+.[\\S]*\\s*");


    /** property declaration for tag attribute: width. */
    private String width;

    /** property declaration for tag attribute: nameclass. */
    private String nameclass;

    /** property declaration for tag attribute: valueclass. */
    private String valueclass;

    /** property declaration for tag attribute: tableexists. */
    private String tableexists;

    /**
     * Bean method to return the <code>width</code> parameter.
     *
     * @return String - returns the <code>width</code> parameter
     */
    public String getWidth() {
        return width;
    }

    /**
     * Bean method to set the <code>width</code> parameter.
     *
     * @param value new value of <code>width</code>
     */
    public void setWidth(String value) {
        width = value;
    }

    /**
     * Bean method to return the <code>nameclass</code> parameter.
     *
     * @return String - returns the <code>nameclass</code> parameter
     */
    public String getNameclass() {
        return nameclass;
    }

    /**
     * Bean method to set the <code>nameclass</code> parameter.
     *
     * @param value new value of <code>nameclass</code>
     */
    public void setNameclass(String value) {
        nameclass = value;
    }

    /**
     * Bean method to return the <code>valueclass</code> parameter.
     *
     * @return String - returns the <code>valueclass</code> parameter
     */
    public String getValueclass() {
        return valueclass;
    }

    /**
     * Bean method to set the <code>valueclass</code> parameter.
     *
     * @param value new value of <code>valueclass</code>
     */
    public void setValueclass(String value) {
        valueclass = value;
    }

    /**
     * Bean method to return the <code>tableexists</code> parameter.
     *
     * @return String - returns the <code>tableexists</code> parameter
     */
    public String getTableexists() {
        return tableexists;
    }

    /**
     * Bean method to set the <code>tableexists</code> parameter.
     *
     * @param value new value of <code>tableexists</code>
     */
    public void setTableexists(String value) {
        tableexists = value;
    }

    /**
     * This method prints out the table and all attributes contained in the supplied Iterator.
     *
     * @param strOut buffer to receive the printing result
     * @param iAttribs Iterator of <code>OntologyFormatter.Tupel</code> objects containing the attribute name and value
     */
    private void printAttributeTable( StringBuffer strOut, Iterator iAttribs ) {

        // for each element in the iterator
        while ( iAttribs.hasNext() ) {

            OntologyFormatter.Tupel current = (OntologyFormatter.Tupel) iAttribs.next();

            if ( current != null ) {

                // add a new row
                strOut.append( "<tr>" );

                // add a new column for the attribute name
                strOut.append( "<td " );

                // create two columns, the first will retrieve the attribute name, the second the attribute value

                // eventually set the css class for the attribute name
                if ( getNameclass() != null ) {
                    strOut.append( "class=\"" + getNameclass() + "\"" );
                }

                // calculate the width of the column
                strOut.append(" width=50%\">");

                // print the attribute name
                strOut.append((String) current.a);

                // close the column
                strOut.append("</td>");

                // add a new column for the attribute value
                strOut.append( "<td " );

                // eventually set the css class for the value name
                if ( getValueclass() != null ) {
                    strOut.append( "class=\"" + getValueclass() + "\"" );
                }

                // calculate the width of the column
                strOut.append(" width=50%\">");

                // print the value name (try to figure out, if it's a link)
                strOut.append( linkedAttribute((String)current.b ));

                // close the column
                strOut.append("</td>");

                // close the row
                strOut.append( "</tr>" );
            }
        }
    }

    /**
     * Checks the attribute value against a couple of patterns to see if it's a link (if 
     * yes it is formatted accordingly)
	 * @param string
	 * @return
	 */
	private String linkedAttribute(String string) {
		Matcher m = eMailPatten.matcher(string);
		if (m.matches()) {
			return ("<a href=\"mailto:"+string+"\">"+string+"</a>");
		}
		m = urlPattern.matcher(string);
		if (m.matches()) {
			return ("<a target=\"_blank\" href=\""+string+"\">"+string+"</a>");
		}
		return string;
	}

	/**
     * Depending on the set attributes a new table will be created or not.
     *
     * This method is called when the JSP engine encounters the start tag,
     * after the attributes are processed.
     * Scripting variables (if any) have their values set here.
     *
     * @return EVAL_BODY_BUFFERED if the JSP engine should evaluate the tag body, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doStartTag() throws JspException {

        // check if we have read the iterator from the page context
        if ( getIteratorname() != null ) {
            Iterator it = (Iterator) pageContext.getAttribute( getIteratorname() );

            if ( it != null ) {
                addIterator( it );
            }
        }

        // create the table if necessary
        StringBuffer strOut = new StringBuffer();

        if (!"true".equals(getTableexists())) {
            strOut.append( "<table" );

            if ( width != null ) {
                strOut.append( " width=\"" + getWidth() + "\"" );
            }

            strOut.append( ">" );
        }

        try {
            pageContext.getOut().print( strOut.toString() );
        }
        catch (IOException e) {
            throw new JspException(e);
        }

        return EVAL_BODY_BUFFERED;
    }

    /**
     * Depending on the supplied parameters the attributes of the object will be queried
     * and printed out.
     *
     * This method is called after the JSP engine finished processing the tag.
     *
     * @return EVAL_PAGE if the JSP engine should continue evaluating the JSP page, otherwise return SKIP_PAGE.
     * @throws JspException will be thrown when some error occures
     */
    public int doEndTag() throws JspException {
        StringBuffer strOut = new StringBuffer();

        printAttributeTable( strOut, getIterator() );
        // a table might have been created
        if (!"true".equals(getTableexists())) {
            // close table
            strOut.append( "</table>" );
        }

        try {
            pageContext.getOut().print( strOut.toString() );
        }
        catch (IOException e) {
            throw new JspException(e);
        }

        return EVAL_PAGE;
    }

    /**
     * This method is called after the JSP engine processes the body content of the tag.
     * If the tag's bodyContent is set to "empty," then this method
     * will not be called.
     *
     * @return EVAL_BODY_TAG if the JSP engine should evaluate the tag body again, otherwise return SKIP_BODY.
     * @throws JspException will be thrown when some error occures
     */
    public int doAfterBody() throws JspException {
        try {
            JspWriter out = getPreviousOut();
            BodyContent bodyContent = getBodyContent();

            bodyContent.writeOut(out);
            bodyContent.clearBody();

        } catch (Exception ex) {
            throw new JspException("error in AttributeTableTag: " + ex);
        }

        return SKIP_BODY;
    }
}
