package de.fzi.wim.kaonportal.tags;

import javax.servlet.jsp.JspException;
import javax.servlet.jsp.tagext.TagData;
import javax.servlet.jsp.tagext.TagExtraInfo;
import javax.servlet.jsp.tagext.VariableInfo;
import java.util.Iterator;
import java.util.ArrayList;

/**
 * This TagExtraInfo class serves as an abstract handler for the scripting variables
 * of the Tag classes. To export scripting variables you just
 * need to create a new class extending this class. Then call
 * the <code>addVariable</code> method for all scripting variables
 * that should be exported. In the class that should export the
 * variable enter code to create an attribute which gives the name for
 * the added attribute and then place the variable in the <code>pageContext</code>.<br>
 * For example enter code like this to add the attribute ( a new attribute entry in the tld file has to be added as well ):
 * <code><pre>
 * private String someattribute;
 *
 * public String getSomeattribute() {
 *      return someattribute;
 * }
 *
 * public void setSomeattribute(String value) {
 *      someattribute = value;
 * }
 * </pre></code>
 * And then add the code to place the variable in the page context:
 * <code><pre>
 * if ( getSomeattribute() != null ) {
 *      pageContext.setAttribute( getSomeattribute(), some_variable );
 * }
 * </pre></code>
 * The new TEI class can look like this for example:
 * <code><pre>
 * public class SomeTagTEI extends AbstractTEI {
 *      public SomeTagTEI() throws JspException {
 *         super();
 *
 *          // register variables to export
 *          addVariable( "someattribute",
 *                      "java.lang.String",
 *                      true,
 *                      VariableInfo.NESTED );
 *      }
 * }
 * </pre></code><br>
 * Finally add the <code>teiclass</code> tag in the
 * tld file ( the tag specifies the new TEI class you just created ).
 * @version 08-05-2002
 * @author Tammo Riedinger
 */
public abstract class AbstractTEI extends TagExtraInfo {

    /** This array contains the names of the variables which
     * should be exported for scripting
     */
    protected ArrayList m_arrVarnames = new ArrayList();

    /**
     * This class is used for storing
     * information about a certain variable which should be exported for scripting.
     *
     * @see javax.servlet.jsp.tagext.VariableInfo
     */
    protected class TEIVariableDef {
        /** name with which the variable name can be accessed */
        public String strAccessname;
        /** class name of the stored variable */
        public String strClassname;
        /** indicated if a variable should be declared new */
        public boolean bDeclare;
        /** scope in which the variable is valid */
        public int iScope;

        /**
         * Creates a new <code>TEIVariableDef</code> object.
         *
         * @param straccessname name with which the variable can be accessed in the context
         * @param strclassname classname of the variable
         * @param bdeclare indicates whether the variable should be declared new or not
         * @param iscope scope in which the variable is valid
         */
        public TEIVariableDef(String straccessname, String strclassname, boolean bdeclare, int iscope) {
            strAccessname = straccessname;
            strClassname = strclassname;
            bDeclare = bdeclare;
            iScope = iscope;
        }

        /**
         * Generates a hashcode for this object. The hashcode will be only depending on
         * the Accessname.
         *
         * @return int - hashcode for this object
         */
        public int hashCode() {
            if (strAccessname == null)
                return 0;

            return strAccessname.hashCode();
        }
    }

    /**
     * Register variables that are to be exported as scripting
     * variables.
     *
     * @param strName name with which the variable can be accessed in the context
     * @param strClassname classname of the variable
     * @param bDeclare indicates whether the variable should be declared new or not
     * @param iScope scope in which the variable is valid
     * @throws JspException thrown when the variable cannot be added
     */
     protected void addVariable( String strName, String strClassname, boolean bDeclare, int iScope ) throws JspException {
        TEIVariableDef newvar = new TEIVariableDef( strName, strClassname, bDeclare, iScope );

        if ( m_arrVarnames.contains( newvar ) ) {
            throw new JspException("Error adding variable: " + newvar.strAccessname + ". This variable is already registered." );
         }

         m_arrVarnames.add( newvar );
     }

    /**
     * This method is called by the JSP engine to set up
     * the scripting variables for a tag.
     *
     * @param data the supplied context for retrieving the variable name
     * @return VariableInfo[] array of <code>VariableInfo</code> objects for each variable
     */
    public VariableInfo[] getVariableInfo(TagData data) {
        ArrayList vars = new ArrayList();

        // copy all variable infos to normal array
        for (Iterator en = m_arrVarnames.iterator(); en.hasNext(); ) {
            TEIVariableDef curvar = (TEIVariableDef) en.next();

            //retrieve the name that variable should be stored with
            String outputname = data.getAttributeString( curvar.strAccessname );

            if ( outputname != null ) {
                //create the variable info
                VariableInfo attrvar = new VariableInfo( outputname,
                    curvar.strClassname,
                    curvar.bDeclare,
                    curvar.iScope );

                vars.add( attrvar );
            }
        }

        VariableInfo varInfoArray[] = new VariableInfo[vars.size()];
        int i=0;
        for (Iterator en = vars.iterator(); en.hasNext(); ) {
            varInfoArray[i] = (VariableInfo)en.next();
            i++;
        }

        return varInfoArray;
    }

    /**
     * This method is called by the JSP engine to verify
     * that the attributes are valid.
     * Modify this method to write code to accomplish this verification.
     *
     * @param data the supplied context
     * @return boolean returns if the supplied attributes are valid or not
     */
    public boolean isValid(TagData data) {
        return true;
    }

}
