package de.fzi.wim.kaonportal.tags;

import javax.servlet.jsp.tagext.BodyTagSupport;
import java.util.Collections;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * Abstract base class for an iterator consumer. Use this class if your tag shall be able 
 * to receive an iterator from an enclosed tag (that tag has to find this tag as its
 * parent and call the <code>addIterator</code> function) or from an enclosing tag that exports a
 * scripting variable with name <code>iteratorname</code>. The tag that extends this class
 * needs to add an attribute variable with name "iteratorname" to the tld file and has
 * to care for loading the actual iterator from the <code>pageContext</code> if the iteratorname has been
 * specified. For example the following code could be used in the <code>doStartTag</code>
 * method to accomplish this task:<br>
 * <code><pre>
 *  if ( getIteratorname() != null ) {
 *      Iterator it = (Iterator) pageContext.getAttribute( getIteratorname() );
 *      if ( it != null ) {
 *          addIterator( it );
 *      }
 *  }
 * </pre></code><br>
 *
 * @version 08-05-2002
 * @author Tammo Riedinger
 */
public abstract class AbstractIteratorConsumer extends BodyTagSupport implements IteratorConsumer {
    /** property declaration for tag attribute: iteratorname. */
    private String iteratorname;

    /**
     * Bean method to return the <code>iteratorname</code> parameter.
     *
     * @return String - returns the <code>iteratorname</code> parameter
     */
    public String getIteratorname() {
        return iteratorname;
    }

    /**
     * Bean method to set the <code>iteratorname</code> parameter.
     *
     * @param value new value of <code>iteratorname</code> parameter
     */
    public void setIteratorname(String value) {
        iteratorname = value;
    }
    
    /** This variable is used to store the current iterator in the iteration process */
    private Iterator m_iterator;

    /**
     * Returns the iterator.
     *
     * @return Iterator - returns the current iterator or an empty iterator
     */
    public Iterator getIterator() {
        if (m_iterator == null)
            return Collections.EMPTY_SET.iterator();
        else
            return m_iterator;
    }

    /**
     * Adds an iterator to this consumer. The consumer appends this iterator to
     * a combined list of iterators, which will be returned by the <code>getIterator</code>
     * method.
     *
     * @param iterator iterator which is to be added to the list
     */
    public void addIterator(Iterator iterator) {
        if (m_iterator == null)
            m_iterator = iterator;
        else
            m_iterator = new CompoundIterator(m_iterator, iterator);
    }

    /**
     * This class is used to store and merge the list of added iterators. It also functions
     * as the source for the combined iterator which is returned by the <code>getIterator</code>
     * method.
     */
    protected static class CompoundIterator implements Iterator {
        /** stores the first iterator in this list */        
        protected Iterator m_first;
        /** stores the second iterator in this list */
        protected Iterator m_second;

        /**
         * Creates a new <code>CompoundIterator</code>.
         *
         * @param first first iterator to be stored in this list
         * @param second second iterator to be stored in this list
         */
        public CompoundIterator(Iterator first,Iterator second) {
            m_first = first;
            m_second = second;
        }

        /**
        * Returns the next available value from the stored iterators.
        * 
        * @return java.lang.Object - the found object
        * @throws NoSuchElementException will be thrown when no element can be found
        */
        public java.lang.Object next() throws NoSuchElementException {
            if (m_first.hasNext())
                return m_first.next();
            else
                return m_second.next();
        }

        /**
        * Queries if another element is available in the stored iterators.
        * 
        * @return boolean - <code>true</code> if an element is available, <code>false</code> otherwise
        */
        public boolean hasNext() {
            if (m_first.hasNext())
                return true;
            else
                return m_second.hasNext();
        }

        /**
         * This method is not supported.
         *
         * @throws UnsupportedOperationException this abstract class does not support this operation
         */
        public void remove() throws UnsupportedOperationException {
            throw new UnsupportedOperationException( "AbstractIteratorConsumer: Remove operation is not supported by this iterator." );
        }
    }
}
