package de.fzi.wim.kaonportal.multilingual;

import java.net.URL;
import java.util.*;
import java.io.*;
import org.xml.sax.InputSource;
import org.w3c.dom.*;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import org.xml.sax.SAXParseException;
import org.xml.sax.SAXException;
import javax.servlet.ServletContext;

/**
 * This class loads data from a language definition file. It hashes
 * the loaded Language objects which can be queried by specifying their Locale.
 *
 * @author  Tammo Riedinger
 * @version 1.0
 * @see de.fzi.wim.kaonportal.multilingual.Language
 */
public class LanguageLoader {

    /**
     * This class is used for exceptions generated by <code>LanguageLoader</code>.
     *
     * @author  Tammo Riedinger
     * @version 1.0
     */
    public static class LanguageLoaderException extends Exception {
        /**
         * Constructor for an exception without message.
         */
        public LanguageLoaderException() {
            super();
        }

        /**
         * Constructor for an exception with a human-readable message.
         *
         * @param message The exception message.
         */
        public LanguageLoaderException( String message ) {
            super( message );
        }
    }

    /** Map of Language objects indexed by the hashcode of their Locale. */
    private java.util.HashMap m_languagesmap;
    /**
     * This variable stores the current servlet context. This is necessary
     * to include other language definition files via the <code>languageinclude</code> tag.
     */
    protected ServletContext m_context = null;


    /**
     * Creates a new LanguageLoader.
     *
     * @param context ServletContext to retrieve the files from
     * @param strFilename file name to a language definition file
     * @throws NullPointerException if the argument is <code>null</code>
     * @throws LanguageLoaderException if some error occured, while opening the provided language definition file
     */
    public LanguageLoader( ServletContext context, String strFilename ) throws NullPointerException, LanguageLoaderException {
        if ( ( strFilename == null ) || ( context == null ) ) {
            throw new NullPointerException();
        }

        try {
            m_context = context;
            URL url = m_context.getResource( strFilename );
            loadLanguagesFromUrl( url );
        }
        catch ( Exception e ) {
            throw new LanguageLoaderException( e.getMessage() );
        }
    }

    /**
     * Creates a new LanguageLoader.
     *
     * @param strfileName file name to a language definition file
     * @throws NullPointerException if the argument is <code>null</code>
     * @throws LanguageLoaderException if some error occured, while opening the provided language definition file
     */
    public LanguageLoader( String strfileName ) throws NullPointerException, LanguageLoaderException {
        if ( strfileName == null ) {
            throw new NullPointerException();
        }

        try {
            loadLanguagesFromFile( strfileName );
        }
        catch ( Exception e ) {
            throw new LanguageLoaderException( e.getMessage() );
        }
    }

    /**
     * Creates a new LanguageLoader.
     *
     * @param url URL to a language definition file
     * @throws NullPointerException if the argument is <code>null</code>
     * @throws LanguageLoaderException if some error occured, while opening the provided language definition file
     */
    public LanguageLoader( URL url ) throws NullPointerException, LanguageLoaderException {
        if ( url == null ) {
            throw new NullPointerException();
        }

        try {
            loadLanguagesFromUrl( url );
        }
        catch ( Exception e ) {
            throw new LanguageLoaderException( e.getMessage() );
        }
    }


    /**
     * Loads the data from a language definition file specified by an URL.
     *
     * @param url URL to the language definition file
     * @throws LanguageLoaderException if some error occured, while accessing the provided url
     */
    private void loadLanguagesFromUrl(URL url) throws LanguageLoaderException {
        try {
            loadLanguagesFromInputSource( new InputSource( url.openStream() ) );
        }
        catch (Exception e) {
            throw new LanguageLoaderException( e.getMessage() );
        }
    }

    /**
     * Loads the data from a language definition file specified by a filename.
     *
     * @param strfileName file name of the language definition file
     * @throws LanguageLoaderException if some error occured, while accessing the provided file name
     */
    private void loadLanguagesFromFile(String strfileName) throws LanguageLoaderException {
        try {
            File file = new File( strfileName );
            loadLanguagesFromInputSource( new InputSource( new FileInputStream( file ) ) );
        }
        catch ( FileNotFoundException fe ) {
            throw new LanguageLoaderException( "The specified file could not be found: " + fe.getMessage() + ", " + fe.toString() );
        }
        catch (Exception e) {
            throw new LanguageLoaderException( e.getMessage() );
        }
    }

    /**
     * Handles the loading of a phrase node. It adds the loaded key and associated phrase to
     * the provided Language object.
     *
     * @param lan Language object to add the found phrase
     * @param phrasenode node to handle
     */
    private void handlePhraseNode( Language lan, Node phrasenode ) {
        if ( ( lan == null ) || ( phrasenode == null ) )
            return;

        // find "name" attribute and use it as the key
        NamedNodeMap attrs = ((Element) phrasenode).getAttributes();
        Node namenode = attrs.getNamedItem( "name" );

        if ( namenode != null ) {
            if ( namenode.getNodeType() == Node.ATTRIBUTE_NODE ) {
                String strKey = namenode.getNodeValue();

                String strValue = null;
                Node valuenode = attrs.getNamedItem( "value" );
                if ( valuenode != null ) {
                    if ( valuenode.getNodeType() == Node.ATTRIBUTE_NODE ) {
                        strValue = valuenode.getNodeValue();
                    }
                }

                if ( strValue == null ) {
                    Node textnode = phrasenode.getFirstChild();
                    if ( textnode != null && textnode.getNodeType() == Node.TEXT_NODE ) {
                        strValue = textnode.getNodeValue();
                    }
                }

                if (strValue != null) {
                    strValue = strValue.trim();
                }

                if ( strKey != null && strValue != null && strKey.length() > 0 ) {
                    lan.addPhrase( strKey, strValue );
                }
            }
        }
    }

    /**
     * Reads the specified attributes from the NamedNodeMap and tries to create a <code>Locale</code>
     * object from their values.
     *
     * @param attributes the attributes nodes
     * @param lang_attrib_name the name of the language attribute
     * @param attributes the name of the country attribute
     * @return Locale the created <code>Locale</code> object or null
     */
    private Locale createLocaleFromAttributes( NamedNodeMap attributes, String lang_attrib_name, String country_attrib_name ) {
        Locale lret = null;
        String strlangname = "";
        String strcountryname = "";
        Node attribnode;

        attribnode = attributes.getNamedItem( lang_attrib_name );
        if ( attribnode != null ) {
            if ( attribnode.getNodeType() == Node.ATTRIBUTE_NODE ) {
                strlangname = attribnode.getNodeValue();
            }
        }

        attribnode = attributes.getNamedItem( country_attrib_name );
        if ( attribnode != null ) {
            if ( attribnode.getNodeType() == Node.ATTRIBUTE_NODE ) {
                strcountryname = attribnode.getNodeValue();
            }
        }

        if ( ( strlangname != null) && ( strlangname.length() > 0 ) ) {
            if ( strcountryname == null ) {
                lret = new Locale( strlangname );
            }
            else {
                lret = new Locale( strlangname, strcountryname );
            }
        }

        return lret;
    }

    /**
     * Reads the specified attributes from the NamedNodeMap and tries to create a <code>Language</code>
     * object from their values. If the language already exists in the list of loaded languages,
     * that specific <code>Language</code> object will be returned with its fields set to the new attributes.
     * This function will also handle the retrieval of the parent language. If a not yet loaded language
     * is specified as parent a new "empty" <code>Language</code> object will be created.
     *
     * @param attributes the attributes nodes
     * @return Language the created <code>Language</code> object or null
     */
    private Language createLanguageFromAttributes( NamedNodeMap attributes ) {
        Language lret = null;

        Locale newlocale = createLocaleFromAttributes( attributes, "name", "country" );
        Locale parentlocale = createLocaleFromAttributes( attributes, "parentlanguage", "parentcountry" );

        if ( newlocale != null ) {
            boolean bVisible = false;
            Node attribnode = attributes.getNamedItem( "display" );
            if ( attribnode != null ) {
                if ( attribnode.getNodeType() == Node.ATTRIBUTE_NODE ) {
                    String strvisible = attribnode.getNodeValue();

                    if ( "true".equals( strvisible ) ) {
                        bVisible = true;
                    }
                }
            }

            attribnode = attributes.getNamedItem( "orientation" );
            String strorientation = null;

            if ( attribnode != null ) {
                if ( attribnode.getNodeType() == Node.ATTRIBUTE_NODE ) {
                    strorientation = attribnode.getNodeValue();
                }
            }

            Language lparent = null;
            if ( m_languagesmap.containsKey( parentlocale ) ) {
                lparent = (Language) m_languagesmap.get( parentlocale );
            }
            else {
                // create an empty parent language
                lparent = new Language( parentlocale, null, false, null );
                m_languagesmap.put( lparent.getLocale(), lparent );
            }

            if (! m_languagesmap.containsKey( newlocale ) ) {
                lret = new Language( newlocale, lparent, bVisible, strorientation );
                m_languagesmap.put( lret.getLocale(), lret );
            }
            else {
                lret = (Language) m_languagesmap.get( newlocale );
                lret.setVisible( bVisible );
                lret.setParent( lparent );
                lret.setOrientation( strorientation );
            }
        }

        return lret;
    }

    /**
     * Handles the loading of a language node.
     *
     * @param langnode node to handle
     */
    private void handleLanguageNode( Node langnode ) {
        if ( langnode == null )
            return;

        // construct the locale from the nodes attributes and create a new Language object
        Language newLan = createLanguageFromAttributes( ((Element) langnode).getAttributes() );

        // now load all phrases for this language as well as all sublanguages
        NodeList listOfChildren = ((Element) langnode).getChildNodes();

        int numChildren = listOfChildren.getLength();

        for ( int i = 0; i < numChildren; i ++ ) {

            Node childnode = listOfChildren.item( i );

            // check if this node is an element
            if ( childnode.getNodeType() == Node.ELEMENT_NODE ){
                String nodename = ((Element) childnode).getNodeName();

                if ( "phrase".equals( nodename ) ) {
                    handlePhraseNode( newLan, childnode );
                }
            }
        }
    }

    /**
     * Loads the data from an input source connected to a language definition. Files specified by the
     * <code>languageinclude</code> tag will be parsed here as well.
     *
     * @param input input source of the language definition
     * @throws LanguageLoaderException if some error occured, while accessing the provided file name
     */
    private void loadLanguagesFromInputSource( InputSource input ) throws LanguageLoaderException {

        Document document = null;

        // Create Document with Sun's parser
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();

            document = builder.parse( input );
        }
        catch (SAXParseException spe) {
            throw new LanguageLoaderException( "Parsing error: Line: " + spe.getLineNumber() + ", uri " + spe.getSystemId() + " Extended information: " + spe.getMessage() + ", " + spe.toString() );
        }
        catch ( SAXException e ) {
            throw new LanguageLoaderException( e.getMessage() + ", " + e.toString() );
        }
        catch (ParserConfigurationException pce) {
            // Parser with specified options can't be built
            throw new LanguageLoaderException( "Parser configuration error: " + pce.getMessage() + ", " + pce.toString() );
        }
        catch (IOException e) {
            throw new LanguageLoaderException( e.getMessage() + ", " + e.toString() );
        }

        // process files that should be included
        NodeList listOfFiles = document.getElementsByTagName("languageinclude");

        for (int i = 0; i < listOfFiles.getLength(); i ++ ) {
            Node filenode = listOfFiles.item( i );

            // check if this node is an element
            if ( filenode.getNodeType() == Node.ELEMENT_NODE ){
                Node attribnode = ((Element) filenode).getAttributes().getNamedItem( "filename" );
                String strfile = null;

                if ( attribnode != null ) {
                    if ( attribnode.getNodeType() == Node.ATTRIBUTE_NODE ) {
                        strfile = attribnode.getNodeValue();
                    }
                }

                if ( strfile != null ) {
                    try {
                        if ( ( m_context != null ) && ( !strfile.startsWith( "file:" ) ) ) {
                            URL url = m_context.getResource( strfile );
                            loadLanguagesFromUrl( url );
                        }
                        else {
                            loadLanguagesFromFile( strfile );
                        }
                    }
                    catch ( Exception e ) {
                        throw new LanguageLoaderException( e.getMessage() );
                    }
                }
            }
        }

        // find all languages in this file
        NodeList listOfLanguages = document.getElementsByTagName("language");

        if ( m_languagesmap == null)
            m_languagesmap = new HashMap( );

        for (int i = 0; i < listOfLanguages.getLength(); i ++ ) {
            Node langnode = listOfLanguages.item( i );

            // check if this node is an element
            if ( langnode.getNodeType() == Node.ELEMENT_NODE ){
                handleLanguageNode( langnode );
            }
        }

    }

    /**
     * Returns the number of languages which have been loaded from the language definition.
     *
     * @return int - number of currently loaded languages
     */
    public int numberOfLanguages() {
        if ( m_languagesmap == null )
            return 0;

        return m_languagesmap.size();
    }

    /**
     * Returns an iterator of Language objects which have constructed from the language definition.
     *
     * @return Iterator - iterator of all Language objects or <code>null</code>
     */
    public Iterator getLanguages() {
        if ( m_languagesmap == null )
            return null;
        if ( m_languagesmap.values() == null )
            return null;
        return m_languagesmap.values().iterator();
    }

    /**
     * Returns the Language object associated with the provided Locale.
     *
     * @param llan the <code>Locale</code> object specifying the desired language
     * @return Language - found language or <code>null</code>
     */
    public Language getLanguage( Locale llan ) {
        if ( ( m_languagesmap == null ) || ( llan == null ) )
            return null;

        Language leRet = null;

        if ( llan.getCountry() == null ) {
            leRet = (Language)m_languagesmap.get( llan );
        }
        else {
            leRet = (Language)m_languagesmap.get( llan );

            if ( leRet == null ) {
                leRet = (Language)m_languagesmap.get( llan );
            }
        }

        return leRet;
    }

}
