/*
 * Language.java
 *
 * Created on 10. Mai 2002, 21:08
 */

package de.fzi.wim.kaonportal.multilingual;
import java.util.Locale;

/**
 * Contains all phrases of a certain language. And some additional information
 * about the language itself (<code>Locale</code> object, visible state for the
 * list of languages, the output encoding and the orientation). Query the <code>LanguageLoader</code>
 * class to get an object of this type.
 *
 * @author Tammo Riedinger
 * @version 1.0
 * @see de.fzi.wim.kaonportal.multilingual.LanguageLoader
 */
public class Language {

    /** This variable holds the orientation that should be used with this language.*/
    protected String m_strOrientation = null;

    /** This variable indicates whether a language should be displayed
     * in the list of languages.
     */
    protected boolean m_bVisible = true;

    /** This variable holds the Locale to identify this
     * Language. The hashcode of this language will be calculated
     * from this variable.
     */
    protected java.util.Locale m_Locale;

    /** This variable holds the parent language
     * of this language. In this case parent means
     * a more general language which will be
     * used to find the phrase for a given key. The
     * last possible parent, which itself does not
     * have a parent will be the default language.
     */
    protected Language m_Parent;

    /** This variable holds all phrases belonging to this language.
     * The getPhrase() and setPhrase() functions will use the key
     * for this hashmap for the key parameters.
     */
    protected java.util.HashMap m_Contents;

    /**
     * Creates a new language element.
     *
     * @param locale locale identifying this language
     * @param parent parent language of this language
     * @param bVisible indicates whether this language shall be displayed in the list of languages or not
     * @param strOrientation orientation that should be used with this language
     */
    public Language( java.util.Locale locale, Language parent, boolean bVisible, String strOrientation ) {
        m_Locale = locale;
        m_Parent = parent;
        m_bVisible = bVisible;
        m_strOrientation = strOrientation;

        m_Contents = new java.util.HashMap();
    }

    /**
     * Calculates the hashcode for this language object. The
     * hashcode is soleley dependent on the local of a language object.
     *
     * @return int - hashcode identifying this language
     */
    public int hashCode() {
        if ( m_Locale == null )
            return 0;

        return m_Locale.hashCode();
    }

    /**
     * Compares two Language objects, solely depending on the locale
     * of both objects.
     *
     * @param obj object to compare this Language with
     * @return boolean - <code>true</code> if two languages are identical, <code>false</code> otherwise
     */
    public boolean equals(java.lang.Object obj) {
        // return false if obj is not a Language Object
        if ( obj == null )
            return false;
        if ( !( obj instanceof de.fzi.wim.kaonportal.multilingual.Language ) )
            return false;

        // compare the locale objects
        de.fzi.wim.kaonportal.multilingual.Language other = (de.fzi.wim.kaonportal.multilingual.Language) obj;
        if ( this.getLocale() == null ) {
            if ( other.getLocale() != null )
                return false;
        }
        else {
            return this.getLocale().equals( other.getLocale() );
        }

        return true;
    }

    /**
     * Adds a phrase to this Language, providing the key to find that
     * phrase again later. An already existing phrase with the same key will be overwritten.
     *
     * @param key key to locate the inserted phrase later
     * @param phrase phrase that is to be inserted
     * @return boolean - <code>true</code> if phrase has been added, <code>false</code> otherwise
     */
    public boolean addPhrase(java.lang.String key, java.lang.String phrase) {
        if ( ( m_Contents == null ) || ( key == null ) || ( phrase == null ) )
            return false;

        m_Contents.put( key, phrase );

        return true;
    }

    /**
     * Returns the phrase for the provided key. Returns <code>key</code>
     * if the key cannot be found neither in this Language object
     * nor its parents.
     *
     * @param key key to the desired phrase
     * @return String - found phrase or null.
     */
    public java.lang.String getPhrase(java.lang.String key) {
        String strRet = null;

        if ( m_Contents == null )
            return null;

        if ( m_Contents.containsKey( key ) == true ) {
            strRet = (java.lang.String) m_Contents.get( key );
        }
        else {
            // check the parent
            if ( m_Parent == null )
                return key;

            strRet = m_Parent.getPhrase( key );
        }

        return strRet;
    }

    /**
     * Checks if this language object contains the phrase
     * belonging to the provided key. If the key cannot be
     * found in this object, then the parents will be checked.
     *
     * @param key key to the desired phrase
     * @return String - language name of the object.
     */
    public boolean containsPhrase(java.lang.String key) {
        if ( ( key == null ) || ( m_Contents == null ) )
            return false;

        if ( m_Contents.containsKey( key ) == true )
            return true;

        if ( m_Parent == null )
            return false;

        return m_Parent.containsPhrase( key );
    }

    /**
     * Returns the display name of this language object
     * depending in the supplied Locale. In the language definition file
     * the display name for all languages can be specified by inserting
     * a tag <code>languagedef_displayname_xx_YY</code> where xx is the language code
     * and YY the country code of the <code>Locale</code> object.
     * For the german language you could for example specify display names of English, English (US), German and Spanish in the following way:<br>
     * <code><pre>
     * <phrase name="languagedef_displayname_en" value="Englisch"/>
     * <phrase name="languagedef_displayname_en_US" value="Englisch (Vereinigte Staaten von Amerika)"/>
     * <phrase name="languagedef_displayname_de" value="Deutsch"/>
     * <phrase name="languagedef_displayname_es" value="Spanisch"/>
     * </pre></code><br>
     *
     * @param loutLan Locale to specify the language for the displayname
     * @return String - language name of the object.
     */
    public String getDisplayName( Locale loutLan ) {
        if ( loutLan != null ) {
            String strKey = "languagedef_displayname";

            if ( ( loutLan.getLanguage() != null ) && ( loutLan.getLanguage().length() > 0 ) ) {
                strKey = strKey + "_" + loutLan.getLanguage();
            }

            if ( ( loutLan.getCountry() != null ) && ( loutLan.getCountry().length() > 0 ) ) {
                strKey = strKey + "_" + loutLan.getCountry();
            }

            String strRet = getPhrase( strKey );

            if ( strRet.equals( strKey ) ) {
                return null;
            }

            return strRet;
        }

        return "";
    }

    /**
     * Returns the name of this language object.
     * It is equal to the string returned by <code>Locale.getLanguage()</code>.
     *
     * @return String - language name of the object.
     */
    public String getName() {
        String strRet = new String("");

        if ( this.getLocale() != null ) {
            strRet = this.getLocale().getLanguage();
        }

        return strRet;
    }

    /**
     * Returns the country assigned to this language object.
     * It is equal to the string returned by <code>Locale.getCountry()</code>.
     *
     * @return String - country of the object.
     */
    public String getCountry() {
        String strRet = new String("");

        if ( this.getLocale() != null ) {
            strRet = this.getLocale().getCountry();
        }

        return strRet;
    }

    /**
     * Returns the Locale assigned to this language object.
     *
     * @return Locale - Locale of the object.
     */
    public java.util.Locale getLocale() {
        return m_Locale;
    }

    /**
     * Returns the visible state.
     *
     * @return boolean - should this language be displayed in the list of languages or not
     */
    public boolean getVisible() {
        return m_bVisible;
    }

    /**
     * Sets the visible state.
     *
     * @param bVisible indication if this language should displayed in the list of languages or not
     */
    public void setVisible( boolean bVisible ) {
        m_bVisible = bVisible;
    }

    /**
     * Returns the orientation. Valid strings are "rtl" (right to left), "ltr" (left to right).
     *
     * @return String - returns the orientation for this language or null
     */
    public String getOrientation() {
        return m_strOrientation;
    }

    /**
     * Sets the orientation that should be used for this language.
     *
     * @param strOrientation new orientation for this language
     */
    public void setOrientation( String strOrientation ) {
        m_strOrientation = strOrientation;
    }

    /**
     * Returns the parent of this language.
     *
     * @return Language - the parent <code>Language</code> object or null
     */
    public Language getParent() {
        return m_Parent;
    }

    /**
     * Sets the parent of this language.
     *
     * @param value new parent <code>Language</code> of this language
     */
    public void setParent( Language value ) {
        m_Parent = value;
    }
}
