package de.fzi.wim.kaonportal.logging;

import java.io.IOException;
import java.util.Locale;
import java.util.Collection;
import java.util.List;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Date;
import java.util.Collections;
import java.util.regex.Pattern;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;

import de.fzi.wim.kaonportal.OntologyInfo;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;



/**
 * This class is a logger implementation.
 *
 * @author  Tammo Riedinger
 * @author  Ljiljana Stojanovic
 * @version 1.0
 */
public class OntologyEventLogger implements EventLogger {
    /** The URI of the usage log model. */
    public static final String USAGE_OIMODEL_URI="http://kaon.semanticweb.org/2002/07/kaon-usage";
    /** The namespace for the usage log. */
    public static final String USAGENS=USAGE_OIMODEL_URI+"#";

    static {
        try {
            KAONManager.registerWellKnownRDFOIModel(OntologyEventLogger.class.getResource("res/kaon-usage.xml").toString());
        }
        catch (KAONException e) {
            throw new RuntimeException("Cannot initialize models.",e);
        }
    }

    /** The connection for the log. */
    protected KAONConnection m_connection;
    /** OI-model for usage tracking. */
    protected OIModel m_usageOIModelInstance;

    /**
     * Creates an instance of this class.
     */
    public OntologyEventLogger() {
    }
    /**
     * Starts up the logger.
     *
     * @param parameters            the map of parameters
     * @param physicalURI           the physical URI of the model
     * @param logicalURI            the logical URI of the model
     * @throws ServletException     thrown if there is an error
     */
    public void startup(Map parameters,String physicalURI,String logicalURI) throws ServletException {
        try {
            m_connection=KAONManager.getKAONConnection(parameters);
            try {
                m_usageOIModelInstance=m_connection.openOIModelPhysical(physicalURI);
            }
            catch (KAONException e) {
                m_usageOIModelInstance=m_connection.createOIModel(physicalURI,logicalURI);
            }
            if (m_usageOIModelInstance.getIncludedOIModel(USAGE_OIMODEL_URI)==null) {
                OIModel usageOIModel=m_connection.openOIModelLogical(USAGE_OIMODEL_URI);
                m_usageOIModelInstance.applyChanges(Collections.singletonList(new AddIncludedOIModel(usageOIModel)));
            }
        }
        catch (KAONException e) {
            throw new ServletException("Cannot open the usage log OI-model",e);
        }
    }
    /**
     * Terminates the logger.
     *
     * @throws ServletException     thrown if an error occures while interrupting the thread
     */
    public synchronized void shutDown() throws ServletException {
        try {
            if (m_connection!=null)
                m_connection.close();
            m_connection=null;
        }
        catch (KAONException e) {
            throw new ServletException("Cannot open the usage log OI-model",e);
        }
    }
    /**
     * Generic logging command.
     *
     * @param request               the request
     * @param command               command
     * @throws IOException          thrown if an error occures while logging the data
     */
    public synchronized void logCommand(HttpServletRequest request,String command) throws IOException {
        logPrepare();
        createUsageInstances(request,"Command","description",command);
    }
    /**
     * Logs the fact that the user started a search.
     *
     * @param request               the request
     * @param searchString          search string
     */
    public synchronized void logSearch(HttpServletRequest request,String searchString) {
    }
    /**
     * Logs the search results.
     *
     * @param request           the request
     * @param searchString      search string
     * @param searchResults     the search results
     * @throws IOException      thrown if an error occures while logging the data
     */
    public synchronized void logSearchResults(HttpServletRequest request,String searchString,Collection searchResults) throws IOException {
        logPrepare();
        // logs the search string
        Instance eventInstance=createUsageInstances(request,"Query","queryString",searchString);
        // logs the search results
        try {
            if (searchResults!=null) {
                List changes=new LinkedList();
                Iterator iterator=searchResults.iterator();
                while (iterator.hasNext()) {
                    String entity=((Entity)iterator.next()).getURI();
                    changes.add(new AddPropertyInstance(m_usageOIModelInstance.getProperty(USAGENS+"resultsIn"),eventInstance,entity));
                }
                changes.add(new AddPropertyInstance(m_usageOIModelInstance.getProperty(USAGENS+"numOfResults"),eventInstance,Integer.toString(searchResults.size())));
                m_usageOIModelInstance.applyChanges(changes);
                m_usageOIModelInstance.save();
            }
        }
        catch (KAONException e) {
            IOException error=new IOException("KAON exception");
            error.initCause(e);
            throw error;
        }
    }
    /**
     * Logs the fact that the user selected an object.
     *
     * @param request               the request
     * @param uri                   URI of the object that was selected
     * @throws IOException          thrown if an error occures while logging the data
     */
    public synchronized void logObjectSelected(HttpServletRequest request,String uri) throws IOException {
        logPrepare();
        createUsageInstances(request,"Browse","relatedTo",uri);
    }
    /**
     * Logs the fact that the user changed the language.
     *
     * @param request               the request
     * @param locale                new locale
     * @throws IOException          thrown if an error occures while logging the data
     */
    public synchronized void logChangeLanguage(HttpServletRequest request,Locale locale) throws IOException {
        logPrepare();
        createUsageInstances(request,"SetLanguage","currentLanguage",locale.toString());
    }
    /**
     * Logs the fact that the user has changed the ontology.
     *
     * @param request               the request
     * @param ontologyInfo          new ontology
     * @throws IOException          thrown if an error occures while logging the data
     */
    public synchronized void logChangeOntology(HttpServletRequest request,OntologyInfo ontologyInfo) throws IOException {
        logPrepare();
        if (ontologyInfo==null)
            createUsageInstances(request,"SetOntology","physicalURI","NULL");
        else
            createUsageInstances(request,"SetOntology","physicalURI",ontologyInfo.getPhysicalURI());
    }
    /**
     * Creates a new instance of the event concept.
     *
     * @param request               the request
     * @param eventConcept          event that has happened
     * @param changes               the list of changes
     * @return                      the event instance
     * @throws KAONException        thrown if an error occures
     */
    protected Instance createEventInstance(HttpServletRequest request,Concept eventConcept,List changes) throws KAONException {
        Instance eventInstance=m_usageOIModelInstance.getInstance(m_usageOIModelInstance.createNewURI());
        changes.add(new AddEntity(eventInstance));
        changes.add(new AddInstanceOf(eventConcept,eventInstance));
        changes.add(new AddPropertyInstance(m_usageOIModelInstance.getProperty(USAGENS+"performedBy"),eventInstance,getUserID(request)));
        changes.add(new AddPropertyInstance(m_usageOIModelInstance.getProperty(USAGENS+"sessionID"),eventInstance,request.getSession().getId()));
        changes.add(new AddPropertyInstance(m_usageOIModelInstance.getProperty(USAGENS+"clientIP"),eventInstance,request.getRemoteAddr()));
        Date date=new Date();
        changes.add(new AddPropertyInstance(m_usageOIModelInstance.getProperty(USAGENS+"date"),eventInstance,date.toString()));
        return eventInstance;
    }
    /**
     * Creates usage instances for given event.
     *
     * @param request               the request
     * @param event                 event that has happened
     * @param property              property that describes event
     * @param value                 property value
     * @return                      the instance created by the method
     * @throws IOException          thrown if an error occures while logging the data
     */
    protected Instance createUsageInstances(HttpServletRequest request,String event,String property,String value) throws IOException {
        try {
            List changes=new LinkedList();
            Instance eventInstance=createEventInstance(request,m_usageOIModelInstance.getConcept(USAGENS+event),changes);
            if (property.equalsIgnoreCase("queryString")) {
                Pattern pattern=Pattern.compile("\\s+");
                String[] stringSet=pattern.split(value);
                for (int i=0;i<stringSet.length;i++)
                    changes.add(new AddPropertyInstance(m_usageOIModelInstance.getProperty(USAGENS+property),eventInstance,stringSet[i]));
            }
            else
                changes.add(new AddPropertyInstance(m_usageOIModelInstance.getProperty(USAGENS+property),eventInstance,value));
            m_usageOIModelInstance.applyChanges(changes);
            m_usageOIModelInstance.save();
            return eventInstance;
        }
        catch (KAONException e) {
            IOException error=new IOException("KAON exception");
            error.initCause(e);
            throw error;
        }
    }
    /**
     * Retruns the user ID from the request.
     *
     * @param request               the request
     * @return                      the user ID
     */
    protected String getUserID(HttpServletRequest request) {
        return (String)request.getSession().getAttribute("userID");
    }
    /**
     * Called before logging.
     *
     * @throws IOException          thrown if there is an error
     */
    protected void logPrepare() throws IOException {
        try {
            m_usageOIModelInstance.refresh();
        }
        catch (KAONException e) {
            IOException error=new IOException("Error refreshing the model.");
            error.initCause(e);
            throw error;
        }
    }
}
