package de.fzi.wim.kaonportal.configuration;

import java.util.Map;
import java.util.HashMap;
import java.util.Collections;
import java.net.URL;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import org.xml.sax.InputSource;
import org.w3c.dom.*;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import org.xml.sax.SAXParseException;
import org.xml.sax.SAXException;

/**
 * This class loads data from a kaon configuration file.
 *
 * @author  Tammo Riedinger
 * @version 1.0
 */
public class ConfigurationReader {
    /** URL of the configuration. */
    protected URL m_url;
    /** File of the configuration. */
    protected File m_file;
    /** Listener for configuration events. */
    protected ConfigurationListener m_listener;

    /**
     * Creates a new ConfigurationReader.
     *
     * @param url URL of the configuration file
     * @param listener <code>ConfigurationListener</code> which receives the loaded data
     */
    public ConfigurationReader(URL url,ConfigurationListener listener) {
        m_url=url;
        m_listener=listener;
    }

    /**
     * Creates a new ConfigurationReader.
     *
     * @param file input stream of configuration
     * @param listener <code>ConfigurationListener</code> which receives the loaded data
     */
    public ConfigurationReader( File file, ConfigurationListener listener) {
        m_file=file;
        m_listener=listener;
    }

    /**
     * Reads configuration.
     * @throws ConfigurationReaderException if some error occured, while reading from the inputstream
     */
    public void readConfiguration() throws ConfigurationReaderException {
        InputSource inputSource=null;
        try {
            if (m_url!=null)
                inputSource=new InputSource(m_url.openStream());
        }
        catch (IOException e) {
            throw new ConfigurationReaderException("Cannot open URL "+m_url,e);
        }
        try {
            if (m_file!=null)
                inputSource=new InputSource(new FileInputStream(m_file));
        }
        catch (IOException e) {
            throw new ConfigurationReaderException("Cannot open file "+m_file,e);
        }
        if (inputSource==null)
            throw new ConfigurationReaderException("Invalid configuration specification.");
        loadConfigurationFromInputSource(inputSource);
    }

    /**
     * Handles the loading of an ontology node.
     *
     * @param ontologynode node to handle
     */
    private void handleOntologyNode( Node ontologynode ) throws ConfigurationReaderException {
        // find "name" attribute and use it as the key
        NamedNodeMap attrs = ontologynode.getAttributes();

        String strName = null;
        Node namenode = attrs.getNamedItem( "name" );
        if ( namenode != null  && namenode.getNodeType() == Node.ATTRIBUTE_NODE ) {
            strName = namenode.getNodeValue();
        }

        String strUri = null;
        Node urinode = attrs.getNamedItem( "physicalURI" );
        if ( urinode != null && urinode.getNodeType() == Node.ATTRIBUTE_NODE ) {
            strUri = urinode.getNodeValue();
        }

        boolean computeSimilarity = false;
        Node smilaritynode = attrs.getNamedItem( "similarity" );
        if ( smilaritynode != null && smilaritynode.getNodeType() == Node.ATTRIBUTE_NODE ) {
            computeSimilarity = "true".equalsIgnoreCase(smilaritynode.getNodeValue());
        }

        boolean isDefault = false;
        Node defaultnode = attrs.getNamedItem( "default" );
        if ( defaultnode != null && defaultnode.getNodeType() == Node.ATTRIBUTE_NODE ) {
            isDefault = "true".equalsIgnoreCase(defaultnode.getNodeValue());
        }

        String defaultLanguage = null;
        Node langnode = attrs.getNamedItem( "language" );
        if ( langnode != null && langnode.getNodeType() == Node.ATTRIBUTE_NODE ) {
            defaultLanguage = langnode.getNodeValue();
        }

        Map connectionParameters=getConnectionParameters(ontologynode);
        Map usageLogParameters=Collections.EMPTY_MAP;
        String usageLogLogicalURI="";
        String usageLogPhysicalURI="";
        String eventLoggerClass="de.fzi.wim.kaonportal.logging.NullEventLogger";

        // try to see which logger should be installed
        NodeList children=ontologynode.getChildNodes();
        for (int i=0;i<children.getLength();i++) {
            Node child=children.item(i);
            if (child.getNodeType()==Node.ELEMENT_NODE && "eventlogger".equals(child.getNodeName())) {
                Element eventLoggerNode=(Element)child;
                usageLogLogicalURI=eventLoggerNode.getAttribute("usageLogLogicalURI");
                usageLogPhysicalURI=eventLoggerNode.getAttribute("usageLogPhysicalURI");
                if (eventLoggerNode.hasAttribute("classname"))
                    eventLoggerClass=eventLoggerNode.getAttribute("classname");
                usageLogParameters=getConnectionParameters(eventLoggerNode);
                break;
            }
        }
        m_listener.ontologyDefinition(strName,strUri,defaultLanguage,connectionParameters,computeSimilarity,isDefault,eventLoggerClass,usageLogLogicalURI,usageLogPhysicalURI,usageLogParameters);
    }

    /**
     * Returns the map of parameters that are read from the children of given node.
     *
     * @param node                  the node
     * @return                      the map of parameters
     */
    private Map getConnectionParameters(Node node) {
        Map connectionParameters=new HashMap();
        NodeList listOfChildren = node.getChildNodes();
        int numChildren = listOfChildren.getLength();
        for ( int i = 0; i < numChildren; i ++ ) {
            Node childnode = listOfChildren.item( i );
            if (childnode.getNodeType()==Node.ELEMENT_NODE && "connparam".equals(childnode.getNodeName())) {
                Element element=(Element)childnode;
                String key=element.getAttribute("key");
                if (key.length()!=0) {
                    String value=element.getAttribute("value");
                    connectionParameters.put(key,value);
                }
            }
        }
        return connectionParameters;
    }


    /**
     * Handles the loading of an ontologies node.
     *
     * @param ontologiesnode node to handle
     */
    private void handleOntologiesNode( Node ontologiesnode ) throws ConfigurationReaderException {
        // check all elements of the config node
        NodeList listOfChildren = ((Element) ontologiesnode).getChildNodes();

        int numChildren = listOfChildren.getLength();
        for ( int i = 0; i < numChildren; i ++ ) {

            Node childnode = listOfChildren.item( i );

            // check if this node is an element
            if ( childnode.getNodeType() == Node.ELEMENT_NODE ){
                String nodename = ((Element) childnode).getNodeName();

                if ( "ontology".equals( nodename ) ) {
                    handleOntologyNode( childnode );
                }
            }
        }
    }

    /**
     * Handles the loading of a configuration.
     *
     * @param config node to handle
     */
    private void handleConfigNode( Node confignode ) throws ConfigurationReaderException {
        if ( confignode == null )
            return;

        // check all elements of the config node
        NodeList listOfChildren = ((Element) confignode).getChildNodes();

        int numChildren = listOfChildren.getLength();

        for ( int i = 0; i < numChildren; i ++ ) {

            Node childnode = listOfChildren.item( i );

            // check if this node is an element
            if ( childnode.getNodeType() == Node.ELEMENT_NODE ){
                String nodename = ((Element) childnode).getNodeName();

                if ( "ontologies".equals( nodename ) ) {
                    handleOntologiesNode( childnode );
                }
            }
        }
    }

    /**
     * Loads the data from an input source connected to a configuration file.
     *
     * @param input input source of the configuration
     * @throws ConfigurationReaderException if some error occured, while accessing the provided file name
     */
    private void loadConfigurationFromInputSource( InputSource input ) throws ConfigurationReaderException {

        Document document = null;

        // Create Document with Sun's parser
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();

            document = builder.parse( input );
        }
        catch (SAXParseException e) {
            throw new ConfigurationReaderException( "Parsing error: Line: " + e.getLineNumber() + ", uri " + e.getSystemId(), e);
        }
        catch ( SAXException e ) {
            throw new ConfigurationReaderException( e.getMessage(), e);
        }
        catch (ParserConfigurationException e) {
            // Parser with specified options can't be built
            throw new ConfigurationReaderException( "Parser configuration error: " + e.getMessage(), e);
        }
        catch (IOException e) {
            throw new ConfigurationReaderException( e.getMessage(), e);
        }

        // read the configuration
        NodeList listOfLanguages = document.getElementsByTagName("config");

        Node confignode = listOfLanguages.item( 0 );
        // check if this node is an element
        if ( confignode.getNodeType() == Node.ELEMENT_NODE ){
            handleConfigNode( confignode );
        }
    }
}
