package de.fzi.wim.kaonportal;

import java.util.Iterator;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.ArrayList;
import java.util.HashSet;
import java.net.URLEncoder;
import java.io.UnsupportedEncodingException;
import javax.servlet.http.HttpServletResponse;

import edu.unika.aifb.kaon.api.KAONException;
import edu.unika.aifb.kaon.api.vocabulary.KAONVocabularyAdaptor;
import edu.unika.aifb.kaon.api.oimodel.Entity;
import edu.unika.aifb.kaon.api.oimodel.Concept;
import edu.unika.aifb.kaon.api.oimodel.Property;
import edu.unika.aifb.kaon.api.oimodel.Instance;

/**
 * A small class just to encapsulate the functions to output an ontology .
 * all methods are only implemented rudimentory, and should be refined in case
 * anyone wants a nice output.
 *
 * @author Valentin Zacharias
 */
public class OntologyFormatter {
    public static final String JSP_FILE="dispatcher";
    private static final String SPACER="&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
    private static final String BR="<br>";

    private OntologyAccess access;
    private String languageURI;
    private HttpServletResponse response;
    private Set filterURIs;

    public OntologyFormatter(OntologyAccess access,String language,HttpServletResponse response) {
        this.access = access;
        this.languageURI=KAONVocabularyAdaptor.INSTANCE.getLanguageURI(language);
        filterURIs=new HashSet();
        filterURIs.add(KAONVocabularyAdaptor.INSTANCE.getLanguage());
        filterURIs.add(KAONVocabularyAdaptor.INSTANCE.getReferences());
        filterURIs.add(KAONVocabularyAdaptor.INSTANCE.getLexicalEntry());
        this.response = response;
    }

    protected void filterURIs(Collection input) throws KAONException {
        Iterator iterator=input.iterator();
        while (iterator.hasNext()) {
            Entity entity=(Entity)iterator.next();
            if (filterURIs.contains(entity.getURI()))
                iterator.remove();
        }
    }

    public Iterator getRootConceptsEnum() throws KAONException {
        return getSubConceptsEnum(access.getOIModel().getRootConcept());
    }

    public Iterator getSuperConceptsEnum(Concept cp) throws KAONException {
        return enumNamed(cp.getSuperConcepts());
    }

    public Iterator getSubConceptsEnum(Concept cp) throws KAONException {
        Set set=new HashSet(cp.getSubConcepts());
        filterURIs(set);
        return enumNamed(set);
    }

    public Iterator getInstancesEnum(Concept cp) throws KAONException {
        return enumNamed(cp.getInstances());
    }

    public Iterator getIncomingPropertiesEnum(Concept cp) throws KAONException {
        Set set=new HashSet(cp.getAllPropertiesToConcept());
        filterURIs(set);
        return enumNamed(set);
    }

    public Iterator getOutgoingPropertiesNoAttributesEnum(Concept cp) throws KAONException {
        Set set=new HashSet(cp.getAllPropertiesFromConcept());
        Iterator iterator=set.iterator();
        while (iterator.hasNext()) {
            Property property=(Property)iterator.next();
            if (property.isAttribute())
                iterator.remove();
        }
        return enumNamed(set);
    }

    public Iterator getAttributesEnum(Concept cp) throws KAONException {
        Set set=new HashSet(cp.getAllPropertiesFromConcept());
        Iterator iterator=set.iterator();
        while (iterator.hasNext()) {
            Property property=(Property)iterator.next();
            if (!property.isAttribute())
                iterator.remove();
        }
        return enumNamed(set);
    }

    public Iterator getDomain(Property prop) throws KAONException {
        return enumNamed(prop.getDomainConcepts());
    }

    public Iterator getRange(Property prop) throws KAONException {
        if (prop.isAttribute()) {
            Set set=new HashSet();
            set.add("Text");
            return set.iterator();
        }
        else
            return enumNamed(prop.getRangeConcepts());
    }

    public Iterator getConceptsEnum(Instance inst) throws KAONException {
        Set set=new HashSet(inst.getParentConcepts());
        set.remove(access.getOIModel().getRootConcept());
        return enumNamed(set);
    }

    public Iterator getSimilarEnum(Instance inst) throws KAONException {
        if (inst.getParentConcepts().size()==1 && inst.getParentConcepts().contains(access.getOIModel().getRootConcept()))
            return Collections.EMPTY_SET.iterator();
        else {
            Set set=new HashSet(access.getSimilar(inst));
            filterURIs(set);
            return enumNamed(set);
        }
    }

    public Iterator getInstanceAttributes(Instance inst) throws KAONException {
        Map map = inst.getFromPropertyValues();
        return getInstanceAttributes(map);
    }

    public Iterator getAllInstanceAttributes(Instance inst) throws KAONException {
        Map map = access.getAllFromPropertyValues(inst);
        return getInstanceAttributes(map);
    }

    protected Iterator getInstanceAttributes(Map map) throws KAONException {
        Iterator it = map.keySet().iterator();
        List toReturn=new ArrayList();
        while (it.hasNext()) {
            Property property = (Property)it.next();
            if (property.isAttribute()) {
                String propertyFormatted=formatNamed(property);
                Iterator values = ((Set)map.get(property)).iterator();
                while (values.hasNext()) {
                    Object value = values.next();
                    if (value instanceof String) {
                        Tupel tp = new Tupel();
                        tp.a = propertyFormatted;
                        tp.b = value;
                        toReturn.add(tp);
                    }
                }
            }
        }
        return toReturn.iterator();
    }

    public Iterator getFromRelatedEnum(Instance inst) throws KAONException {
        Map map = inst.getFromPropertyValues();
        return getFromRelatedEnum(map);
    }

    public Iterator getAllFromRelatedEnum(Instance inst) throws KAONException {
        Map map = access.getAllFromPropertyValues(inst);
        return getFromRelatedEnum(map);
    }

    public Iterator getFromRelatedEnum(Map map) throws KAONException {
        Iterator it = map.keySet().iterator();
        List toReturn=new ArrayList();
        while (it.hasNext()) {
            Property property = (Property)it.next();
            if (!property.isAttribute() && !filterURIs.contains(property.getURI())) {
                Tupel tp = new Tupel();
                tp.a = formatNamed(property);
                tp.b = enumNamed((Set)map.get(property));
                toReturn.add(tp);
            }
        }
        return toReturn.iterator();
    }

    public Iterator enumNamed(Collection entities) {
        List list = new ArrayList(entities.size());
        Iterator iterator=entities.iterator();
        while (iterator.hasNext()) {
            Entity entity=(Entity)iterator.next();
            list.add(formatNamed(entity));
        }
        return list.iterator();
    }

    public String getLexicalAttribute(Entity entity,String typeURI) {
        String lexicalAttribute=null;
        try {
            lexicalAttribute=entity.getLexicalAttribute(typeURI,languageURI,KAONVocabularyAdaptor.INSTANCE.getValue());
            if (lexicalAttribute==null)
                lexicalAttribute=entity.getLexicalAttribute(typeURI,KAONVocabularyAdaptor.INSTANCE.getLanguageURI("en"),KAONVocabularyAdaptor.INSTANCE.getValue());
        }
        catch (KAONException ignored) {
        }
        try {
            if (lexicalAttribute==null)
                lexicalAttribute=entity.getURI();
        }
        catch (KAONException ignored) {
        }
        if (lexicalAttribute==null)
            lexicalAttribute="no-name";
        return lexicalAttribute;
    }

    public OntologyAccess getOntologyAccess() {
        return access;
    }

    public String getLanguageURI() {
        return languageURI;
    }

    protected String formatNamed(Entity entity) {
        try {
            return formatObject(entity.getURI(),getLexicalAttribute(entity,KAONVocabularyAdaptor.INSTANCE.getKAONLabel()));
        }
        catch (KAONException ignored) {
            return "object-reading-error";
        }
    }

    public String formatObject(String uri,String label) {
        if (label.indexOf('#')!=-1)
            label=label.substring(label.lastIndexOf('#')+1,label.length());
        if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getRoot())) {
            StringBuffer buffer = new StringBuffer();
            buffer.append("<a href=\"");
            buffer.append(JSP_FILE);
            buffer.append("?cmd_showroot=true\">");
            buffer.append(label);
            buffer.append("</a>");
            return buffer.toString();
        }
        else {
            StringBuffer href=new StringBuffer();
            href.append(JSP_FILE);
            href.append("?cmd_changeobject=true&uri=");
            try {
                href.append(URLEncoder.encode(uri,"UTF-8"));
            }
            catch (UnsupportedEncodingException cantHappen) {
                href.append("no-name");
            }
            StringBuffer buffer=new StringBuffer();
            buffer.append("<a href=\"");
            if (response!=null)
                buffer.append(response.encodeURL(href.toString()));
            else
                buffer.append(href.toString());
            buffer.append("\">");
            buffer.append(label);
            buffer.append("</a>");
            return buffer.toString();
        }
    }

    public static class Tupel {
        public Object a;
        public Object b;
    }
}
