package de.fzi.wim.kaonportal;

import java.net.URL;
import java.io.File;
import java.io.IOException;
import java.util.Locale;
import java.util.Map;
import java.util.List;
import java.util.Set;
import java.util.Iterator;
import java.util.ArrayList;
import javax.servlet.ServletException;
import javax.servlet.ServletConfig;
import javax.servlet.RequestDispatcher;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import edu.unika.aifb.kaon.api.KAONException;
import edu.unika.aifb.kaon.api.oimodel.OIModel;
import edu.unika.aifb.kaon.api.oimodel.Concept;
import edu.unika.aifb.kaon.api.oimodel.Property;
import edu.unika.aifb.kaon.api.oimodel.Instance;

import de.fzi.wim.kaonportal.multilingual.Language;
import de.fzi.wim.kaonportal.multilingual.LanguageLoader;
import de.fzi.wim.kaonportal.configuration.ConfigurationReader;
import de.fzi.wim.kaonportal.configuration.ConfigurationListener;
import de.fzi.wim.kaonportal.configuration.ConfigurationReaderException;


/**
 * This class is the servlet class of the portal application. All request should be
 * processed by this class first. The global environment of the application will be
 * initialized and managed in this class as well. It will load the supported languages and
 * ontologies. This class has to be the entry point for each session request. Otherwise
 * the JSPs and tags do not find the needed environment to function correctly.
 *
 * @author  Tammo Riedinger
 */
public class Dispatcher extends HttpServlet {
    /**
     * This method initializes the servlet. It loads the language definition file
     * and creates the <code>OntologyStore</code> object.
     *
     * @param config initialization information for the servlet
     * @throws ServletException will be thrown if some error occures
     */
    public void init(ServletConfig config) throws ServletException {
        super.init(config);
        OntologyFormatterFactory ontologyFormatterFactory=createOntologyFormatterFactory();
        getServletContext().setAttribute("ontologyformatterfactory", ontologyFormatterFactory);
        final OntologyStore store = new OntologyStore();
        getServletContext().setAttribute("ontologystore", store);
        // create a configuration listener
        ConfigurationListener listener=new ConfigurationListener() {
            public void ontologyDefinition(String name,String uri,String defaultLanguage,Map connectionParameters,boolean computeSimilarity,boolean isDefault,String eventLoggerClass,String usageLogLogicalURI,String usageLogPhysicalURI,Map usageLogParameters) throws ConfigurationReaderException {
                try {
                    store.addOntology(name,uri,defaultLanguage,connectionParameters,computeSimilarity,isDefault,eventLoggerClass,usageLogLogicalURI,usageLogPhysicalURI,usageLogParameters);
                }
                catch (KAONException e) {
                    throw new ConfigurationReaderException("Error initializing ontology",e);
                }
            }
        };
        // extract the file name to the language definition file from the servlet context
        String strlangdeffile=getServletContext().getInitParameter("language_def");
        try {
            // load the file and save it in the context
            LanguageLoader lanLoader=new LanguageLoader(config.getServletContext(), strlangdeffile);
            getServletContext().setAttribute("languageloader", lanLoader);
        }
        catch (LanguageLoader.LanguageLoaderException e) {
            throw new ServletException("Cannot initialize language loader",e);
        }

        // extract the root of the configuration
        String strconfigroot=getServletContext().getInitParameter("configuration_root");
        // extract the file name to the configuration file from the servlet context
        String strconfigfile=getServletContext().getInitParameter("configuration_file");
        ConfigurationReader configReader=null;
        try {
            if ("/WEB-INF".equals(strconfigroot)) {
                URL configurationURL = config.getServletContext().getResource("/WEB-INF" + strconfigfile);
                configReader = new ConfigurationReader(configurationURL,listener);
            }
            else {
                String strenv = Environment.getEnvironmentVariable(strconfigroot);
                configReader = new ConfigurationReader(new File(strenv+strconfigfile),listener);
            }
        }
        catch (Exception e) {
            throw new ServletException("Cannot read configuration",e);
        }
        if (configReader==null) {
            throw new ServletException( "Cannot initialize configuration reader." );
        }
        try {
            configReader.readConfiguration();
        }
        catch (ConfigurationReaderException e) {
            throw new ServletException("Cannot read configuration", e);
        }
        if (store.getOntologies().size()==0)
            throw new ServletException("KAON Portal cannot start: no ontologies have been loaded.");
    }
    /**
     * Called when the server is being terminated.
     *
     * @throws ServletException
     */
    public void destroy() {
        super.destroy();
        OntologyStore ontologyStore=(OntologyStore)getServletContext().getAttribute("ontologystore");
        ontologyStore.uninitialize();
    }

    /**
     * This method processes requests for both HTTP <code>GET</code> and <code>POST</code> methods.
     * It will also evaluate the request parameters and manipulate the current application state.
     *
     * @param request servlet request
     * @param response servlet response
     * @throws ServletException will be thrown if some other error occures
     * @throws IOException will be thrown if an io exception occures
     */
    protected void processRequest(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        HttpSession session = request.getSession(true);

        // set to anonymous user if no user has been set
        if (session.getAttribute("userID") == null) {
            session.setAttribute("userID", "ANONYMOUS_USER");
        }

        // load the default ontology if no ontology has been set
        if (session.getAttribute("ontologyinfo") == null) {
            OntologyStore store = (OntologyStore) getServletContext().getAttribute("ontologystore");
            OntologyInfo ontologyInfo=store.getDefaultOntology();
            if (ontologyInfo!=null) {
                try {
                    ontologyInfo.initialize();
                }
                catch (KAONException e) {
                    throw new ServletException("Cannot initialize default ontology",e);
                }
                session.setAttribute("ontologyinfo", ontologyInfo);
            }
        }

        // set default language if no language has been set
        if (session.getAttribute("language") == null) {
            Locale locale;
            OntologyInfo ontologyInfo=(OntologyInfo)session.getAttribute("ontologyinfo");
            if (ontologyInfo!=null)
                locale=new Locale(ontologyInfo.getDefaultLanguage(),"");
            else
                locale=new Locale("en","");
            LanguageLoader lanload = (LanguageLoader) getServletContext().getAttribute("languageloader");
            Language language = lanload.getLanguage(locale);
            if (language == null) {
                throw new ServletException("Cannot load default language.");
            }
            session.setAttribute("language", language);
        }

        // prepare the response
        response.setContentType("text/html; charset=UTF-8");

        // process commands
        CommandProcessor processor=createCommandProcessor(session,request,response);
        processor.processCommands();

        try {
            setUpOntologyAccess(session,request);

            // generate response
            String url = (String) request.getAttribute("url");
            if (url == null) url = "viewer.jsp";
            RequestDispatcher dispatcher=request.getRequestDispatcher(url);
            dispatcher.forward(request,response);
        }
        catch (KAONException e) {
            throw new ServletException("Cannot set up ontology access",e);
        }
        finally {
            OntologyAccess ontologyAccess=(OntologyAccess)request.getAttribute("ontologyaccess");
            if (ontologyAccess!=null)
                ontologyAccess.close();
        }
    }

    /**
     * This method handles the HTTP <code>GET</code> method.
     *
     * @param request servlet request
     * @param response servlet response
     * @throws ServletException will be thrown if some other error occures
     * @throws IOException will be thrown if an io exception occures
     */
    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        processRequest(request, response);
    }

    /**
     * This method handles the HTTP <code>POST</code> method.
     *
     * @param request servlet request
     * @param response servlet response
     * @throws ServletException will be thrown if some other error occures
     * @throws IOException will be thrown if an io exception occures
     */
    protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        processRequest(request, response);
    }

    /**
     * This method returns a short description of the servlet.
     *
     * @return description string for this servlet
     */
    public String getServletInfo() {
        return "KAON Portal Dispatcher";
    }

    /**
     * Sets up the current ontology access.
     *
     * @param session the session
     * @param request the request
     * @throws KAONException thrown if access cannot be set up
     */
    protected void setUpOntologyAccess(HttpSession session,HttpServletRequest request) throws KAONException {
        OntologyInfo ontologyInfo=(OntologyInfo)session.getAttribute("ontologyinfo");
        if (ontologyInfo==null)
            return;
        OntologyAccess access=ontologyInfo.createAccess();
        request.setAttribute("ontologyaccess",access);
        String currentURI=(String)session.getAttribute("currentURI");
        if (currentURI!=null) {
            Concept concept=access.getConcept(currentURI);
            Property property=access.getProperty(currentURI);
            Instance instance=access.getInstance(currentURI);
            List objects=new ArrayList(3);
            objects.add(concept);
            objects.add(property);
            objects.add(instance);
            access.getOIModel().loadObjects(objects,OIModel.LOAD_CONCEPT_ALL | (OIModel.LOAD_PROPERTY_ALL & ~OIModel.LOAD_PROPERTY_INSTANCES) | OIModel.LOAD_INSTANCE_ALL | OIModel.LOAD_LEXICON);
            objects=new ArrayList();
            if (concept.isInOIModel()) {
                request.setAttribute("currentConcept",concept);
                objects.addAll(concept.getSuperConcepts());
                objects.addAll(concept.getSubConcepts());
                objects.addAll(concept.getPropertiesFromConcept());
                objects.addAll(concept.getPropertiesToConcept());
                objects.addAll(concept.getInstances());
            }
            if (property.isInOIModel()) {
                request.setAttribute("currentProperty",property);
                objects.addAll(property.getDomainConcepts());
                objects.addAll(property.getRangeConcepts());
                objects.addAll(property.getSuperProperties());
                objects.addAll(property.getSubProperties());
            }
            if (instance.isInOIModel()) {
                request.setAttribute("currentInstance",instance);
                objects.addAll(instance.getParentConcepts());
                addInstances(objects,instance.getFromPropertyValues());
                addInstances(objects,instance.getToPropertyValues());
            }
            if (!objects.isEmpty())
                access.getOIModel().loadObjects(objects,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_INSTANCE_BASICS | OIModel.LOAD_LEXICON);
        }
        else {
            Concept root=access.getOIModel().getRootConcept();
            // store a strong reference to the root somewhere, so that it won't be garbage-collected accidentally
            request.setAttribute("rootConcept",root);
            List objects=new ArrayList(1);
            objects.add(root);
            access.getOIModel().loadObjects(objects,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUB_CONCEPTS | OIModel.LOAD_LEXICON);
            objects=new ArrayList();
            objects.addAll(root.getSubConcepts());
            if (!objects.isEmpty())
                access.getOIModel().loadObjects(objects,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_INSTANCE_BASICS | OIModel.LOAD_LEXICON);
        }
    }
    /**
     * Adds all instances from the map of values to the collection.
     *
     * @param list the collection being loaded
     * @param valuesMap map of values
     */
    protected void addInstances(List list,Map valuesMap) {
        Iterator sets=valuesMap.values().iterator();
        while (sets.hasNext()) {
            Iterator values=((Set)sets.next()).iterator();
            while (values.hasNext()) {
                Object value=values.next();
                if (value instanceof Instance)
                    list.add(value);
            }
        }
    }
    /**
     * Subclasses can override this method to provide a different formatter.
     *
     * @return the ontology formatter factory
     */
    protected OntologyFormatterFactory createOntologyFormatterFactory() {
        return new OntologyFormatterFactory() {
            public OntologyFormatter createOntologyFormatter(OntologyAccess access,String language,HttpServletResponse response) {
                return new OntologyFormatter(access,language,response);
            }
        };
    }
    /**
     * Creates a command processor.
     *
     * @param session the session
     * @param request the request
     * @param response the response
     * @return the command processor
     */
    protected CommandProcessor createCommandProcessor(HttpSession session,HttpServletRequest request,HttpServletResponse response) {
        return new CommandProcessor(getServletContext(),session,request,response);
    }

    /**
     * The creator for the ontology formatter. An instance of this class is registered into the application scope.
     */
    public static interface OntologyFormatterFactory {
        OntologyFormatter createOntologyFormatter(OntologyAccess access,String language,HttpServletResponse response);
    }
}
