package de.fzi.wim.kaonportal;

import java.io.IOException;
import java.util.Enumeration;
import java.util.Locale;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;

import edu.unika.aifb.kaon.api.KAONException;

import de.fzi.wim.kaonportal.multilingual.Language;
import de.fzi.wim.kaonportal.multilingual.LanguageLoader;
import de.fzi.wim.kaonportal.logging.EventLogger;
import de.fzi.wim.kaonportal.logging.NullEventLogger;
import de.fzi.wim.kaonportal.tree.Tree;
import de.fzi.wim.kaonportal.tree.TreeException;

/**
 * This class is used by the servlet to process commands.
 */
public class CommandProcessor {
    /** No parameters. */
    protected static final Object[] NO_PARAMETERS=new Object[0];

    /** Current servlet context. */
    protected ServletContext m_servletContext;
    /** Current HTTP session. */
    protected HttpSession m_session;
    /** Current request. */
    protected HttpServletRequest m_request;
    /** Current response. */
    protected HttpServletResponse m_response;
    /** Event logger. */
    protected EventLogger m_logger;

    /**
     * Creates an instance of this class.
     *
     * @param servletContext            current servlet context
     * @param session                   current session
     * @param request                   current request
     * @param response                  current response
     */
    public CommandProcessor(ServletContext servletContext,HttpSession session,HttpServletRequest request,HttpServletResponse response) {
        m_servletContext=servletContext;
        m_session=session;
        m_request=request;
        m_response=response;
        OntologyInfo ontologyInfo=(OntologyInfo)m_session.getAttribute("ontologyinfo");
        if (ontologyInfo==null)
            m_logger=NullEventLogger.INSTANCE;
        else
            m_logger=ontologyInfo.getEventLogger();
    }
    /**
     * This method processes the <code>showroot</code> command.
     * @throws IOException if some i/o error occured, while processing the command
     */
    public void cmd_showroot() throws IOException {
        m_logger.logCommand(m_request,"cmd_showroot");
        m_session.removeAttribute("currentURI");
        m_session.removeAttribute("currentSearchString");
    }
    /**
     * This method processes the <code>search</code> command.
     * @throws IOException if some i/o error occured, while processing the command
     */
    public void cmd_search() throws IOException {
        String searchString=m_request.getParameter("searchstring");
        m_session.setAttribute("currentSearchString",searchString);
        m_logger.logSearch(m_request,searchString);
    }
    /**
     * This method processes the <code>chooseontology</code> command.
     * @throws IOException if some i/o error occured, while processing the command
     */
    public void cmd_chooseontology() throws IOException {
        m_session.removeAttribute("ontologyinfo");
        m_session.removeAttribute("currentURI");
        m_session.removeAttribute("currentSearchString");
        m_logger.logChangeOntology(m_request,null);
    }
    /**
     * This method processes the <code>changeobject</code> command.
     * @throws IOException if some i/o error occured, while processing the command
     */
    public void cmd_changeobject() throws IOException {
        String uri=m_request.getParameter("uri");
        if (uri!=null && uri.length()!=0) {
            m_session.removeAttribute("currentSearchString");
            m_session.setAttribute("currentURI",uri);
            m_logger.logObjectSelected(m_request,uri);
        }
    }
    /**
     * This method processes the <code>changelanguage</code> command.
     * @throws ServletException if some error occured, while processing the command
     * @throws IOException if some i/o error occured, while processing the command
     */
    public void cmd_changelanguage() throws ServletException, IOException {
        String newLanguageName=m_request.getParameter("language");
        if (newLanguageName!=null && newLanguageName.length()!=0) {
            String newCountryName=m_request.getParameter("country");
            if (newCountryName==null)
                newCountryName="";
            LanguageLoader languageLoader=(LanguageLoader)m_servletContext.getAttribute("languageloader");
            Locale newLocale=new Locale(newLanguageName,newCountryName);
            Language newLanguage=languageLoader.getLanguage(newLocale);
            if (newLanguage==null)
                throw new ServletException("Cannot find language for locale "+newLocale);
            m_session.setAttribute("language",newLanguage);
            m_logger.logChangeLanguage(m_request,newLocale);
        }
    }
    /**
     * This method processes the <code>changeontology</code> command.
     * @throws ServletException if some error occured, while processing the command
     * @throws IOException if some i/o error occured, while processing the command
     */
    public void cmd_changeontology() throws ServletException, IOException {
        String newURI=m_request.getParameter("uri");
        OntologyStore store=(OntologyStore)m_servletContext.getAttribute("ontologystore");
        OntologyInfo ontologyInfo=store.getOntologyInfo(newURI);
        if (ontologyInfo==null)
            throw new ServletException("Ontology with URI "+newURI+" doesn't exist");
        try {
            ontologyInfo.initialize();
        }
        catch (KAONException e) {
            throw new ServletException("Cannot initialize ontology with URI "+newURI,e);
        }
        m_session.setAttribute("ontologyinfo",ontologyInfo);
        m_logger.logChangeOntology(m_request,ontologyInfo);
    }
    /**
     * This method processes the <code>login</code> command.
     * @throws IOException  if some error occured, while processing the command
     */
    public void cmd_login() throws IOException {
        String newName=m_request.getParameter("username");
        String password=m_request.getParameter("password");
        String newUserID=(String)m_session.getAttribute("userID");
        if (newName==null ) {
            m_logger.logCommand(m_request,"login as: ANONYMOUS_USER");
            newUserID="ANONYMOUS_USER";
        }
        else {
            m_logger.logCommand(m_request,"login as: "+newName);
            if (("Boris".equals(newName) && "Motik".equals(password)) || ("Alexander".equals(newName) && "Maedche".equals(password)) || ("ANONYMOUS_USER".equals(newName)))
                newUserID=newName;
            else {
                if (!"Boris".equals(newName)&& !"Alexander".equals(newName))
                    m_logger.logCommand(m_request,"user not registered: "+newName);
                else
                    m_logger.logCommand(m_request,"wrong password for user: "+newName);
            }
        }
        m_session.setAttribute("userID",newUserID);
    }
    /**
     * This method processes the <code>logout</code> command.
     * @throws IOException  if some error occured, while processing the command
     */
    public void cmd_logout() throws IOException {
        m_logger.logCommand(m_request,"logout");
        m_session.setAttribute("userID","ANONYMOUS_USER");
    }
    /**
     * This method expands the node in the tree.
     */
    public void cmd_expandtreenode() {
        String treeID=m_request.getParameter("treeid");
        String nodeID=m_request.getParameter("nodeid");
        if (treeID!=null && nodeID!=null) {
            Tree tree=(Tree)m_session.getAttribute(treeID);
            if (tree!=null)
                tree.setExpanded(nodeID,true);
        }
    }

    /**
     * Used to forward the request to a jsp page. Mostly used for editing, but could be
     * used for any other jsp page that need the ontology objects.
     */
    public void cmd_forward() {
        String url = m_request.getParameter("cmd_forward");
        m_request.setAttribute("url", url);
    }

    /**
     * This method collapses the node in the tree.
     */
    public void cmd_collapsetreenode() {
        String treeID=m_request.getParameter("treeid");
        String nodeID=m_request.getParameter("nodeid");
        if (treeID!=null && nodeID!=null) {
            Tree tree=(Tree)m_session.getAttribute(treeID);
            if (tree!=null)
                tree.setExpanded(nodeID,false);
        }
    }
    /**
     * This method refreshes the tree.
     *
     * @throws ServletException if some error occured while processing the command
     */
    public void cmd_refreshtree() throws ServletException {
        try {
            String treeID=m_request.getParameter("treeID");
            if (treeID!=null) {
                Tree tree=(Tree)m_session.getAttribute(treeID);
                if (tree!=null)
                    tree.refresh();
            }
        }
        catch (TreeException e) {
            throw new ServletException("Error refreshing the tree.",e);
        }
    }
    /**
     * This method processes all command parameters. Those parameters have to start with a "cmd_" prefix.
     * If a parameter of the form "cmd_xxx" is supplied this method tried to find a public member method of
     * this class with the name "processCommand_xxx". Such a method must have two parameters. The first one
     * of type <code>HttpServletRequest</code>, the second of type <code>HttpServletResponse</code>. If such a method
     * is found it will be invoked.
     * Commands should be used to manipulate the current application status.
     *
     * @throws ServletException will be thrown if some other error occures
     * @throws IOException will be thrown if an io exception occures
     */
    public void processCommands() throws ServletException, IOException {
        Enumeration parameters=m_request.getParameterNames();
        while (parameters.hasMoreElements()) {
            String parameterName=(String)parameters.nextElement();
            if (parameterName.startsWith("cmd_")) {
                Method method;
                try {
                    method=getClass().getMethod(parameterName,null);
                }
                catch (NoSuchMethodException e) {
                    throw new ServletException("Cannot find a command handler for command "+parameterName,e);
                }
                try {
                    method.invoke(this,NO_PARAMETERS);
                }
                catch (IllegalAccessException e) {
                    throw new ServletException("Method corresponding to the command ("+parameterName+") could not be called correctly",e);
                }
                catch (IllegalArgumentException e) {
                    throw new ServletException("Method corresponding to the command ("+parameterName+") could not be called correctly",e);
                }
                catch (InvocationTargetException e) {
                    Throwable cause=e.getCause();
                    if (cause instanceof ServletException)
                        throw (ServletException)cause;
                    else if (cause instanceof IOException)
                        throw (IOException)cause;
                    else
                        throw new ServletException("There was an error processing command "+parameterName,cause);
                }
            }
        }
    }
}
