package edu.unika.aifb.kaon.virtualoimodel;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Implementation of an virtual instance of a property.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class VirtualPropertyInstance implements PropertyInstance {
    /** Original property instance. */
    protected PropertyInstance m_propertyInstance;
    /** Property that this object is instance of. */
    protected VirtualProperty m_property;
    /** Source instance. */
    protected VirtualInstance m_sourceInstance;
    /** Target value (may be <code>VirtualInstance</code> or <code>String</code>. */
    protected Object m_targetValue;
    /** Set to <code>true</code> if this object is in the OI-model. */
    protected boolean m_isInOIModel;
    /** The source original OI-model. */
    protected OIModel m_sourceOIModelOriginal;

    /**
     * Creates an instance of this class.
     */
    public VirtualPropertyInstance() {
    }
    /** Initializes this instnace
     *
     * @param propertyInstance                  the original property instance
     * @param oimodel                           the OI-model
     * @param isInOIModel                       determines whether this property instance is in OI-model
     */
    public void initialize(PropertyInstance propertyInstance,VirtualOIModel oimodel,boolean isInOIModel) throws KAONException {
        m_propertyInstance=propertyInstance;
        m_property=oimodel.getProperty(propertyInstance.getProperty());
        m_sourceInstance=oimodel.getInstance(propertyInstance.getSourceInstance());
        Object targetValue=propertyInstance.getTargetValue();
        if (targetValue instanceof Instance)
            m_targetValue=oimodel.getInstance((Instance)targetValue);
        else
            m_targetValue=targetValue;
        m_isInOIModel=isInOIModel;
    }
    /**
     * Returns the property.
     *
     * @return                                  property
     */
    public Property getProperty() {
        return m_property;
    }
    /**
     * Returns the instance that this relation points from.
     *
     * @return                                  source instance
     */
    public Instance getSourceInstance() {
        return m_sourceInstance;
    }
    /**
     * Returns the value of this relation instnace (may be <code>String</code> or <code>Instance</code>).
     *
     * @return                                  value of this relaiton instance (<code>String</code> or <code>Instnace</code>)
     */
    public Object getTargetValue() {
        return m_targetValue;
    }
    /**
     * Returns the virtual property.
     *
     * @return                                  virtual property
     */
    public VirtualProperty getVirtualProperty() {
        return m_property;
    }
    /**
     * Returns the virtual instance that this relation points from.
     *
     * @return                                  virtual source instance
     */
    public VirtualInstance getVirtualSourceInstance() {
        return m_sourceInstance;
    }
    /**
     * Returns the original property instance.
     *
     * @return                                  original property instance
     */
    public PropertyInstance getPropertyInstance() {
        return m_propertyInstance;
    }
    /**
     * Returns the OI-model of this object.
     *
     * @return                                  the OI-model of this object
     */
    public OIModel getOIModel() {
        return m_property.getOIModel();
    }
    /**
     * Checks whether this property instance is in the OI-model.
     *
     * @return                                  <code>true</code> if this property instance is in the OI-model.
     */
    public boolean isInOIModel() {
        return m_isInOIModel;
    }
    /**
     * Returns <code>true</code> if this entity has been declared in the OI-model as returned
     * by <code>getOIModel()</code> call.
     *
     * @return                          <code>true</code> if this entity case declared in the OI-model as returned by <code>getOIModel()</code> call
     */
    public boolean isDeclaredLocally() {
        return m_isInOIModel;
    }
    /**
     * Returns the source OI-model of this object.
     *
     * @return                                  the source OI-model of this object
     */
    public OIModel getSourceOIModel() {
        if (isInOIModel())
            return m_property.getOIModel();
        else
            return null;
    }
    /**
     * Returns the source original OI-model of this object.
     *
     * @return                                  the source original OI-model of this object
     */
    public OIModel getSourceOIModelOriginal() throws KAONException {
        if (m_sourceOIModelOriginal==null && m_isInOIModel)
            m_sourceOIModelOriginal=m_propertyInstance.getSourceOIModel();
        return m_sourceOIModelOriginal;
    }
    /**
     * Determines whether the property instance is in OI-model.
     *
     * @param isInOIModel                       determines whether the property is in OI-model
     */
    void setIsInOIModel(boolean isInOIModel) {
        m_isInOIModel=isInOIModel;
        if (m_isInOIModel)
            m_sourceOIModelOriginal=((VirtualOIModel)m_property.getOIModel()).getOIModel();
        else
            m_sourceOIModelOriginal=null;
    }
}
