package edu.unika.aifb.kaon.virtualoimodel;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Implementation of the VirtualProperty
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class VirtualProperty extends VirtualEntity implements Property {
    /** Inverse property. */
    protected VirtualProperty m_inverseProperty;
    /** The original OI-model for the inverse property. */
    protected OIModel m_inversePropertyOIModelOriginal;
    /** <code>true</code> if this property is symmetric. */
    protected boolean m_symmetric;
    /** <code>true</code> if this property is transitive. */
    protected boolean m_transitive;
    /** <code>true</code> if this property is attribute. */
    protected boolean m_isAttribute;
    /** Map tracker of domain concepts. */
    protected MapTracker m_domainConcepts;
    /** Map tracker of range concepts. */
    protected MapTracker m_rangeConcepts;
    /** Map of cardinalities indexed by the concept. */
    protected Map m_cardinalities;
    /** Set tracker of subproperties. */
    protected SetTracker m_subProperties;
    /** Map tracker of superproperties. */
    protected MapTracker m_superProperties;
    /** Set tracker of property instances. */
    protected SetTracker m_propertyInstances;

    /**
     * Creates an instance of this class and attaches it to the OI-model.
     *
     * @param oimodel                       virtual OI-model
     */
    public VirtualProperty(VirtualOIModel oimodel) {
        super(oimodel);
    }
    /**
     * Initializes this instance
     *
     * @param property                      original property
     * @param isInOIModel                   determines whether this property is in OI-model
     */
    void initialize(Property property,boolean isInOIModel) throws KAONException {
        super.initialize(property,isInOIModel);
        Property inverseProperty=property.getInverseProperty();
        if (inverseProperty!=null) {
            m_inverseProperty=m_oimodel.getProperty(inverseProperty);
            m_inversePropertyOIModelOriginal=m_inverseProperty.getInversePropertyOIModel();
        }
        m_symmetric=property.isSymmetric();
        m_transitive=property.isTransitive();
        m_isAttribute=property.isAttribute();
        m_cardinalities=new HashMap();
    }
    /**
     * Returns the original property for the virtual property.
     *
     * @return                          original property
     */
    public Property getProperty() {
        return (Property)m_entity;
    }
    /**
     * Returns the inverse property of this property. If this property doesn't have an inverse property, then
     * <code>null</code> is returned.
     *
     * @return                          inverse property of this property (or <code>null</code> if there is no inverse property)
     */
    public Property getInverseProperty() {
        return m_inverseProperty;
    }
    /**
     * Returns the OI-model in which the inverse property of this property has been declared.
     *
     * @return                          the OI-model in which the inverse property of this property has been declared
     */
    public OIModel getInversePropertyOIModel() {
        return getInverseProperty()==null ? null : m_oimodel;
    }
    /**
     * Returns the original OI-model in which the inverse property of this property has been declared.
     *
     * @return                          the original OI-model in which the inverse property of this property has been declared
     */
    public OIModel getInversePropertyOIModelOriginal() {
        return m_inversePropertyOIModelOriginal;
    }
    /**
     * Returns <code>true</code> if the inverse property of this property has been specified within the OI-model as returned
     * by <code>getOIModel()</code>.
     *
     * @return                          <code>true</code> if the inverse property of this property has been specified whithin the OI-model as returned by <code>getOIModel()</code>
     */
    public boolean isInversePropertyDeclaredLocally() {
        return getInverseProperty()!=null;
    }
    /**
     * Returns <code>true</code> if this property is symmetric.
     *
     * @return                          <code>true</code> if this property is symmetric
     */
    public boolean isSymmetric() {
        return m_symmetric;
    }
    /**
     * Returns <code>true</code> if this property is transitive.
     *
     * @return                          <code>true</code> if this property is transitive
     */
    public boolean isTransitive() {
        return m_transitive;
    }
    /**
     * Returns the map tracker of all virtual concepts that consitute the domain of this virtual property.
     *
     * @return                          map tracker for domain concepts
     */
    public MapTracker getDomainConceptsTracker() {
        if (m_domainConcepts==null)
            m_domainConcepts=new MapTracker() {
                protected Object getMissingValue(Object key) throws KAONException {
                    return getProperty().getDomainConceptOIModel(((VirtualConcept)key).getConcept());
                }
                protected Map loadMap() throws KAONException {
                    return getVirtualConceptsMap(getProperty().getDomainConcepts(),true);
                }
            };
        return m_domainConcepts;
    }
    /**
     * Returns the set of all virtual concepts that consitute the domain of this virtual property.
     *
     * @return                          array of concepts that consitute the domain of this property (may be empty)
     */
    public Set getDomainConcepts() throws KAONException {
        return getDomainConceptsTracker().keySet();
    }
    /**
     * Return all concepts that consitute the domain of this property.
     *
     * @return                          set of concepts that consitute the domain of this property (may be empty)
     */
    public Set getAllDomainConcepts() throws KAONException {
        // if there are hierarcy cycles, this method will loop forever
        Set result=new HashSet(getDomainConcepts());
        Iterator superProperties=getSuperProperties().iterator();
        while (superProperties.hasNext()) {
            Property property=(Property)superProperties.next();
            result.addAll(property.getAllDomainConcepts());
        }
        return result;
    }
    /**
     * Checks whether supplied concept is in the domain of this virtual relation.
     *
     * @param concept                   concept that is checked
     * @return                          <code>true</code> if supplied concept is in the domain
     */
    public boolean isDomainConcept(Concept concept) throws KAONException {
        return getDomainConcepts().contains(concept);
    }
    /**
     * Returns the OI-model in which the supplied concept was added as the domain to this property.
     *
     * @param concept                   concept that is checked
     * @return                          the OI-model in which supplied concept was added as the domain to this property
     */
    public OIModel getDomainConceptOIModel(Concept concept) throws KAONException {
        return isDomainConcept(concept) ? m_oimodel : null;
    }
    /**
     * Returns the original OI-model in which the supplied concept was added as the domain to this property.
     *
     * @param concept                   concept that is checked
     * @return                          the original OI-model in which supplied concept was added as the domain to this property
     */
    public OIModel getDomainConceptOIModelOrigial(VirtualConcept concept) throws KAONException {
        return (OIModel)getDomainConceptsTracker().getValue(concept);
    }
    /**
     * Returns <code>true</code> if supplied concept was added as the domain to this property in
     * the model which returned by <code>getOIModel()</code>.
     *
     * @param concept                   the concept that is cheched
     * @return                          <code>true</code> if supplied concept was added to the domain concepts wihtin the <code>getOIModel()</code> model
     */
    public boolean isDomainConceptDeclaredLocally(Concept concept) throws KAONException {
        return isDomainConcept(concept);
    }
    /**
     * Returns the minimum cardinality for given domain concept.
     *
     * @param concept                   concept for which cardinality is requested
     * @return                          minimum cardinality of this property for given concept
     */
    public int getMinimumCardinality(Concept concept) throws KAONException {
        Cardinality cardinality=(Cardinality)m_cardinalities.get(concept);
        if (cardinality==null) {
            cardinality=new Cardinality(-1,-1);
            m_cardinalities.put(concept,cardinality);
        }
        if (cardinality.m_minimum==-1)
            cardinality.m_minimum=getProperty().getMinimumCardinality(concept);
        return cardinality.m_minimum;
    }
    /**
     * Returns the maximum cardinality for given domain concept. In case of unlimited cardinality, Integer.MAX_VALUE is returned.
     *
     * @param concept                   concept for which cardinality is requested
     * @return                          maximum cardinality of this property for given concept
     */
    public int getMaximumCardinality(Concept concept) throws KAONException {
        Cardinality cardinality=(Cardinality)m_cardinalities.get(concept);
        if (cardinality==null) {
            cardinality=new Cardinality(-1,-1);
            m_cardinalities.put(concept,cardinality);
        }
        if (cardinality.m_maximum==-1)
            cardinality.m_maximum=getProperty().getMaximumCardinality(concept);
        return cardinality.m_maximum;
    }
    /**
     * Returns the set tracker of range concepts of this property. If a property is an attribute, empty set is returned.
     *
     * @return                          set of range concepts of this property, or an empty set if property is an attribute
     */
    public MapTracker getRangeConceptsTracker() {
        if (m_rangeConcepts==null)
            m_rangeConcepts=new MapTracker() {
                protected Object getMissingValue(Object key) throws KAONException {
                    return getProperty().getRangeConceptOIModel(((VirtualConcept)key).getConcept());
                }
                protected Map loadMap() throws KAONException {
                    return getVirtualConceptsMap(getProperty().getRangeConcepts(),true);
                }
            };
        return m_rangeConcepts;
    }
    /**
     * Returns the set of range concepts of this property. If a property is an attribute, empty set is returned.
     *
     * @return                          set of range concepts of this property, or an empty set if property is an attribute
     */
    public Set getRangeConcepts() throws KAONException {
        return getRangeConceptsTracker().keySet();
    }
    /**
     * Returns the set of all range concepts of this property. If a property is an attribute, empty set is returned.
     *
     * @return                          set of range concepts of this property, or an empty set if property is an attribute
     */
    public Set getAllRangeConcepts() throws KAONException {
        // if there are hierarcy cycles, this method will loop forever
        Set result=new HashSet(getRangeConcepts());
        Iterator superProperties=getSuperProperties().iterator();
        while (superProperties.hasNext()) {
            Property property=(Property)superProperties.next();
            result.addAll(property.getAllRangeConcepts());
        }
        return result;
    }
    /**
     * Checks whether supplied concept is in the range of this property.
     *
     * @param concept                   concept that is checked
     * @return                          <code>true</code> if supplied concept is in the range
     */
    public boolean isRangeConcept(Concept concept) throws KAONException {
        return getRangeConcepts().contains(concept);
    }
    /**
     * Returns the OI-model in which the supplied concept was added as the range to this property.
     *
     * @param concept                   concept that is checked
     * @return                          the OI-model in which supplied concept was added as the range to this property
     */
    public OIModel getRangeConceptOIModel(Concept concept) throws KAONException {
        return isRangeConcept(concept) ? m_oimodel : null;
    }
    /**
     * Returns the original OI-model in which the supplied concept was added as the range to this property.
     *
     * @param concept                   concept that is checked
     * @return                          the original OI-model in which supplied concept was added as the range to this property
     */
    public OIModel getRangeConceptOIModel(VirtualConcept concept) throws KAONException {
        return (OIModel)getRangeConceptsTracker().getValue(concept);
    }
    /**
     * Returns <code>true</code> if supplied concept was added as the range to this property in
     * the model which returned by <code>getOIModel()</code>.
     *
     * @param concept                   the concept that is cheched
     * @return                          <code>true</code> if supplied concept was added to the range concepts wihtin the <code>getOIModel()</code> model
     */
    public boolean isRangeConceptDeclaredLocally(Concept concept) throws KAONException {
        return isRangeConcept(concept);
    }
    /**
     * Returns whether this relation is an attribute.
     *
     * @return                          <code>true</code> if relation is a literal
     */
    public boolean isAttribute() {
        return m_isAttribute;
    }
    /**
     * Returns the set tracker of direct subproperties of this property.
     *
     * @return                          set of subproperties (may be empty)
     */
    public SetTracker getSubPropertiesTracker() {
        if (m_subProperties==null)
            m_subProperties=new SetTracker() {
                protected Set loadSet() throws KAONException {
                    return getVirtualProperties(getProperty().getSubProperties(),true);
                }
            };
        return m_subProperties;
    }
    /**
     * Returns the set of direct subproperties of this property.
     *
     * @return                          set of subproperties (may be empty)
     */
    public Set getSubProperties() throws KAONException {
        return getSubPropertiesTracker().getSet();
    }
    /**
     * Returns the set of all subproperties of this property.
     *
     * @return                          set of all subproperties (may be empty)
     */
    public Set getAllSubProperties() throws KAONException {
        // if there are hierarcy cycles, this method will loop forever
        Set subProperties=getSubProperties();
        Set result=new HashSet(subProperties);
        Iterator iterator=subProperties.iterator();
        while (iterator.hasNext()) {
            Property property=(Property)iterator.next();
            result.addAll(property.getAllSubProperties());
        }
        return result;
    }
    /**
     * Tests if this property is a direct subproperty of given property. If properties are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperProperty    property being tested
     * @return                          <code>true</code> if this property is a direct subproperty
     */
    public boolean isDirectSubPropertyOf(Property potentialSuperProperty) throws KAONException {
        return getSuperProperties().contains(m_oimodel.getProperty(potentialSuperProperty));
    }
    /**
     * Returns the OI-model containing the this superproperty - subproperty association.
     *
     * @param superProperty             the superproperty of this property
     * @return                          the OI-model in which the super-subproperty association is declared
     */
    public OIModel getSuperSubPropertyOIModel(Property superProperty) throws KAONException {
        return isDirectSubPropertyOf(superProperty) ? m_oimodel : null;
    }
    /**
     * Returns the original OI-model containing the this superproperty - subproperty association.
     *
     * @param superProperty             the superproperty of this property
     * @return                          the original OI-model in which the super-subproperty association is declared
     */
    public OIModel getSuperSubPropertyOIModelOriginal(VirtualProperty superProperty) throws KAONException {
        return (OIModel)getSuperPropertiesTracker().getValue(superProperty);
    }
    /**
     * Returns <code>true</code> if this property has been declared a subproperty of given property in this OI-model.
     *
     * @param superProperty             the superproperty of this property
     * @return                          <code>true</code> if this property has been declared a superproperty of this property in this OI-model
     */
    public boolean isSuperSubPropertyDeclaredLocally(Property superProperty) throws KAONException {
        return isDirectSubPropertyOf(superProperty);
    }
    /**
     * Tests if this property is a superproperty of given property. If properties are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperProperty    property being tested
     * @return                          <code>true</code> if given relation is a subproperty
     */
    public boolean isSubPropertyOf(Property potentialSuperProperty) throws KAONException {
        return getAllSuperProperties().contains(m_oimodel.getProperty(potentialSuperProperty));
    }
    /**
     * Returns the set tracker of direct superproperties of this property.
     *
     * @return                          set of superproperties (may be empty)
     */
    public MapTracker getSuperPropertiesTracker() {
        if (m_superProperties==null)
            m_superProperties=new MapTracker() {
                protected Object getMissingValue(Object key) throws KAONException {
                    return getProperty().getSuperSubPropertyOIModel(((VirtualProperty)key).getProperty());
                }
                protected Map loadMap() throws KAONException {
                    return getVirtualPropertiesMap(getProperty().getSuperProperties(),true);
                }
            };
        return m_superProperties;
    }
    /**
     * Returns the set of direct superproperties of this property.
     *
     * @return                          set of superproperties (may be empty)
     */
    public Set getSuperProperties() throws KAONException {
        return getSuperPropertiesTracker().keySet();
    }
    /**
     * Returns the set of all superproperties of this property.
     *
     * @return                          set of all superproperties (may be empty)
     */
    public Set getAllSuperProperties() throws KAONException {
        // if there are hierarcy cycles, this method will loop forever
        Set superProperties=getSuperProperties();
        Set result=new HashSet(superProperties);
        Iterator iterator=superProperties.iterator();
        while (iterator.hasNext()) {
            Property property=(Property)iterator.next();
            result.addAll(property.getAllSuperProperties());
        }
        return result;
    }
    /**
     * Returns the set tracker of instances of this property.
     *
     * @return                          set of <code>PropertyInstance</code> objects of this property
     */
    public SetTracker getPropertyInstancesTracker() {
        if (m_propertyInstances==null)
            m_propertyInstances=new SetTracker() {
                protected Set loadSet() throws KAONException {
                    return getVirtualPropertyInstances(getProperty().getPropertyInstances(),true);
                }
            };
        return m_propertyInstances;
    }
    /**
     * Returns the set of instances of this property.
     *
     * @return                          set of <code>PropertyInstance</code> objects of this property
     */
    public Set getPropertyInstances() throws KAONException {
        return getPropertyInstancesTracker().getSet();
    }
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    public void accept(EntityVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
    /**
     * Adds a domain concept to the property.
     *
     * @param virtualConcept            virtual concept to add to the domain
     * @param sourceOIModel             the source OI-model where this relationship was created
     */
    void addDomainConcept(VirtualConcept virtualConcept,OIModel sourceOIModel) {
        getDomainConceptsTracker().add(virtualConcept,sourceOIModel);
        m_cardinalities.put(virtualConcept,new Cardinality(0,Integer.MAX_VALUE));
    }
    /**
     * Removes a domain concept from the property.
     *
     * @param virtualConcept            virtual concept to remove from the domain
     */
    void removeDomainConcept(VirtualConcept virtualConcept) {
        getDomainConceptsTracker().remove(virtualConcept);
    }
    /**
     * Sets the minimum cardinality for the property.
     *
     * @param virtualConcept            virtual concept to add to the domain
     * @param value                     cardinality
     */
    void setMinimumCardinality(VirtualConcept virtualConcept,int value) {
        Cardinality cardinality=(Cardinality)m_cardinalities.get(virtualConcept);
        if (cardinality==null) {
            cardinality=new Cardinality(-1,-1);
            m_cardinalities.put(virtualConcept,cardinality);
        }
        cardinality.m_minimum=value;
    }
    /**
     * Sets the maximum cardinality for the property.
     *
     * @param virtualConcept            virtual concept to add to the domain
     * @param value                     cardinality
     */
    void setMaximumCardinality(VirtualConcept virtualConcept,int value) {
        Cardinality cardinality=(Cardinality)m_cardinalities.get(virtualConcept);
        if (cardinality==null) {
            cardinality=new Cardinality(-1,-1);
            m_cardinalities.put(virtualConcept,cardinality);
        }
        cardinality.m_maximum=value;
    }
    /**
     * Sets the inverse property of this property.
     *
     * @param inverseProperty           inverse property of this property
     * @param inversePropertyOIModelOriginal    the original OI-model where properties are made inverse
     */
    void setInverseProperty(VirtualProperty inverseProperty,OIModel inversePropertyOIModelOriginal) {
        m_inverseProperty=inverseProperty;
        m_inversePropertyOIModelOriginal=inversePropertyOIModelOriginal;
    }
    /**
     * Sets the symmetric flag of this property.
     *
     * @param symmetric                 symmetric flag of this property
     */
    void setSymmetric(boolean symmetric) {
        m_symmetric=symmetric;
    }
    /**
     * Sets the transitive flag of this property.
     *
     * @param transitive                transitive flag of this property
     */
    void setTransitive(boolean transitive) {
        m_transitive=transitive;
    }
    /**
     * Sets the attribute flag of this property.
     *
     * @param isAttribute               attribute flag of this property
     */
    void setIsAttribute(boolean isAttribute) {
        m_isAttribute=isAttribute;
    }
    /**
     * Sets the flag that determines whether this entity is in the model.
     *
     * @param isInOIModel               flag determining whether this entity is in the model
     * @param sourceOIModel             the source OI-model
     */
    void setIsInOIModel(boolean isInOIModel,OIModel sourceOIModel) {
        super.setIsInOIModel(isInOIModel,sourceOIModel);
        if (!isInOIModel) {
            m_inverseProperty=null;
            m_symmetric=false;
            m_transitive=false;
            m_isAttribute=false;
            getDomainConceptsTracker().clear();
            getRangeConceptsTracker().clear();
            m_cardinalities=new HashMap();
            getSubPropertiesTracker().clear();
            getSuperPropertiesTracker().clear();
            getPropertyInstancesTracker().clear();
        }
    }

    /**
     * Structure tracking cardinalities.
     */
    protected static class Cardinality {
        public int m_minimum;
        public int m_maximum;

        public Cardinality(int minimum,int maximum) {
            m_minimum=minimum;
            m_maximum=maximum;
        }
    }
}
