package edu.unika.aifb.kaon.virtualoimodel;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Collection;
import java.util.Collections;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

/**
 * Virtual ontology implementation
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class VirtualOIModel implements OIModel {
    /** Original OI-model. */
    protected OIModel m_oimodel;
    /** Map of concept objects. */
    protected Map m_virtualConcepts;
    /** Map of property objects. */
    protected Map m_virtualProperties;
    /** Map of lexical entry objects. */
    protected Map m_virtualLexicalEntries;
    /** Map of property instance objects. */
    protected Map m_virtualPropertyInstances;
    /** Visitor for applying changes to this OI-model. */
    protected ChangeVisitor m_applyChangeVisitor;
    /** The attributes of this OI-model. */
    protected Map m_attributes;

    /**
     * Creates an instance of this class.
     *
     * @param oimodel                      OI-model
     */
    public VirtualOIModel(OIModel oimodel) throws KAONException {
        m_oimodel=oimodel;
        m_applyChangeVisitor=createApplyChangeVisitor();
        m_virtualConcepts=new HashMap();
        m_virtualProperties=new HashMap();
        m_virtualLexicalEntries=new HashMap();
        m_virtualPropertyInstances=new HashMap();
        // load the root concept
        getRootConcept();
    }
    /**
     * Creates a visitor for change events.
     *
     * @return                              visitor for change events
     */
    protected ChangeVisitor createApplyChangeVisitor() {
        return new VirtualApplyChangeVisitor(this);
    }
    /**
     * Returns the capabilities of the model.
     *
     * @return                          the bit-mask defining the model's capaibilities
     */
    public int getCapabilities() {
        return 0;
    }
    /**
     * Returns the KAON connection of this OI-model.
     *
     * @return                          KAON connection of this OI-model
     */
    public KAONConnection getKAONConnection() throws KAONException {
        return m_oimodel.getKAONConnection();
    }
    /**
     * Returns original OI-model.
     *
     * @return                              original OI-model
     */
    public OIModel getOIModel() {
        return m_oimodel;
    }
    /**
     * Returns the logical URI of this OI-model.
     *
     * @return                          logical URI of this OI-model
     */
    public String getLogicalURI() throws KAONException {
        return m_oimodel.getLogicalURI();
    }
    /**
     * Returns the physical URI of this OI-model.
     *
     * @return                          physical URI of this OI-model
     */
    public String getPhysicalURI() throws KAONException {
        return m_oimodel.getPhysicalURI();
    }
    /**
     * Returns the set of included OI-models.
     *
     * @return                          set of included OI-models
     */
    public Set getIncludedOIModels() {
        return Collections.EMPTY_SET;
    }
    /**
     * Returns the set of models that include this model.
     *
     * @return                          set of OI-models that include this model
     */
    public Set getAllIncludedByOIModels() {
        return Collections.EMPTY_SET;
    }
    /**
     * Returns an included OI-model with given URI.
     *
     * @param uri                       URI of the requested OI-model
     * @return                          OI-model with given URI or <code>null</code> if OI-model with this URI wasn't included
     */
    public OIModel getIncludedOIModel(String uri) {
        return null;
    }
    /**
     * Adds a listener to this OI-model.
     *
     * @param listener                  listener to be added
     */
    public void addOIModelListener(OIModelListener listener) {
        throw new UnsupportedOperationException();
    }
    /**
     * Removes a listener from this OI-model.
     *
     * @param listener                  listener to be removed
     */
    public void removeOIModelListener(OIModelListener listener) {
        throw new UnsupportedOperationException();
    }
    /**
     * Suspends entity pool events until {@link #resumeEvents} is called.
     */
    public void suspendEvents() {
    }
    /**
     * Resumes entity pool events.
     */
    public void resumeEvents() {
    }
    /**
     * Processes changes in the change list.
     *
     * @param changeList                list of changes to the model
     */
    public void applyChanges(List changeList) throws KAONException {
        Iterator iterator=changeList.iterator();
        while (iterator.hasNext()) {
            ChangeEvent changeEvent=(ChangeEvent)iterator.next();
            changeEvent.accept(m_applyChangeVisitor);
        }
    }
    /**
     * Loads objects in the collection.
     *
     * @param objects                   objects to be loaded
     * @param loadFlag                  the flag of what to load
     */
    public void loadObjects(Collection objects,int loadFlag) throws KAONException {
        List list=new LinkedList();
        Iterator iterator=objects.iterator();
        while (iterator.hasNext()) {
            VirtualEntity entity=(VirtualEntity)iterator.next();
            list.add(entity.getEntity());
        }
        m_oimodel.loadObjects(list,loadFlag);
    }
    /**
     * Executes a query and returns the set of instances or pairs matching given query.
     *
     * @param queryString               the query to be executed
     * @return                          the collection of result objects
     */
    public Collection executeQuery(String queryString) throws KAONException {
        throw new KAONException("Virtual OI-model cannot execute queries.");
    }
    /**
     * Notifies this entity pool about a change in the system. Entity pool should update its internal state and
     * forward the notification to its listeners. Users should typically not call this method, but rely on
     * entity pool to generate notifications itself.
     *
     * @param changeEvent               change event in the pool
     */
    public void notifyChangeEvent(ChangeEvent changeEvent) {
    }
    /**
     * Returns a virtual concept with given URI.
     *
     * @param uri                       URI of the requested concept
     * @return                          virtual concept with given URI
     */
    public Concept getConcept(String uri) throws KAONException {
        return getConcept(m_oimodel.getConcept(uri));
    }
    /**
     * Returns a virtual concept with given original concept.
     *
     * @param concept                   original concept
     * @return                          virtual concept for a given concept
     */
    public VirtualConcept getConcept(Concept concept) throws KAONException {
        String uri=concept.getURI();
        VirtualConcept virtualConcept=(VirtualConcept)m_virtualConcepts.get(uri);
        if (virtualConcept==null) {
            virtualConcept=new VirtualConcept(this);
            m_virtualConcepts.put(uri,virtualConcept);
            Concept conceptInThisModel=m_oimodel.getConcept(uri);
            virtualConcept.initialize(conceptInThisModel,conceptInThisModel.isInOIModel());
        }
        return virtualConcept;
    }
    /**
     * Returns a virtual concept with given original concept. This method is used internally when one knows whether a concept
     * is in OI-model.
     *
     * @param concept                   original concept
     * @param isInOIModel               determines whether the concept is in OI-model
     * @return                          virtual concept for a given concept
     */
    public VirtualConcept getConcept(Concept concept,boolean isInOIModel) throws KAONException {
        String uri=concept.getURI();
        VirtualConcept virtualConcept=(VirtualConcept)m_virtualConcepts.get(uri);
        if (virtualConcept==null) {
            virtualConcept=new VirtualConcept(this);
            m_virtualConcepts.put(uri,virtualConcept);
            Concept conceptInThisModel=m_oimodel.getConcept(uri);
            virtualConcept.initialize(conceptInThisModel,isInOIModel);
        }
        return virtualConcept;
    }
    /**
     * Returns the root concepts of the virtual ontology. Root concepts are concepts that are not subclasses of any other concept.
     *
     * @return                          root concept
     */
    public Concept getRootConcept() throws KAONException {
        return getConcept(m_oimodel.getRootConcept());
    }
    /**
     * Returns all virtual concepts of this ontology. This is a very expensive operation.
     *
     * @return                          array of all virtual concepts in this ontology
     */
    public Set getConcepts() throws KAONException {
        Set result=new HashSet(m_virtualConcepts.values());
        Iterator iterator=result.iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            if (!concept.isInOIModel())
                iterator.remove();
        }
        Iterator original=m_oimodel.getConcepts().iterator();
        while (original.hasNext()) {
            Concept concept=(Concept)original.next();
            if (!m_virtualConcepts.containsKey(concept.getURI()) && concept.isInOIModel())
                result.add(getConcept(concept));
        }
        return result;
    }
    /**
     * Returns a virtual property with given URI.
     *
     * @param uri                       URI of the property
     * @return                          virtual property with given URI
     */
    public Property getProperty(String uri) throws KAONException {
        return getProperty(m_oimodel.getProperty(uri));
    }
    /**
     * Returns a virtual property with given original property.
     *
     * @param property                  original property
     * @return                          virtual property for a given property
     */
    public VirtualProperty getProperty(Property property) throws KAONException {
        String uri=property.getURI();
        VirtualProperty virtualProperty=(VirtualProperty)m_virtualProperties.get(uri);
        if (virtualProperty==null) {
            virtualProperty=new VirtualProperty(this);
            m_virtualProperties.put(uri,virtualProperty);
            Property propertyInThisModel=m_oimodel.getProperty(uri);
            virtualProperty.initialize(propertyInThisModel,propertyInThisModel.isInOIModel());
        }
        return virtualProperty;
    }
    /**
     * Returns a virtual property with given original property.
     *
     * @param property                  original property
     * @param isInOIModel               determines whether the property is in OI-model
     * @return                          virtual property for a given property
     */
    public VirtualProperty getProperty(Property property,boolean isInOIModel) throws KAONException {
        String uri=property.getURI();
        VirtualProperty virtualProperty=(VirtualProperty)m_virtualProperties.get(uri);
        if (virtualProperty==null) {
            virtualProperty=new VirtualProperty(this);
            m_virtualProperties.put(uri,virtualProperty);
            Property propertyInThisModel=m_oimodel.getProperty(uri);
            virtualProperty.initialize(propertyInThisModel,isInOIModel);
        }
        return virtualProperty;
    }
    /**
     * Returns all virtual properties of this ontology. This is a very expensive operation.
     *
     * @return                          set of all virtual properties of this ontology (may be empty)
     */
    public Set getProperties() throws KAONException {
        Set result=new HashSet(m_virtualProperties.values());
        Iterator iterator=result.iterator();
        while (iterator.hasNext()) {
            Property property=(Property)iterator.next();
            if (!property.isInOIModel())
                iterator.remove();
        }
        Iterator original=m_oimodel.getProperties().iterator();
        while (original.hasNext()) {
            Property property=(Property)original.next();
            if (!m_virtualProperties.containsKey(property.getURI()) && property.isInOIModel())
                result.add(getProperty(property));
        }
        return result;
    }
    /**
     * Returns a virtual instance with given URI.
     *
     * @param uri                       URI of the requested instnace
     * @return                          instance with given URI
     */
    public Instance getInstance(String uri) throws KAONException {
        return getInstance(m_oimodel.getInstance(uri));
    }
    /**
     * Returns a virtual instance for given original instance.
     *
     * @param instance                  original instance
     * @return                          virtual instance for given original instance
     */
    public VirtualInstance getInstance(Instance instance) throws KAONException {
        String uri=instance.getURI();
        VirtualLexicalEntry virtualLexicalEntry=(VirtualLexicalEntry)m_virtualLexicalEntries.get(uri);
        if (virtualLexicalEntry==null) {
            virtualLexicalEntry=new VirtualLexicalEntry(this);
            m_virtualLexicalEntries.put(uri,virtualLexicalEntry);
            Instance instanceInThisModel=m_oimodel.getInstance(uri);
            virtualLexicalEntry.initialize(instanceInThisModel,instanceInThisModel.isInOIModel());
        }
        return virtualLexicalEntry;
    }
    /**
     * Returns a virtual instance for given original instance.
     *
     * @param instance                  original instance
     * @param isInOIModel               determines whether the property is in OI-model
     * @return                          virtual instance for given original instance
     */
    public VirtualInstance getInstance(Instance instance,boolean isInOIModel) throws KAONException {
        String uri=instance.getURI();
        VirtualLexicalEntry virtualLexicalEntry=(VirtualLexicalEntry)m_virtualLexicalEntries.get(uri);
        if (virtualLexicalEntry==null) {
            virtualLexicalEntry=new VirtualLexicalEntry(this);
            m_virtualLexicalEntries.put(uri,virtualLexicalEntry);
            Instance instanceInThisModel=m_oimodel.getInstance(uri);
            virtualLexicalEntry.initialize(instanceInThisModel,isInOIModel);
        }
        return virtualLexicalEntry;
    }
    /**
     * Returns all virtual instances of this ontology. This is a very expensive operation.
     *
     * @return                          array of all virtual instances in this ontology
     */
    public Set getInstances() throws KAONException {
        Set result=new HashSet(m_virtualLexicalEntries.values());
        Iterator iterator=result.iterator();
        while (iterator.hasNext()) {
            Instance instance=(Instance)iterator.next();
            if (!instance.isInOIModel())
                iterator.remove();
        }
        Iterator original=m_oimodel.getInstances().iterator();
        while (original.hasNext()) {
            Instance instance=(Instance)original.next();
            if (!m_virtualLexicalEntries.containsKey(instance.getURI()) && instance.isInOIModel())
                result.add(getInstance(instance));
        }
        return result;
    }
    /**
     * Returns a virtual lexical entry with given URI.
     *
     * @param uri                       URI of the requested vritual lexical entry
     * @return                          virtual lexical entry with given URI
     */
    public LexicalEntry getLexicalEntry(String uri) throws KAONException {
        return (LexicalEntry)getInstance(m_oimodel.getLexicalEntry(uri));
    }
    /**
     * Returns a virtual lexical entry for given original entry.
     *
     * @param lexicalEntry              original lexical entry
     * @return                          virtual lexical entry for given original lexical entry
     */
    public VirtualLexicalEntry getLexicalEntry(LexicalEntry lexicalEntry) throws KAONException {
        return (VirtualLexicalEntry)getInstance(lexicalEntry);
    }
    /**
     * Returns a virtual lexical entry for given original entry.
     *
     * @param lexicalEntry              original lexical entry
     * @param isInOIModel               determines whether the property is in OI-model
     * @return                          virtual lexical entry for given original lexical entry
     */
    public VirtualLexicalEntry getLexicalEntry(LexicalEntry lexicalEntry,boolean isInOIModel) throws KAONException {
        return (VirtualLexicalEntry)getInstance(lexicalEntry,isInOIModel);
    }
    /**
     * Factory method for property instance objects.
     *
     * @param property                  property
     * @param sourceInstance            source instance
     * @param targetValue               target value (<code>Instance</code> or a <code>String</code>)
     * @return                          virtual property instance
     */
    public PropertyInstance getPropertyInstance(Property property,Instance sourceInstance,Object targetValue) throws KAONException {
        String key;
        if (targetValue instanceof Instance)
            key=property.getURI()+"$"+sourceInstance.getURI()+"$"+((Instance)targetValue).getURI();
        else
            key=property.getURI()+"$"+sourceInstance.getURI()+"$"+targetValue.toString();
        VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)m_virtualPropertyInstances.get(key);
        if (virtualPropertyInstance==null) {
            Object targetValueInThisModel;
            if (targetValue instanceof Instance)
                targetValueInThisModel=m_oimodel.getInstance(((Instance)targetValue).getURI());
            else
                targetValueInThisModel=targetValue;
            PropertyInstance propertyInstanceInThisModel=m_oimodel.getPropertyInstance(m_oimodel.getProperty(property.getURI()),m_oimodel.getInstance(sourceInstance.getURI()),targetValueInThisModel);
            virtualPropertyInstance=new VirtualPropertyInstance();
            virtualPropertyInstance.initialize(propertyInstanceInThisModel,this,propertyInstanceInThisModel.isInOIModel());
            m_virtualPropertyInstances.put(key,virtualPropertyInstance);
        }
        return virtualPropertyInstance;
    }
    /**
     * Factory method for property instance objects.
     *
     * @param propertyInstance          property instance
     * @return                          virtual property instance
     */
    public VirtualPropertyInstance getPropertyInstance(PropertyInstance propertyInstance) throws KAONException {
        String key;
        Property property=propertyInstance.getProperty();
        Instance sourceInstance=propertyInstance.getSourceInstance();
        Object targetValue=propertyInstance.getTargetValue();
        if (targetValue instanceof Instance)
            key=property.getURI()+"$"+sourceInstance.getURI()+"$"+((Instance)targetValue).getURI();
        else
            key=property.getURI()+"$"+sourceInstance.getURI()+"$"+targetValue.toString();
        VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)m_virtualPropertyInstances.get(key);
        if (virtualPropertyInstance==null) {
            Object targetValueInThisModel;
            if (targetValue instanceof Instance)
                targetValueInThisModel=m_oimodel.getInstance(((Instance)targetValue).getURI());
            else
                targetValueInThisModel=targetValue;
            PropertyInstance propertyInstanceInThisModel=m_oimodel.getPropertyInstance(m_oimodel.getProperty(property.getURI()),m_oimodel.getInstance(sourceInstance.getURI()),targetValueInThisModel);
            virtualPropertyInstance=new VirtualPropertyInstance();
            virtualPropertyInstance.initialize(propertyInstanceInThisModel,this,propertyInstanceInThisModel.isInOIModel());
            m_virtualPropertyInstances.put(key,virtualPropertyInstance);
        }
        return virtualPropertyInstance;
    }
    /**
     * Factory method for property instance objects.
     *
     * @param propertyInstance          property instance
     * @param isInOIModel               determines whether the property instance is in OI-model
     * @return                          virtual property instance
     */
    public VirtualPropertyInstance getPropertyInstance(PropertyInstance propertyInstance,boolean isInOIModel) throws KAONException {
        String key;
        Property property=propertyInstance.getProperty();
        Instance sourceInstance=propertyInstance.getSourceInstance();
        Object targetValue=propertyInstance.getTargetValue();
        if (targetValue instanceof Instance)
            key=property.getURI()+"$"+sourceInstance.getURI()+"$"+((Instance)targetValue).getURI();
        else
            key=property.getURI()+"$"+sourceInstance.getURI()+"$"+targetValue.toString();
        VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)m_virtualPropertyInstances.get(key);
        if (virtualPropertyInstance==null) {
            Object targetValueInThisModel;
            if (targetValue instanceof Instance)
                targetValueInThisModel=m_oimodel.getInstance(((Instance)targetValue).getURI());
            else
                targetValueInThisModel=targetValue;
            PropertyInstance propertyInstanceInThisModel=m_oimodel.getPropertyInstance(m_oimodel.getProperty(property.getURI()),m_oimodel.getInstance(sourceInstance.getURI()),targetValueInThisModel);
            virtualPropertyInstance=new VirtualPropertyInstance();
            virtualPropertyInstance.initialize(propertyInstanceInThisModel,this,isInOIModel);
            m_virtualPropertyInstances.put(key,virtualPropertyInstance);
        }
        return virtualPropertyInstance;
    }
    /**
     * Creates a new URI unique wihtin this OI-model.
     *
     * @return                          an new URI unique within this OI-model
     */
    public String createNewURI() throws KAONException {
        return m_oimodel.createNewURI();
    }
    /**
     * Creates an URI by combining the logical URI with the given suffix.
     *
     * @param suffix                    the suffix
     * @return                          a new URI with given suffix (may not be unique)
     */
    public String getURI(String suffix) throws KAONException {
        return m_oimodel.getURI(suffix);
    }
    /**
     * Makes sure that all data of this ontology is saved.
     */
    public void save() throws KAONException {
        m_oimodel.save();
    }
    /**
     * Returns <code>true</code> if the OI-model has been changed since the last save.
     *
     * @return                          <code>true</code> if the OI-model has been changed since the last save
     */
    public boolean hasUnsavedChanges() throws KAONException {
        return m_oimodel.hasUnsavedChanges();
    }
    /**
     * Deletes this OI-model.
     */
    public void delete() throws KAONException {
        m_oimodel.delete();
    }
    /**
     * Refreshes the contents of this OI-model.
     */
    public void refresh() throws KAONException {
        m_oimodel.refresh();
    }
    /**
     * Returns the value of the OI-model attribute with given key.
     *
     * @param key                       the key
     * @return                          the value (or <code>null</code> if there is no attribute with given key)
     */
    public String getAttribute(String key) throws KAONException {
        return (String)getAttributes().get(key);
    }
    /**
     * Sets the value of the OI-model attribute with given key.
     *
     * @param key                       the key
     * @param value                     the value (or <code>null</code> if the attribute should be deleted)
     */
    public void setAttribute(String key,String value) throws KAONException {
        Map attributes=getAttributes();
        if (value==null)
            attributes.remove(key);
        else
            attributes.put(key,value);
    }
    /**
     * Returns the map of all key-value pairs of this OI-model.
     *
     * @return                          the map of key-value keys
     */
    public Map getAttributes() throws KAONException {
        if (m_attributes==null)
            m_attributes=new HashMap(m_oimodel.getAttributes());
        return m_attributes;
    }
}
